package com.xebialabs.xlrelease.repository.sql.security

import com.xebialabs.xlrelease.domain.Team
import com.xebialabs.xlrelease.limits.LimitEnforcer
import com.xebialabs.xlrelease.repository.{FolderRepository, TeamRepository}
import com.xebialabs.xlrelease.service.{CreateTeamOperation, DeleteTeamOperation, TeamLimitEnforcer, TeamUpdateOperation}
import com.xebialabs.xlrelease.utils.TenantContext
import grizzled.slf4j.Logging
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.context.annotation.Lazy
import org.springframework.stereotype.Service
import org.springframework.util.StringUtils.hasText

import scala.jdk.CollectionConverters._

/**
 * Enforcer responsible for folder team limits.
 * Handles counting existing (non-system) teams in folders and enforcing maximum teams per folder limits.
 * System teams (Team.isSystemTeam) are excluded from counting/enforcement.
 */
@Service
class FolderTeamLimitEnforcer(limitEnforcer: LimitEnforcer) extends TeamLimitEnforcer with Logging {

  private case class TeamNetChange(createsCount: Int, deletesCount: Int) {
    def increment: Int = createsCount - deletesCount
  }

  @Autowired
  @Lazy
  private val folderRepository: FolderRepository = null

  @Autowired
  @Lazy
  private val teamRepository: TeamRepository = null

  override def enforceLimitForTeamCreation(folderId: String, teamName: String): Unit = {
    if (Team.isSystemTeam(teamName)) {
      logger.debug(s"[FolderTeamLimitEnforcer] Skipping system team: $teamName")
    } else {
      enforceLimitForTeamChange(folderId, 1, 0) // 1 create, 0 deletes
    }
  }

  def enforceLimitForBulkTeamCreation(folderId: String, desiredTeams: Seq[Team]): Unit = {
    val currentTeams = teamRepository.getTeams(folderId).asScala.toSeq
    val teamNetChange = calculateTeamNetChange(currentTeams, desiredTeams)
    enforceLimitForTeamChange(folderId, teamNetChange.createsCount, teamNetChange.deletesCount)
  }

  override def enforceLimitForTeamOperations(folderId: String, operations: java.util.List[TeamUpdateOperation]): Unit = {
    if (operations.isEmpty) return

    @inline def isCustom(name: String): Boolean = !Team.isSystemTeam(name)

    var creates = 0
    var deletes = 0

    operations.asScala.foreach {
      case op: CreateTeamOperation if isCustom(op.team) => creates += 1
      case op: DeleteTeamOperation if isCustom(op.team) => deletes += 1
      case _ => // ignore system teams and other operation types
    }

    enforceLimitForTeamChange(folderId, creates, deletes)
  }

  private def enforceLimitForTeamChange(folderId: String, createCount: Int, deleteCount: Int): Unit = {
    val teamNetChange = TeamNetChange(createCount, deleteCount)

    logger.debug(s"[FolderTeamLimitEnforcer] Evaluating changes for folder $folderId: creates=$createCount, deletes=$deleteCount, netChange=${teamNetChange.increment}")

    // Only enforce if we have any create operations (even if there are also deletes)
    if (createCount > 0) {
      val currentCustomTeamCount = teamRepository.getTeams(folderId).asScala.count(!_.isSystemTeam)
      logger.debug(s"[FolderTeamLimitEnforcer] Enforcing limit: current=$currentCustomTeamCount, netChange=${teamNetChange.increment}")

      val tenantId = getTenantId(folderId)
      // For atomic operations, the net change is what matters for limit checking
      // We use the actual current count for error messages but check against net change
      limitEnforcer.enforceLimit(tenantId, LimitEnforcer.LimitType.FOLDER_TEAMS, teamNetChange.increment, () => currentCustomTeamCount)
    }
  }

  private def calculateTeamNetChange(currentTeams: Seq[Team], desiredTeams: Seq[Team]): TeamNetChange = {
    @inline def customNames(teams: Seq[Team]): Set[String] = teams.iterator.map(_.getTeamName).filterNot(Team.isSystemTeam).toSet

    val current = customNames(currentTeams)
    val desired = customNames(desiredTeams)

    val creates = desired.count(n => !current.contains(n))
    val deletes = current.count(n => !desired.contains(n))

    TeamNetChange(creates, deletes)
  }

  private def getTenantId(folderId: String): String = {
    val tenantIdFromContext = TenantContext.getTenant()
    if (hasText(tenantIdFromContext)) {
      tenantIdFromContext
    } else {
      folderRepository.getFolderTenant(folderId)
    }
  }
}