package com.xebialabs.xlrelease.repository.sql

import com.xebialabs.deployit.plugin.api.reflect.Type
import com.xebialabs.xlrelease.config.CacheManagementConstants.{ALL_TENANT_LIMITS_CACHE, TENANT_CACHE_MANAGER, TENANT_LIMITS_CACHE}
import com.xebialabs.xlrelease.configuration.TenantLimit
import com.xebialabs.xlrelease.db.sql.SqlBuilder.Dialect
import com.xebialabs.xlrelease.db.sql.transaction.{IsReadOnly, IsTransactional}
import com.xebialabs.xlrelease.repository.TenantLimitRepository
import com.xebialabs.xlrelease.repository.sql.persistence.PersistenceConstants.BLOB_TYPE
import com.xebialabs.xlrelease.repository.sql.persistence.PersistenceSupport
import com.xebialabs.xlrelease.repository.sql.persistence.Schema.TENANT_LIMITS
import com.xebialabs.xlrelease.serialization.json.utils.CiSerializerHelper
import grizzled.slf4j.Logging
import org.springframework.cache.annotation.{CacheConfig, CacheEvict, Cacheable}
import org.springframework.dao.EmptyResultDataAccessException
import org.springframework.jdbc.core.namedparam.MapSqlParameterSource
import org.springframework.jdbc.core.support.SqlBinaryValue
import org.springframework.jdbc.core.{JdbcTemplate, RowMapper}
import org.springframework.util.Assert

import java.nio.charset.StandardCharsets
import java.sql.ResultSet
import scala.jdk.CollectionConverters._

@IsTransactional
@CacheConfig(cacheManager = TENANT_CACHE_MANAGER)
class SqlTenantLimitRepository(implicit val jdbcTemplate: JdbcTemplate,
                               implicit val dialect: Dialect)
  extends TenantLimitRepository
    with PersistenceSupport
    with Logging {

  private val STMT_SELECT_LIMIT_BY_TENANT_ID =
    s"""SELECT
       | ${TENANT_LIMITS.TENANT_ID},
       | ${TENANT_LIMITS.LIMIT_DATA}
       |FROM ${TENANT_LIMITS.TABLE}
       |WHERE ${TENANT_LIMITS.TENANT_ID} = :tenantId""".stripMargin

  private val STMT_SELECT_ALL_LIMITS =
    s"""SELECT
       | ${TENANT_LIMITS.TENANT_ID},
       | ${TENANT_LIMITS.LIMIT_DATA}
       |FROM ${TENANT_LIMITS.TABLE}""".stripMargin

  private val STMT_INSERT_LIMIT =
    s"""INSERT INTO ${TENANT_LIMITS.TABLE}
       | (${TENANT_LIMITS.TENANT_ID}, ${TENANT_LIMITS.LIMIT_DATA})
       |VALUES (:tenantId, :limitData)""".stripMargin

  private val STMT_UPDATE_LIMIT =
    s"""UPDATE ${TENANT_LIMITS.TABLE}
       |SET ${TENANT_LIMITS.LIMIT_DATA} = :limitData
       |WHERE ${TENANT_LIMITS.TENANT_ID} = :tenantId""".stripMargin

  private val STMT_DELETE_LIMIT =
    s"""DELETE FROM ${TENANT_LIMITS.TABLE}
       |WHERE ${TENANT_LIMITS.TENANT_ID} = :tenantId""".stripMargin

  private val tenantLimitRowMapper: RowMapper[TenantLimit] = (rs: ResultSet, _: Int) => {
    val tenantId = rs.getString(TENANT_LIMITS.TENANT_ID)
    val limitDataBytes = rs.getBytes(TENANT_LIMITS.LIMIT_DATA)
    val limitDataJson = new String(limitDataBytes, StandardCharsets.UTF_8)

    val limit = CiSerializerHelper.deserialize(limitDataJson, null).asInstanceOf[TenantLimit]
    limit.setTenantId(tenantId)
    limit
  }

  @IsReadOnly
  @Cacheable(cacheNames = Array(TENANT_LIMITS_CACHE))
  override def findByTenantId(tenantId: String): Option[TenantLimit] = {
    try {
      val params = new MapSqlParameterSource()
      params.addValue("tenantId", tenantId)

      val limit = namedTemplate.queryForObject(STMT_SELECT_LIMIT_BY_TENANT_ID, params, tenantLimitRowMapper)
      Some(limit)
    } catch {
      case _: EmptyResultDataAccessException => None
    }
  }

  @IsReadOnly
  @Cacheable(cacheNames = Array(TENANT_LIMITS_CACHE), key = "'effective:' + #tenantId")
  override def getEffectiveLimit(tenantId: String): TenantLimit = {
    Assert.hasText(tenantId, "Tenant ID cannot be empty")
    findByTenantId(tenantId).getOrElse {
      logger.debug(s"No tenant limit found for tenant ID '$tenantId', falling back to runtime definition")
      Type.valueOf(classOf[TenantLimit]).getDescriptor.newInstance[TenantLimit]("")
    }
  }

  @IsReadOnly
  @Cacheable(cacheNames = Array(ALL_TENANT_LIMITS_CACHE))
  override def findAll(): Seq[TenantLimit] = {
    namedTemplate.query(STMT_SELECT_ALL_LIMITS, tenantLimitRowMapper).asScala.toSeq
  }

  @CacheEvict(cacheNames = Array(ALL_TENANT_LIMITS_CACHE, TENANT_LIMITS_CACHE), allEntries = true)
  override def create(limit: TenantLimit): TenantLimit = {
    val params = new MapSqlParameterSource()
    params.addValue("tenantId", limit.getTenantId)
    params.addValue("limitData", new SqlBinaryValue(serialize(limit).getBytes(StandardCharsets.UTF_8)), BLOB_TYPE)

    namedTemplate.update(STMT_INSERT_LIMIT, params)
    limit
  }

  @CacheEvict(cacheNames = Array(ALL_TENANT_LIMITS_CACHE, TENANT_LIMITS_CACHE), allEntries = true)
  override def update(limit: TenantLimit): Unit = {
    val params = new MapSqlParameterSource()
    params.addValue("tenantId", limit.getTenantId)
    params.addValue("limitData", new SqlBinaryValue(serialize(limit).getBytes(StandardCharsets.UTF_8)), BLOB_TYPE)

    val rowsAffected = namedTemplate.update(STMT_UPDATE_LIMIT, params)
    if (rowsAffected == 0) {
      throw new IllegalArgumentException(s"Tenant limit with tenant ID '${limit.getTenantId}' does not exist")
    }
  }

  @CacheEvict(cacheNames = Array(ALL_TENANT_LIMITS_CACHE, TENANT_LIMITS_CACHE), allEntries = true)
  override def delete(tenantId: String): Unit = {
    val params = new MapSqlParameterSource()
    params.addValue("tenantId", tenantId)

    namedTemplate.update(STMT_DELETE_LIMIT, params)
  }

  private def serialize(limit: TenantLimit): String = {
    limit.setId(limit.getTenantId)
    CiSerializerHelper.serialize(limit)
  }

}
