package com.xebialabs.xlrelease.repository.sql

import com.xebialabs.deployit.exception.NotFoundException
import com.xebialabs.xlrelease.db.sql.transaction.IsTransactional
import com.xebialabs.xlrelease.domain.variables.Variable
import com.xebialabs.xlrelease.repository.FolderVariableRepository
import com.xebialabs.xlrelease.repository.IdMatchers.FolderId
import com.xebialabs.xlrelease.repository.sql.persistence.data.FolderVariableRow
import com.xebialabs.xlrelease.repository.sql.persistence.{CiUid, FolderVariablePersistence}
import grizzled.slf4j.Logging
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Repository

import java.util.{List => JList}
import scala.collection.mutable
import scala.jdk.CollectionConverters._

@Repository
@IsTransactional
class SqlFolderVariableRepository @Autowired()(variablePersistence: FolderVariablePersistence, repositoryAdapter: SqlRepositoryAdapter)
  extends FolderVariableRepository
    with Logging {

  override def exists(variableId: String): Boolean = {
    logger.debug(s"checking folder variable exists [$variableId]")
    variablePersistence.exists(variableId)
  }

  override def findById(variableId: String): Variable = {
    logger.debug(s"finding folder variable [$variableId]")
    variablePersistence.findById(variableId)
      .map(toVariable)
      .getOrElse(throw new NotFoundException("Variable with id='%s' not found", variableId))
  }

  override def findByKey(variableKey: String, folderId: String): Variable = {
    logger.debug(s"finding folder variable [key=$variableKey], [folderId=$folderId]")
    variablePersistence.findVariableByKey(variableKey, folderId)
      .map(toVariable)
      .getOrElse(throw new NotFoundException("Variable with key='%s' not found", variableKey))
  }

  override def create(variable: Variable): Variable = {
    logger.debug(s"creating new folder variable [key=${variable.getKey}], [folderId=${variable.getFolderId}]")
    
    interceptCreate(variable)
    
    val ciUid = variablePersistence.create(variable)
    val created = variablePersistence.findByCiUid(ciUid).map(toVariable).get
    
    afterCreate(created)
    created
  }

  override def create(variable: Variable, folderCiUid: CiUid): Variable = {
    logger.debug(s"creating new folder variable [key=${variable.getKey}], [folderId=${variable.getFolderId}]")
    
    interceptCreate(variable)
    
    val ciUid = variablePersistence.create(variable, folderCiUid)
    val created = variablePersistence.findByCiUid(ciUid).map(toVariable).get
    
    afterCreate(created)
    created
  }

  override def update(variable: Variable): Variable = {
    logger.debug(s"updating folder variable [key=${variable.getKey}], [folderId=${variable.getFolderId}]")
    
    interceptUpdate(variable)
    
    variablePersistence.update(variable)
    val updated = findById(variable.getId)
    
    afterUpdate(updated)
    updated
  }

  override def update(variable: Variable, folderCiUid: CiUid): Variable = {
    logger.debug(s"updating folder variable [key=${variable.getKey}], [folderId=${variable.getFolderId}]")
    
    interceptUpdate(variable)
    
    variablePersistence.update(variable, folderCiUid)
    val updated = findById(variable.getId)
    
    afterUpdate(updated)
    updated
  }

  override def delete(variableId: String): Unit = {
    logger.debug(s"deleting folder variable [$variableId])")
    
    interceptDelete(variableId)
    
    variablePersistence.delete(variableId)
    
    afterDelete(variableId)
  }

  override def getAllFromParent(variableParentId: String): JList[Variable] = {
    logger.debug(s"getting all folder variables by parent [id=$variableParentId]")
    variableParentId match {
      case FolderId(folderId) =>
        variablePersistence.findByFolderId(folderId).map(toVariable).asJava
      case _ => mutable.Buffer.empty[Variable].asJava
    }
  }

  override def getAllFromAncestry(variableParentId: String): JList[Variable] = {
    logger.debug(s"getting all folder variables from ancestry [id=$variableParentId]")
    variableParentId match {
      case FolderId(folderId) =>
        variablePersistence.findByParentIdIncludingAncestors(folderId).map(toVariable).asJava
      case _ => mutable.Buffer.empty[Variable].asJava
    }
  }

  private def toVariable(variableRow: FolderVariableRow) = {
    repositoryAdapter.deserialize[Variable](variableRow.json).map { variable =>
      variable.setFolderId(variableRow.folderId.absolute)
      variable
    }.getOrElse(throw new NotFoundException(s"Error reading folder variable '${variableRow.ciUid}', see logs for more details."))
  }
}
