package com.xebialabs.xlrelease.scheduler.logs

import com.xebialabs.xlrelease.db.sql.SqlBuilder.Dialect
import com.xebialabs.xlrelease.db.sql.transaction.{IsReadOnly, IsTransactional}
import com.xebialabs.xlrelease.repository.sql.persistence.Schema.TASK_EXECUTIONS
import com.xebialabs.xlrelease.repository.sql.persistence.TaskPersistence.hash
import com.xebialabs.xlrelease.repository.sql.persistence.Utils._
import com.xebialabs.xlrelease.repository.sql.persistence.{PersistenceSupport, Utils}
import com.xebialabs.xlrelease.scheduler.logs.TaskExecutionRepository._
import grizzled.slf4j.Logging
import io.micrometer.core.annotation.Timed
import org.springframework.data.domain.{Page, Pageable}
import org.springframework.jdbc.core.JdbcTemplate

import java.time.Instant
import scala.util.{Failure, Success, Try}


@IsTransactional
class SqlTaskExecutionRepository(implicit val jdbcTemplate: JdbcTemplate, val dialect: Dialect)
  extends TaskExecutionRepository
    with PersistenceSupport
    with TaskExecutionRowMapper
    with Logging
    with Utils {

  @Timed
  @IsReadOnly
  override def read(taskId: String, executionId: String): Option[TaskExecutionEntry] = {
    findOptional(_.queryForObject(STMT_FIND_TASK_EXECUTION, taskExecutionRowMapper, hash(taskId), executionId))
  }

  private def readByIdHash(taskIdHash: String, executionId: String): Option[TaskExecutionEntry] = {
    findOptional(_.queryForObject(STMT_FIND_TASK_EXECUTION, taskExecutionRowMapper, taskIdHash, executionId))
  }

  @Timed
  override def finishExecution(taskId: String, executionId: String, endDate: Instant): Option[TaskExecutionEntry] = {
    val maybeRow = read(taskId, executionId)
    maybeRow match {
      case Some(row) => Some(updateRow(row.copy(endDate = Option(endDate))))
      case None => None // Must be for a non container based task so do nothing
    }
  }

  override def markAsTruncated(taskId: String, executionId: String): Unit = {
    updateRowToTruncated(hash(taskId), executionId)
  }

  override def update(update: TaskExecutionEntry): Option[TaskExecutionEntry] = {
    val maybeRow = readByIdHash(update.taskIdHash, update.executionId)
    val row = maybeRow match {
      case Some(row) =>
        updateRow(row.mergeMax(update))
      case None =>
        Try(create(update)) match {
          case Failure(exception) =>
            val maybeRow = readByIdHash(update.taskIdHash, update.executionId)
            maybeRow.map(row => updateRow(row.mergeMax(update))).getOrElse(throw exception)
          case Success(row) => row
        }
    }
    Some(row)
  }

  @Timed
  override def delete(operation: DeleteOperation): Unit = {
    operation match {
      case Delete(taskId, executionId) => deleteExecution(hash(taskId), executionId)
      case DeleteWithTaskExecutionEntry(entry) => deleteExecution(entry.taskIdHash, entry.executionId)
    }
  }

  @Timed
  @IsReadOnly
  override def find(findOperation: FindOperation, pageable: Pageable): Page[TaskExecutionEntry] = {
    SqlTaskExecutionQueryBuilder(dialect, namedTemplate)
      .from(findOperation)
      .withPageable(pageable)
      .build()
      .execute()
  }

  private def createSqlParamMap(row: TaskExecutionEntry): Map[String, Any] = {
    Map(
      TASK_EXECUTIONS.TASK_ID_HASH -> row.taskIdHash,
      TASK_EXECUTIONS.EXECUTION_ID -> row.executionId,
      TASK_EXECUTIONS.LAST_JOB -> row.lastJob,
      TASK_EXECUTIONS.LAST_CHUNK -> row.lastChunk,
      TASK_EXECUTIONS.LAST_MODIFIED_DATE -> row.lastModifiedDate.asTimestamp,
      TASK_EXECUTIONS.END_DATE -> row.endDate.map(_.asTimestamp).orNull,
      TASK_EXECUTIONS.LOG_SIZE -> row.logSize
    )
  }

  private def create(row: TaskExecutionEntry): TaskExecutionEntry = {
    namedTemplate.update(STMT_INSERT_ROW, createSqlParamMap(row))
    row
  }

  private def updateRow(row: TaskExecutionEntry): TaskExecutionEntry = {
    namedTemplate.update(STMT_UPDATE_ROW, createSqlParamMap(row))
    row
  }

  private def updateRowToTruncated(taskIdHash: String, executionId: String): Unit = {
    namedTemplate.update(STMT_UPDATE_TRUNCATED, Map(
      TASK_EXECUTIONS.TASK_ID_HASH -> taskIdHash,
      TASK_EXECUTIONS.EXECUTION_ID -> executionId,
      TASK_EXECUTIONS.TRUNCATED -> 1
    ))
  }

  private def deleteExecution(taskIdHash: String, executionId: String): Unit = {
    val params = Map(
      TASK_EXECUTIONS.TASK_ID_HASH -> taskIdHash,
      TASK_EXECUTIONS.EXECUTION_ID -> executionId
    )
    namedTemplate.update(STMT_DELETE_ROW, params)
  }

  private val STMT_INSERT_ROW =
    s"""INSERT INTO ${TASK_EXECUTIONS.TABLE} (
       |  ${TASK_EXECUTIONS.TASK_ID_HASH},
       |  ${TASK_EXECUTIONS.EXECUTION_ID},
       |  ${TASK_EXECUTIONS.LAST_JOB},
       |  ${TASK_EXECUTIONS.LAST_CHUNK},
       |  ${TASK_EXECUTIONS.LAST_MODIFIED_DATE},
       |  ${TASK_EXECUTIONS.END_DATE},
       |  ${TASK_EXECUTIONS.LOG_SIZE}
       | ) VALUES (
       |  :${TASK_EXECUTIONS.TASK_ID_HASH},
       |  :${TASK_EXECUTIONS.EXECUTION_ID},
       |  :${TASK_EXECUTIONS.LAST_JOB},
       |  :${TASK_EXECUTIONS.LAST_CHUNK},
       |  :${TASK_EXECUTIONS.LAST_MODIFIED_DATE},
       |  :${TASK_EXECUTIONS.END_DATE},
       |  :${TASK_EXECUTIONS.LOG_SIZE}
       | )
       |""".stripMargin

  private val STMT_UPDATE_ROW =
    s"""UPDATE ${TASK_EXECUTIONS.TABLE}
       |  SET
       |    ${TASK_EXECUTIONS.LAST_JOB} = :${TASK_EXECUTIONS.LAST_JOB},
       |    ${TASK_EXECUTIONS.LAST_CHUNK} = :${TASK_EXECUTIONS.LAST_CHUNK},
       |    ${TASK_EXECUTIONS.LAST_MODIFIED_DATE} = :${TASK_EXECUTIONS.LAST_MODIFIED_DATE},
       |    ${TASK_EXECUTIONS.END_DATE} = :${TASK_EXECUTIONS.END_DATE},
       |    ${TASK_EXECUTIONS.LOG_SIZE} = :${TASK_EXECUTIONS.LOG_SIZE}
       |  WHERE
       |    ${TASK_EXECUTIONS.TASK_ID_HASH} = :${TASK_EXECUTIONS.TASK_ID_HASH}
       |  AND
       |    ${TASK_EXECUTIONS.EXECUTION_ID} = :${TASK_EXECUTIONS.EXECUTION_ID}
       |""".stripMargin

  private val STMT_UPDATE_TRUNCATED =
    s"""UPDATE ${TASK_EXECUTIONS.TABLE}
       |  SET
       |    ${TASK_EXECUTIONS.TRUNCATED} = :${TASK_EXECUTIONS.TRUNCATED}
       |  WHERE
       |    ${TASK_EXECUTIONS.TASK_ID_HASH} = :${TASK_EXECUTIONS.TASK_ID_HASH}
       |  AND
       |    ${TASK_EXECUTIONS.EXECUTION_ID} = :${TASK_EXECUTIONS.EXECUTION_ID}
       |""".stripMargin


  private val STMT_DELETE_ROW =
    s"""DELETE FROM ${TASK_EXECUTIONS.TABLE}
       |  WHERE
       |    ${TASK_EXECUTIONS.TASK_ID_HASH} = :${TASK_EXECUTIONS.TASK_ID_HASH}
       |  AND
       |    ${TASK_EXECUTIONS.EXECUTION_ID} = :${TASK_EXECUTIONS.EXECUTION_ID}
       |""".stripMargin

  private val STMT_FIND_TASK_EXECUTION =
    s"""SELECT * FROM ${TASK_EXECUTIONS.TABLE}
       |  WHERE ${TASK_EXECUTIONS.TASK_ID_HASH} = ?
       |  AND ${TASK_EXECUTIONS.EXECUTION_ID} = ?
       |""".stripMargin

}
