package com.xebialabs.xlrelease.repository.sql.query

import com.xebialabs.xlrelease.db.sql.SqlBuilder.Dialect
import com.xebialabs.xlrelease.domain.UserProfile
import com.xebialabs.xlrelease.repository.query._
import com.xebialabs.xlrelease.repository.sql.persistence.Schema.USER_PROFILE
import com.xebialabs.xlrelease.repository.sql.persistence.{CompressionSupport, UserProfileMapper}
import com.xebialabs.xlrelease.repository.sql.query.SqlUserProfileQueryBuilder.profileAlias
import com.xebialabs.xlrelease.views.users.UserFilters
import org.springframework.data.domain.Sort
import org.springframework.data.domain.Sort.Order
import org.springframework.jdbc.core.namedparam.NamedParameterJdbcTemplate

import scala.jdk.StreamConverters._

class SqlUserProfileQueryBuilder(val dialect: Dialect, val namedTemplate: NamedParameterJdbcTemplate)
  extends UserProfileQueryBuilder
    with FilterQueryBuilderSupport[UserFilters, UserProfile]
    with CompressionSupport
    with UserProfileMapper {

  implicit private val databaseDialect: Dialect = dialect

  import WhereClause._
  import com.xebialabs.xlrelease.repository.sql.persistence.Utils._

  override def withSortParameters(sorts: (String, String)*): SelfType = {
    this.sortOrderMapping ++= sorts
    this
  }

  override def build(): PageableQuery[UserProfile] = {
    withSortParameters(
      "fullName" -> USER_PROFILE.FULL_NAME,
      "username" -> USER_PROFILE.USERNAME,
      "lastActive" -> USER_PROFILE.LAST_ACTIVE,
      "loginAllowed" -> USER_PROFILE.ENABLED
    )
    buildSortOrderClause(s"$profileAlias.${USER_PROFILE.USERNAME}")
    val resultsQueryString = pageableQuery(queryTemplate)
    val resultsQuery = new SqlListQuery[UserProfile](namedTemplate, resultsQueryString, queryParams.toMap, userProfileRowMapper)
    val totalCountQuery = new SqlQuery[Long](namedTemplate, totalQueryTemplate, queryParams.toMap, (rs, _) => rs.getLong(1))
    new SqlPageableQuery[UserProfile](namedTemplate, this.pageable, resultsQuery, totalCountQuery)
  }

  override def from(filter: UserFilters): SelfType = {
    for {
      wc <- WhereClause()
      wc <- trim(filter.username).fold(wc)(username => wc.orLike(s"$profileAlias.${USER_PROFILE.USERNAME}", USER_PROFILE.USERNAME, username))
      wc <- trim(filter.email).fold(wc)(email => wc.orLike(s"$profileAlias.${USER_PROFILE.EMAIL}", USER_PROFILE.EMAIL, email))
      wc <- trim(filter.fullName).fold(wc)(fullName => wc.orLike(s"$profileAlias.${USER_PROFILE.FULL_NAME}", USER_PROFILE.FULL_NAME, fullName))
      wc <- if (filter.onlyDisabled) {
        wc.and(s"$profileAlias.${USER_PROFILE.ENABLED} = :${USER_PROFILE.ENABLED}", USER_PROFILE.ENABLED -> false.asInteger)
      } else {
        wc
      }
    } {
      if (wc.nonEmpty) {
        whereClauses.addOne(wc.clause)
        queryParams.addAll(wc.params)
      }
    }
    this
  }

  override def buildSortOrderClause(defaultOrderColumn: String): Unit = {
    val defaultOrder = Order.asc(defaultOrderColumn)
    val defaultSort = Sort.by(defaultOrder)
    val orders: Seq[Order] = if (pageable != null) {
      pageable.getSortOr(defaultSort).get().toScala(Seq)
    } else {
      Seq(defaultOrder)
    }
    val (validOrders, invalidOrders) = orders.partition(order => {
      if (order.getProperty.contains("licenseStatus")) {
        val sortOrder = if (order.getDirection.isAscending) "ASC" else "DESC"
        orderClauses += s"CASE WHEN ${USER_PROFILE.LAST_ACTIVE} IS NOT NULL AND ${USER_PROFILE.ENABLED} = 1 THEN 1 ELSE 0 END $sortOrder"
        false
      } else {
        sortOrderMapping.contains(order.getProperty)
      }
    })
    if (invalidOrders.nonEmpty) {
      invalidOrders.filter(_.getProperty.contains("licenseStatus:")).foreach(order => logger.warn(s"Ignoring unexpected sort order parameter: ${order.getProperty}"))
    }
    validOrders.map(order => s"${sortOrderMapping.getOrElse(order.getProperty, s"could not find mapping for ${order.getProperty}")} ${order.getDirection}")
      .foreach(o => orderClauses += o)
  }

  private def queryTemplate =
    s"""
       |SELECT
       |  $profileAlias.${USER_PROFILE.USERNAME},
       |  $profileAlias.${USER_PROFILE.EMAIL},
       |  $profileAlias.${USER_PROFILE.FULL_NAME},
       |  $profileAlias.${USER_PROFILE.ENABLED},
       |  $profileAlias.${USER_PROFILE.CONTENT},
       |  $profileAlias.${USER_PROFILE.LAST_ACTIVE}
       | FROM ${USER_PROFILE.TABLE} $profileAlias
       | $whereClause
       | $orderClause
       |""".stripMargin

  private def totalQueryTemplate =
    s"""
       |SELECT COUNT(1)
       | FROM ${USER_PROFILE.TABLE} $profileAlias
       | $whereClause
       |""".stripMargin
}

object SqlUserProfileQueryBuilder {
  val profileAlias = "up"
}