package com.xebialabs.xlrelease.repository.sql

import com.xebialabs.deployit.repository.{ItemAlreadyExistsException, ItemInUseException}
import com.xebialabs.xlrelease.db.sql.SqlBuilder.Dialect
import com.xebialabs.xlrelease.db.sql.transaction.{IsReadOnly, IsTransactional}
import com.xebialabs.xlrelease.domain._
import com.xebialabs.xlrelease.domain.events.ReleaseCreationSource
import com.xebialabs.xlrelease.domain.status.ReleaseStatus.TEMPLATE
import com.xebialabs.xlrelease.domain.status.{PhaseStatus, ReleaseStatus}
import com.xebialabs.xlrelease.domain.utils.syntax._
import com.xebialabs.xlrelease.exception.LogFriendlyNotFoundException
import com.xebialabs.xlrelease.repository.ReleaseJsonParser.PhaseLevel
import com.xebialabs.xlrelease.repository._
import com.xebialabs.xlrelease.repository.query.{ReleaseBasicData, ReleaseBasicDataExt}
import com.xebialabs.xlrelease.repository.sql.persistence.CiId._
import com.xebialabs.xlrelease.repository.sql.persistence.CommentPersistence.CommentRow.fromComment
import com.xebialabs.xlrelease.repository.sql.persistence.CommentPersistence.TaskCommentRow
import com.xebialabs.xlrelease.repository.sql.persistence.DependencyPersistence.TaskDependency
import com.xebialabs.xlrelease.repository.sql.persistence.TaskTagsPersistence.TaskTag
import com.xebialabs.xlrelease.repository.sql.persistence.TasksSqlBuilder.normalizeTag
import com.xebialabs.xlrelease.repository.sql.persistence._
import com.xebialabs.xlrelease.repository.sql.persistence.configuration.ConfigurationReferencePersistence
import com.xebialabs.xlrelease.repository.sql.persistence.reference.ReleaseCategoryReferencePersistence
import com.xebialabs.xlrelease.risk.domain.progress.ReleaseProgress
import com.xebialabs.xlrelease.security.SecuredCi
import com.xebialabs.xlrelease.serialization.json.repository.ResolveOptions
import com.xebialabs.xlrelease.utils.Diff
import grizzled.slf4j.Logging
import io.micrometer.core.annotation.Timed

import java.util.Collections._
import java.util.{Date, List => JList, Set => JSet}
import scala.jdk.CollectionConverters._
import scala.util.Try

//scalastyle:off number.of.methods
@IsTransactional
class SqlReleaseRepository(protected val releasePersistence: ReleasePersistence,
                           protected val phasePersistence: PhasePersistence,
                           protected val taskPersistence: TaskPersistence,
                           protected val dependencyPersistence: DependencyPersistence,
                           protected val commentPersistence: CommentPersistence,
                           protected val teamRepository: TeamRepository,
                           protected val releaseExtensionsRepository: ReleaseExtensionsRepository,
                           val configurationPersistence: ConfigurationReferencePersistence,
                           val repositoryAdapter: SqlRepositoryAdapter,
                           facetRepositoryDispatcher: FacetRepositoryDispatcher,
                           categoryReferencePersistence: ReleaseCategoryReferencePersistence,
                           categoryPersistence: CategoryPersistence
                          )(implicit val sqlDialect: Dialect)
  extends ReleaseRepository
    with InterceptedRepository[Release]
    with Logging
    with DeserializationSupport
    with ConfigurationReferencesSupport
    with SqlReleaseRepositoryHelper {

  @Timed
  @IsReadOnly
  override def exists(id: String): Boolean = {
    releasePersistence.existsRelease(id.normalized)
  }

  @Timed
  override def create(release: Release, releaseCreationSource: ReleaseCreationSource): Release = {
    if (releasePersistence.existsRelease(release.getId)) {
      throw new ItemAlreadyExistsException("A release with ID [%s] already exists", release.getId)
    }
    interceptCreate(release)

    releasePersistence.insert(release)

    val releaseUid = release.getCiUid
    val phases = release.getPhases.asScala.toSet
    batchInsertPhases(releaseUid, phases)

    val tasks = release.getAllTasks.asScala.toSet
    batchInsertTasks(releaseUid, tasks)

    release.getExtensions.forEach(releaseExtensionsRepository.create[ReleaseExtension])
    updateConfigurationRefs(release)
    updateCategoryReferences(release)

    facetRepositoryDispatcher.liveRepository.createFromTasks(tasks.toSeq)
    release
  }

  private def batchInsertPhases(releaseUid: Integer, phases: Set[Phase]): Unit = {
    if (phases.nonEmpty) {
      phasePersistence.batchInsert(phases, releaseUid)
    }
  }

  private def batchInsertTasks(releaseUid: Integer, tasks: Set[Task]): Unit = {
    taskPersistence.batchInsert(tasks, releaseUid)
    // it looks like some databases do not return generated keys after executeBatch is called
    val taskUidsByTaskId: Map[String, TaskCiUid] = taskPersistence.releaseTaskUids(releaseUid)
    tasks.foreach(task => task.setCiUid(taskUidsByTaskId(Ids.getFolderlessId(task.getId))))
    val taskTags = for {
      t <- tasks
      tag <- Set.from(t.getTags.asScala.map(normalizeTag))
      taskUid <- taskUidsByTaskId.get(t.getId.shortId)
    } yield TaskTag(taskUid, tag)
    taskPersistence.batchInsertTags(taskTags.toList)

    val taskCommentRows = for {
      t <- tasks
      comment <- t.getComments.asScala
      taskUid <- taskUidsByTaskId.get(t.getId.shortId)
    } yield TaskCommentRow(taskUid, commentRow = fromComment(comment))
    commentPersistence.batchCreate(taskCommentRows)

    val taskDependencies = for {
      t <- tasks
      dependency <- t.dependencies
      taskUid <- taskUidsByTaskId.get(t.getId.shortId)
    } yield TaskDependency(taskUid, dependency)
    dependencyPersistence.batchInsert(taskDependencies)

  }

  @Timed
  override def findById(id: String): Release = {
    // TODO this shouldn't load everything, to fix on 9.1
    getRelease(id, ResolveOptions.WITH_DECORATORS)
  }

  @Timed
  override def findById(id: String, resolveOptions: ResolveOptions): Release = {
    getRelease(id, resolveOptions)
  }

  @Timed
  @IsReadOnly
  override def findIdsByKindAndStatus(releaseKinds: Seq[ReleaseKind], statuses: Seq[ReleaseStatus]): Seq[String] = {
    val sqlWithParams = new ReleasesSqlBuilder()
      .selectReleaseId()
      .withOneOfStatuses(statuses)
      .withAnyOfKinds(releaseKinds)
      .build()
    releasePersistence.findReleaseIdsByQuery(sqlWithParams).toSeq
  }

  @Timed
  override def findByCalendarToken(calendarToken: String): Release = {
    val sqlWithParams = new ReleasesSqlBuilder()
      .selectReleaseData()
      .withCalendarToken(calendarToken)
      .build()
    val releases = releasePersistence.findReleaseDatasByQuery(sqlWithParams)
    releases
      .headOption
      .map(deserializeRelease)
      .orNull
  }

  @Timed
  @IsReadOnly
  def findArchivableReleaseIds(date: Date, pageSize: Int): Seq[String] = {
    val sqlWithParams = new ReleasesSqlBuilder()
      .selectReleaseId()
      .withOneOfStatuses(Seq(ReleaseStatus.COMPLETED, ReleaseStatus.ABORTED))
      .withEndDateBefore(date)
      .withPage(Page(0, pageSize, 0))
      .withPreArchived()
      .build()

    releasePersistence.findReleaseIdsByQuery(sqlWithParams).toSeq
  }

  @Timed
  override def setPreArchived(releaseId: String, preArchived: Boolean): Unit = {
    releasePersistence.setPreArchived(releaseId, preArchived)
  }

  @Timed
  @IsReadOnly
  def findPreArchivableReleases(page: Int, pageSize: Int): Seq[Release] = {
    val inactive = Seq(ReleaseStatus.COMPLETED, ReleaseStatus.ABORTED)
    val sqlWithParams = new ReleasesSqlBuilder()
      .selectReleaseData()
      .withOneOfStatuses(inactive)
      .withPreArchived(false)
      .limitAndOffset(pageSize, page * pageSize)
      .build()

    releasePersistence.findReleaseDatasByQuery(sqlWithParams)
      .flatMap(tryDeserializeRelease)
      .filterNot(_.isTutorial)
      .map(releaseExtensionsRepository.decorate)
      .map(commentPersistence.decorate)
  }


  @Timed
  @IsReadOnly
  def findSubReleases(parentReleaseId: String): JList[ReleaseBasicDataExt] = {
    releasePersistence
      .findBasicReleaseRowByQuery(parentReleaseId)
      .map(r => ReleaseBasicDataExt(if (r.folderId != null) {
        r.folderId + "/" + r.releaseId
      } else {
        r.releaseId
      }, r.title, r.status, r.startDate, r.endDate))
      .asJavaMutable()
  }

  @Timed
  override def search(searchParams: ReleaseSearchByParams): JList[Release] = searchParams match {
    case ReleaseSearchByParams(page, folderId, statuses, title, rootReleaseId, autoStart) =>
      val sqlBuilder = {
        val base = new ReleasesSqlBuilder()
          .selectReleaseData()
          .withTitle(title)
          .withRootReleaseId(rootReleaseId)
          .withPage(page)

        if (folderId == null) {
          base
        } else {
          folderId.fold(base.withFolder, base.withAncestor)
        }
      }
      if (statuses.nonEmpty) {
        sqlBuilder.withOneOfStatuses(statuses.toIndexedSeq)
      }
      if (autoStart) {
        sqlBuilder.withAutoStart()
      }
      releasePersistence
        .findReleaseDatasByQuery(sqlBuilder.build())
        .flatMap(tryDeserializeRelease)
        .map(releaseExtensionsRepository.decorate)
        .map(commentPersistence.decorate).asJava
  }

  @Timed
  override def delete(id: String, failIfReferenced: Boolean): Unit = {
    if (failIfReferenced) checkIsNotReferencedByDependencies(id)
    releasePersistence.findUidByReleaseId(id).foreach { uid =>
      deleteReleaseReferences(id, uid.intValue())
    }
    releasePersistence.deleteById(id)
  }

  private def deleteReleaseReferences(id: String, releaseUid: Int): Unit = {
    dependencyPersistence.deleteByReleaseUid(releaseUid)
    val taskCiUids = taskPersistence.findTaskCiUidsByReleaseCiUid(releaseUid)
    dependencyPersistence.deleteByTaskUids(taskCiUids)
    commentPersistence.deleteByRelease(releaseUid, taskCiUids)
    taskPersistence.deleteTasksByReleaseUid(releaseUid)
    phasePersistence.deletePhasesByReleaseUid(releaseUid)
    teamRepository.deleteTeamsFromPlatform(new SecuredCi(id, releaseUid))
    releaseExtensionsRepository.deleteAll(id)
    deleteConfigurationRefs(releaseUid)
    interceptDelete(id)
  }

  @Timed
  override def deleteWithUid(id: String, releaseUid: Int): Unit = {
    deleteReleaseReferences(id, releaseUid)
    releasePersistence.deleteById(id)
  }

  @Timed
  override def move(originalId: String, newId: String): Unit = {
    releasePersistence.move(originalId, newId)
  }

  override def getUid(id: String): Option[CiUid] = releasePersistence.findUidByReleaseId(id)

  @Timed
  override def getReleaseKind(id: String): ReleaseKind = {
    val kind = releasePersistence.findReleaseKind(id.normalized)
      .getOrElse(throw new LogFriendlyNotFoundException("Release [%s] not found", id))
    ReleaseKind.valueOf(kind.toUpperCase)
  }

  @Timed
  override def getStatus(id: String): ReleaseStatus = {
    val status = releasePersistence.findReleaseStatus(id.normalized).orNull
    if (status == null) {
      null
    } else {
      ReleaseStatus.valueOf(status.toUpperCase)
    }
  }

  @Timed
  override def getStatuses(ids: Seq[String]): Seq[ReleaseStatus] = {
    if (ids.isEmpty) {
      Seq()
    } else {
      releasePersistence.findReleaseStatuses(ids).map {
        case (_, status) => ReleaseStatus.valueOf(status.toUpperCase)
      }
    }
  }

  @Timed
  override def getStatusesWithIds(ids: Seq[String]): Seq[(String, ReleaseStatus)] = {
    if (ids.isEmpty) {
      Seq()
    } else {
      releasePersistence.findReleaseStatuses(ids).map {
        case (id, status) => (id, ReleaseStatus.valueOf(status.toUpperCase))
      }
    }
  }

  @Timed
  @IsReadOnly
  override def getRiskScores(ids: Seq[String]): Seq[Int] = {
    releasePersistence.findReleaseRiskScores(ids)
  }

  @Timed
  override def setRiskScores(releaseId: String, score: Int, totalScore: Int): Unit =
    releasePersistence.updateReleaseRiskScores(releaseId, score, totalScore)

  @Timed
  @IsReadOnly
  override def isTemplate(releaseId: String): Boolean =
    TEMPLATE == getStatus(releaseId)

  @Timed
  @IsReadOnly
  override def isWorkflow(releaseId: String): Boolean = {
    ReleaseKind.WORKFLOW == getReleaseKind(releaseId)
  }

  @Timed
  @IsReadOnly
  override def getTitle(id: String): String =
    releasePersistence.findReleaseTitle(id.normalized)
      .getOrElse(throw new LogFriendlyNotFoundException("Release [%s] not found", id))

  @Timed
  @IsReadOnly
  override def getOwner(id: String): String =
    releasePersistence.findReleaseOwner(id.normalized)
      .getOrElse(throw new LogFriendlyNotFoundException("Release [%s] not found", id))

  @Timed
  override def update(release: Release): Release = {
    update(null, release)
  }

  @Timed
  override def update(original: Release, updated: Release): Release = {
    val originalReleaseOpt = Option(original)

    interceptUpdate(updated)

    releasePersistence.update(Option(original), updated)

    originalReleaseOpt.foreach { rel =>
      val phases = Diff(
        rel.getPhases.asScala,
        updated.getPhases.asScala
      ).newValues.toSet
      batchInsertPhases(releaseUid = updated.getCiUid, phases)

      // phase restart can add new tasks and is done through release update
      val tasks = Diff(
        rel.getAllTasks.asScala,
        updated.getAllTasks.asScala
      ).newValues.toSet
      batchInsertTasks(releaseUid = updated.getCiUid, tasks)
    }
    updateConfigurationRefs(updated)
    updateCategoryReferences(updated)

    updated
  }

  private def updateCategoryReferences(release: Release): Unit = {
    if (release.isTemplate) {
      val currentCategoryIds = categoryReferencePersistence.getTargetIds(release.getCiUid)
      val categories = Category.sanitizeTitles(release.getCategories).asScala

      val (existingTitles, existingIds) = if (categories.nonEmpty) {
        categoryPersistence.findByTitles(categories.toSeq).map(c => c.getTitle -> c.getCiUid).unzip
      } else {
        (Seq.empty, Seq.empty)
      }
      val missingTitles = categories.filterNot(title => existingTitles.contains(title))
      val updatedCategoryIds = missingTitles.foldLeft(existingIds) { case (ids, title) =>
        val category = new Category()
        category.setTitle(title)
        category.setActive(true)
        val created = categoryPersistence.create(category)
        ids :+ created.getCiUid
      }

      val diff = Diff(currentCategoryIds, updatedCategoryIds)
      categoryReferencePersistence.delete(release.getCiUid, diff.deletedValues.toSet)
      categoryReferencePersistence.insert(release.getCiUid, diff.newValues.toSet)
    }
  }

  //noinspection ScalaStyle
  @Timed
  override def replace(original: Release, updated: Release): Release = {

    if (Ids.isFolderId(original.getId)) {
      updated.setTeams(emptyList())
    }

    interceptUpdate(updated)
    releasePersistence.update(Option(original), updated)
    if (original != null) {
      // phase restart can add new tasks and is done through release update
      replacePhases(original, updated)
      replaceTasks(original, updated)
      replaceReleaseExtensions(original, updated)
      replaceReleaseTeams(original, updated)
      replaceTemplateLogo(original, updated)
    }
    updateConfigurationRefs(updated)
    updateCategoryReferences(updated)
    updated
  }


  @Timed
  @IsReadOnly
  override def getAllTags(limitNumber: Int): JSet[String] = releasePersistence.findAllTags(limitNumber).asJava

  private def getRelease(releaseId: CiId, resolveOptions: ResolveOptions): Release = {
    val release = repositoryAdapter.read[Release](releaseId, resolveOptions)
    if (null == release) {
      throw new LogFriendlyNotFoundException(s"Repository entity [$releaseId] not found")
    }
    if (resolveOptions.hasDecorators) {
      releaseExtensionsRepository.decorate(release)
      commentPersistence.decorate(release) // <-- can be HUGE! ENG-8031: lazy loading from UI side?
    }
    release
  }

  private def checkIsNotReferencedByDependencies(planItemIdOrItsChildren: CiId): Unit = {
    val externalIncomingDependencies = dependencyPersistence.findByPartialTargetIds(Set(planItemIdOrItsChildren), None)
      .map(_.fullDependencyId)
      .filterNot(_.startsWith(planItemIdOrItsChildren.normalized))
    if (externalIncomingDependencies.nonEmpty) {
      throw new ItemInUseException(s"Cannot delete [$planItemIdOrItsChildren] because it or one of its children is referenced by " +
        s"one or more dependencies: [${externalIncomingDependencies.mkString(", ")}]")
    }
  }

  @Timed
  @IsReadOnly
  override def findSCMDataById(id: String): Option[Integer] = {
    logger.debug(s"Finding scm data for template or release with id $id")
    releasePersistence.findSCMDataById(id).map(Integer.valueOf)
  }

  @Timed
  @IsReadOnly
  override def findOverdueReleaseIds(): Seq[String] = {
    releasePersistence.findOverdueReleaseIds()
  }

  @Timed
  @IsReadOnly
  override def getReleaseJson(releaseId: String): String = {
    releasePersistence.getReleaseJson(releaseId)
      .getOrElse(throw new LogFriendlyNotFoundException(s"Repository entity [$releaseId] not found"))
  }

  @Timed
  @IsReadOnly
  override def getFullId(releaseId: String): String = {
    val row = releasePersistence.getBasicReleaseRow(releaseId)
      .getOrElse(throw new LogFriendlyNotFoundException(s"Repository entity [$releaseId] not found"))
    Ids.formatWithFolderId(row.folderId, row.releaseId)
  }

  @Timed
  @IsReadOnly
  override def getReleasesWithoutComments(releaseIds: List[String]): List[Release] = {
    val sqlWithParams = new ReleasesSqlBuilder().selectReleaseData()
      .withReleaseIds(releaseIds).build()
    val releases = releasePersistence.findReleaseDatasByQuery(sqlWithParams)
      .flatMap(tryDeserializeRelease)
      .map(releaseExtensionsRepository.decorate)

    releases.sortBy(r => releaseIds.indexOf(Ids.getName(r.getId))).toList
  }

  @Timed
  @IsReadOnly
  override def getTemplatesWithDefaultTargetFolder(folderId: String): List[ReleaseBasicData] = {
    releasePersistence.getTemplatesWithDefaultTargetFolder(folderId)
  }

  @Timed
  @IsReadOnly
  override def getReleaseInformation(releaseId: String): Option[ReleaseInformation] = {
    // TODO maybe we should use smaller query if we do not need folder information, risk etc.
    //  plus it can be read only
    val basicRow = releasePersistence.getBasicReleaseRow(releaseId)
    basicRow.map { r =>
      val fullReleaseId = Ids.formatWithFolderId(r.folderId, r.releaseId)
      val releaseKind = ReleaseKind.fromString(r.kind)
      val releaseStatus = ReleaseStatus.valueOf(r.status.toUpperCase)
      val isArchived = false
      val optionalOwner = Option(r.owner)
      ReleaseInformation(fullReleaseId, releaseKind, releaseStatus, isArchived, optionalOwner)
    }
  }

  @Timed
  @IsReadOnly
  override def findAbortableReleaseIds(date: Date, pageSize: Option[Long]): Seq[ReleaseIdWithCiUid] = {
    releasePersistence.findAbortableReleaseIds(date, pageSize)
  }

  @Timed
  @IsReadOnly
  override def findAbortableTaskIdHashes(releaseCiUid: CiUid): Seq[Hash] = {
    releasePersistence.findAbortableTaskIdHashes(releaseCiUid)
  }

  @Timed
  @IsReadOnly
  override def countTemplatesByKind(releaseKind: ReleaseKind): Int = {
    releasePersistence.countTemplatesByKind(releaseKind)
  }

  @Timed
  @IsReadOnly
  override def getTenantIdForRelease(releaseId: TenantId): TenantId = {
    releasePersistence.getTenantIdForRelease(Ids.getName(releaseId))
  }

  @Timed
  @IsReadOnly
  override def tenantTemplateCountByKind(tenantId: TenantId, releaseKind: ReleaseKind): Integer = {
    releasePersistence.tenantTemplateCountByKind(tenantId, releaseKind)
  }

  @Timed
  @IsReadOnly
  override def tenantReleaseCountByKind(tenantId: TenantId, releaseKind: ReleaseKind): Integer = {
    releasePersistence.tenantReleaseCountByKind(tenantId, releaseKind)
  }

  override def getPhases(releaseId: Hash): JList[Phase] = {
    val releaseUid = releasePersistence.findUidByReleaseId(releaseId).orNull
    val releaseData = for {
      json <- releasePersistence.getReleaseJson(releaseId)
      parsed <- Try(ReleaseJsonParser.parse(json, PhaseLevel)).toOption
    } yield parsed

    releaseData.fold(List.empty[Phase]) { data =>
      for {
        phaseData <- data.phases
      } yield {
        val phase = new Phase()
        phase.setReleaseUid(releaseUid)
        phase.setId(phaseData.id)
        phase.setTitle(phaseData.title)
        phase.setColor(phaseData.color)
        phase.setStatus(PhaseStatus.valueOf(phaseData.status))
        phase
      }
    }.asJava
  }

  override def getProgress(releaseId: Hash): ReleaseProgress = {
    val progressId = releaseId + Ids.SEPARATOR + ReleaseProgress.PROGRESS_PREFIX
    releaseExtensionsRepository.read[ReleaseProgress](progressId).orNull
  }
}
