package com.xebialabs.xlrelease.service

import com.xebialabs.deployit.plugin.api.reflect.Type
import com.xebialabs.xlrelease.db.sql.Sql
import com.xebialabs.xlrelease.db.sql.SqlBuilder.Dialect
import com.xebialabs.xlrelease.domain.status.{ReleaseStatus, TaskStatus}
import com.xebialabs.xlrelease.domain.{ParallelGroup, ReleaseKind, SequentialGroup}
import com.xebialabs.xlrelease.repository.sql.persistence.Schema.{FOLDERS, PHASES, RELEASES, TASKS}
import com.xebialabs.xlrelease.repository.sql.persistence.TaskPersistence.TaskRowsQueryParams
import com.xebialabs.xlrelease.repository.sql.persistence.TasksSqlBuilder
import com.xebialabs.xlrelease.views.TasksFilters
import org.springframework.util.StringUtils._

import scala.jdk.CollectionConverters._

// TODO move next to TaskPersistence (where it's used)
object SqlTasksFilterSupport {
  val releaseStatus = "releaseStatus"
  private val statusPriorityColumn = "statusPriorityColumn"
  private val releasesAlias = "releases"
  private val phasesAlias = "phases"
  private val tasksAlias = "tasks"
  private val folderAlias = "folders"

  // status priority is computed value from task status required for ordering by "status"
  private def statusPriorityStmt: String = {
    s"""CASE
       |   WHEN $tasksAlias.${TASKS.STATUS} = 'failed' THEN 1
       |   WHEN $tasksAlias.${TASKS.STATUS} = 'failing' THEN 2
       |   WHEN $tasksAlias.${TASKS.STATUS} = 'facet_check_in_progress' THEN 3
       |   WHEN $tasksAlias.${TASKS.STATUS} = 'precondition_in_progress' THEN 4
       |   WHEN $tasksAlias.${TASKS.STATUS} = 'waiting_for_input' THEN 5
       |   WHEN $tasksAlias.${TASKS.STATUS} = 'failure_handler_in_progress' THEN 6
       |   WHEN $tasksAlias.${TASKS.STATUS} = 'in_progress' THEN 7
       |   WHEN $tasksAlias.${TASKS.STATUS} = 'pending' THEN 8
       |   WHEN $tasksAlias.${TASKS.STATUS} = 'planned' THEN 9
       |   ELSE 10
       |END $statusPriorityColumn
       |""".stripMargin
  }

  private val TASK_ROWS = Seq(
    s"""tasks.${TASKS.TASK_ID}""",
    s"""tasks.${TASKS.CI_UID}""",
    s"""tasks.${TASKS.RELEASE_UID}""",
    s"""tasks.${TASKS.TASK_TYPE}""",
    s"""tasks.${TASKS.TITLE}""",
    s"""tasks.${TASKS.STATUS}""",
    s"""tasks.${TASKS.STATUS_LINE}""",
    s"""tasks.${TASKS.OWNER}""",
    s"""tasks.${TASKS.TEAM}""",
    s"""tasks.${TASKS.START_DATE}""",
    s"""tasks.${TASKS.END_DATE}""",
    s"""tasks.${TASKS.IS_AUTOMATED}""",
    s"""tasks.${TASKS.LOCKED}""",
    s"""tasks.${TASKS.PLANNED_DURATION}""",
    statusPriorityStmt
  )

  def sqlBuilderByFilters(queryParams: TaskRowsQueryParams,
                          enforcePermission: Boolean = true)
                         (implicit sqlDialect: Dialect): TasksSqlBuilder = {
    val columns = TASK_ROWS ++ Seq(
      s"$releasesAlias.${RELEASES.SECURITY_UID}",
      s"$releasesAlias.${RELEASES.RELEASE_ID}",
      s"$releasesAlias.${RELEASES.RELEASE_TITLE}",
      s"$releasesAlias.${RELEASES.STATUS} $releaseStatus",
      s"$phasesAlias.${PHASES.PHASE_TITLE}",
      s"$folderAlias.${FOLDERS.FOLDER_PATH}",
      s"$folderAlias.${FOLDERS.FOLDER_ID}",
    )

    getSqlBuilderByFilters(columns, queryParams, addOrderClause = true, enforcePermission)(sqlDialect)
  }

  def sqlTotalQueryBuilderByFilters(queryParams: TaskRowsQueryParams,
                                    enforcePermission: Boolean = true)
                                   (implicit sqlDialect: Dialect): TasksSqlBuilder = {
    val columns = Seq("COUNT(1)")
    getSqlBuilderByFilters(columns, queryParams, addOrderClause = false, enforcePermission)(sqlDialect)
  }

  // scalastyle:off
  private def getSqlBuilderByFilters(columns: Seq[String],
                                     queryParams: TaskRowsQueryParams,
                                     addOrderClause: Boolean,
                                     enforcePermission: Boolean)
                                    (implicit sqlDialect: Dialect)
  : TasksSqlBuilder = {
    val filters: TasksFilters = queryParams.tasksFilters
    val taskTypes = queryParams.allowedTaskTypes
    val allTaskTypes = queryParams.allTaskTypes
    val principals = queryParams.principals
    val roleIds = queryParams.roleIds
    val username = queryParams.username
    val isAdmin = queryParams.isAdmin

    val builder = new TasksSqlBuilder()

    val preSelect =
      s"""SELECT
         | ${columns.mkString(", ")}
         |FROM
         | ${TASKS.TABLE} $tasksAlias
         | JOIN ${RELEASES.TABLE} $releasesAlias ON $tasksAlias.${TASKS.RELEASE_UID} = $releasesAlias.${RELEASES.CI_UID}
         |""".stripMargin
    val preParams = Seq.empty
    builder.select(Sql(preSelect, preParams))
    builder.joinPhases
    builder.joinFolders
    builder.withReleaseKind(ReleaseKind.RELEASE)
    builder.withOneOfReleaseStatuses(Seq(ReleaseStatus.ACTIVE_STATUSES: _*))
    builder.withOneOfTaskStatuses(getTaskStatuses(filters).toSeq)

    // should also check if equal to all task types (or task types - task groups)
    if (taskTypes.isEmpty || allTaskTypes.forall(taskTypes.contains(_))) { // todo: use set and equals?
      builder.withoutTaskTypes(Seq(Type.valueOf(classOf[ParallelGroup]), Type.valueOf(classOf[SequentialGroup])))
    }
    else {
      val filtered = taskTypes.filterNot(t => t == Type.valueOf(classOf[ParallelGroup]) || t == Type.valueOf(classOf[SequentialGroup]))
      builder.withTaskTypes(filtered)
    }

    builder.withFilter(filters.getFilter)
    builder.withDates(filters.getFrom, filters.getTo)
    builder.withAnyOfTags(filters.getTags.asScala.toSet)
    if (hasText(filters.getFolderId)) {
      builder.withFolderIds(Set(filters.getFolderId))
    }

    if (addOrderClause) {
      // order by release and task uid is needed because of limit
      //   `offset 0 rows fetch next 50 rows only` would work - otherwise query is not valid
      filters.getOrderBy match {
        case "due_date" =>
          builder.orderBy(s"CASE WHEN $tasksAlias.${TASKS.END_DATE} IS NULL THEN 1 ELSE 0 END")
          builder.orderBy(s"$tasksAlias.${TASKS.END_DATE}")
        case "start_date" =>
          builder.orderBy(s"CASE WHEN $tasksAlias.${TASKS.START_DATE} IS NULL THEN 1 ELSE 0 END")
          builder.orderBy(s"$tasksAlias.${TASKS.START_DATE}")
        case "status" =>
          builder.orderBy(s"$statusPriorityColumn")
        case _ => ()
      }
      builder.orderBy(s"$tasksAlias.${TASKS.RELEASE_UID}")
      builder.orderBy(s"$tasksAlias.${TASKS.CI_UID}")
    }

    if (enforcePermission) {
      if (filters.isAssignedToAnybody) {
        builder.withAssignedToAnybody(principals, roleIds, isAdmin)
      } else if (filters.isAssignedToMyTeams) {
        builder.withAssignedToMyTeams(principals, roleIds)
      } else {
        builder.withAssignedToMe(username)
      }
    }

    builder
  }

  private def getTaskStatuses(filters: TasksFilters) = if (filters.getStatuses.isEmpty) {
    TaskStatus.values().toSeq.filter(ts => !ts.isDone && !ts.isDoneInAdvance)
  } else {
    filters.getStatuses.asScala
  }
}
