package com.xebialabs.xlrelease.repository.sql.query

import com.xebialabs.xlrelease.api.v1.filters.RolePrincipalsFilters
import com.xebialabs.xlrelease.db.sql.DatabaseInfo
import com.xebialabs.xlrelease.db.sql.DatabaseInfo.MySql
import com.xebialabs.xlrelease.db.sql.SqlBuilder.Dialect
import com.xebialabs.xlrelease.repository.query._
import com.xebialabs.xlrelease.repository.sql.persistence.Schema.USER_PROFILE
import com.xebialabs.xlrelease.repository.sql.query.SqlRolePrincipalsQueryBuilder._
import com.xebialabs.xlrelease.security.GlobalRolePrincipalTableViewMapper
import com.xebialabs.xlrelease.security.SqlReleaseRoleRepository.GlobalRolePrincipalTableView
import com.xebialabs.xlrelease.security.sql.db.Ids.GLOBAL_SCOPE_CI_ID
import com.xebialabs.xlrelease.security.sql.db.SecuritySchema.{ROLES, ROLE_PRINCIPALS}
import org.springframework.jdbc.core.namedparam.NamedParameterJdbcTemplate

import scala.collection.mutable

object SqlRolePrincipalsQueryBuilder {
  val rolesAlias = "roles"
  val rolePrincipalsAlias = "rolePrincipals"
  val userProfilesAlias = "userProfiles"
}

class SqlRolePrincipalsQueryBuilder(val dbInfo: DatabaseInfo, val dialect: Dialect, val namedTemplate: NamedParameterJdbcTemplate)
  extends RolePrincipalsQueryBuilder
    with FilterQueryBuilderSupport[RolePrincipalsFilters, GlobalRolePrincipalTableView]
    with GlobalRolePrincipalTableViewMapper {

  implicit private val databaseDialect: Dialect = dialect

  import WhereClause._

  private var filter: RolePrincipalsFilters = _
  private val totalQueryWhereClauses: mutable.Buffer[String] = mutable.Buffer()

  override def from(filter: RolePrincipalsFilters): SelfType = {
    this.filter = filter
    this
  }

  private def buildWhereClauses(): Unit = {
    whereClauses.addOne(s"$rolesAlias.${ROLES.ciId} = '$GLOBAL_SCOPE_CI_ID'") // outer select only global roles
    for {
      roleNameClause <- WhereClause(s"r.${ROLES.ciId} = '$GLOBAL_SCOPE_CI_ID'")
      roleNameClause <- trim(filter.getRole()).fold(roleNameClause)(role => roleNameClause.andLike(s"r.${ROLES.name}", ROLES.name, role))
    } {
      if (roleNameClause.nonEmpty) {
        queryParams.addAll(roleNameClause.params)
      }
      val principalNameClause = trim(filter.getPrincipal()).fold(WhereClause())(p => WhereClause().andLike(s"rp.${ROLE_PRINCIPALS.principalName}", ROLE_PRINCIPALS.principalName, p))
      val roleWhereClause = if (principalNameClause.nonEmpty) {
        queryParams.addAll(principalNameClause.params)
        roleNameClause.and(
          s"EXISTS ( SELECT 1 FROM ${ROLE_PRINCIPALS.TABLE} rp WHERE rp.${ROLE_PRINCIPALS.roleId} = r.${ROLES.id} AND ${principalNameClause.clause})"
        )
      } else {
        roleNameClause
      }
      val filteredRolesQuery = s"SELECT r.${ROLES.id} FROM ${ROLES.TABLE} r WHERE ${roleWhereClause.clause}"
      // order is important - we do not want to capture pagination clause for total count query
      // total count clauses (before pagination):
      totalQueryWhereClauses.addAll(whereClauses).addOne(s"$rolesAlias.${ROLES.id} IN (${wrapSubQuery(filteredRolesQuery)})")
      // pagination clauses:
      val subQuery = if (!withoutPaging && pageable != null && !pageable.isUnpaged) {
        addLimitAndOffset(s"$filteredRolesQuery ORDER BY r.${ROLES.name} ASC", pageable.getPageSize, pageable.getOffset)
      } else {
        filteredRolesQuery
      }
      whereClauses.addOne(s"$rolesAlias.${ROLES.id} IN (${wrapSubQuery(subQuery)})")
    }
  }

  private def wrapSubQuery(subQuery: String): String = {
    dbInfo match {
      case MySql(_) => s"""select * from ($subQuery) as t"""
      case _ => subQuery
    }
  }

  override def build(): PageableQuery[GlobalRolePrincipalTableView] = {
    withSortParameters(s"$rolesAlias.${ROLES.name}" -> s"$rolesAlias.${ROLES.name}")
    buildSortOrderClause(s"$rolesAlias.${ROLES.name}")
    buildWhereClauses()
    val resultsQueryString = queryTemplate
    val globalRolePrincipalsQuery = new SqlListQuery[GlobalRolePrincipalTableView](
      namedTemplate,
      resultsQueryString,
      queryParams.toMap,
      globalRolePrincipalViewMapper)
    val totalCountQuery = new SqlQuery[Long](namedTemplate, totalQueryTemplate, queryParams.toMap, (rs, _) => rs.getLong(1))
    new SqlPageableQuery[GlobalRolePrincipalTableView](namedTemplate, this.pageable, globalRolePrincipalsQuery, totalCountQuery)
  }

  private def queryTemplate = {
    s"""SELECT
       | $rolesAlias.${ROLES.id},
       | $rolesAlias.${ROLES.name},
       | $rolesAlias.${ROLES.ciId},
       | $rolePrincipalsAlias.${ROLE_PRINCIPALS.principalName},
       | $userProfilesAlias.${USER_PROFILE.FULL_NAME}
       | FROM ${ROLES.TABLE} $rolesAlias
       |  LEFT OUTER JOIN ${ROLE_PRINCIPALS.TABLE} $rolePrincipalsAlias ON $rolesAlias.${ROLES.id} = $rolePrincipalsAlias.${ROLE_PRINCIPALS.roleId}
       |  LEFT OUTER JOIN ${USER_PROFILE.TABLE} $userProfilesAlias ON LOWER($rolePrincipalsAlias.${ROLE_PRINCIPALS.principalName}) = $userProfilesAlias.${USER_PROFILE.USERNAME}
       |  $whereClause
       |  $orderClause
       |""".stripMargin
  }

  private def totalQueryWhereClause: String = s"WHERE ${totalQueryWhereClauses.mkString(" AND ")}"

  private def totalQueryTemplate: String = s"SELECT COUNT(1) FROM ${ROLES.TABLE} $rolesAlias $totalQueryWhereClause"

}
