package com.xebialabs.xlrelease.repository.sql.persistence

import com.xebialabs.deployit.exception.NotFoundException
import com.xebialabs.deployit.repository.ItemAlreadyExistsException
import com.xebialabs.xlrelease.db.sql.transaction.{IsReadOnly, IsTransactional}
import com.xebialabs.xlrelease.domain.TemplateLogo
import com.xebialabs.xlrelease.domain.id.CiUid
import com.xebialabs.xlrelease.repository.Ids.getFolderlessId
import com.xebialabs.xlrelease.repository.ReleaseJsonParser.IdExtension
import com.xebialabs.xlrelease.repository.query.ReleaseBasicData
import com.xebialabs.xlrelease.repository.sql.persistence.CiId.CiId
import com.xebialabs.xlrelease.repository.sql.persistence.PersistenceConstants.BLOB_TYPE
import com.xebialabs.xlrelease.repository.sql.persistence.Schema.{FOLDERS, PATHS, RELEASES, TEMPLATE_METADATA}
import com.xebialabs.xlrelease.repository.sql.persistence.Utils._
import com.xebialabs.xlrelease.repository.{CiCloneHelper, Ids, TemplateMetadata}
import com.xebialabs.xlrelease.serialization.json.utils.CiSerializerHelper
import com.xebialabs.xlrelease.utils.CiHelper.rewriteWithNewId
import com.xebialabs.xlrelease.utils.FolderId
import org.springframework.dao.{DuplicateKeyException, EmptyResultDataAccessException}
import org.springframework.jdbc.core.support.SqlBinaryValue
import org.springframework.jdbc.core.{RowMapper, SqlParameterValue}
import org.springframework.util.StreamUtils

import java.nio.charset.StandardCharsets
import scala.util.Using

@IsTransactional
trait TemplateMetadataPersistence {
  self: ReleasePersistence =>

  private val STMT_INSERT_TEMPLATE_META =
    s"""
       | INSERT INTO ${TEMPLATE_METADATA.TABLE}
       | ( ${TEMPLATE_METADATA.RELEASE_UID}
       | , ${TEMPLATE_METADATA.AUTHOR}
       | , ${TEMPLATE_METADATA.TARGET_FOLDER_CI_UID}
       | , ${TEMPLATE_METADATA.ALLOW_TARGET_FOLDER_OVERRIDE}
       | , ${TEMPLATE_METADATA.DESCRIPTION}
       | , ${TEMPLATE_METADATA.LOGO_DATA}
       | )
       | VALUES
       | ( :${TEMPLATE_METADATA.RELEASE_UID}
       | , :${TEMPLATE_METADATA.AUTHOR}
       | , :${TEMPLATE_METADATA.TARGET_FOLDER_CI_UID}
       | , :${TEMPLATE_METADATA.ALLOW_TARGET_FOLDER_OVERRIDE}
       | , :${TEMPLATE_METADATA.DESCRIPTION}
       | , :${TEMPLATE_METADATA.LOGO_DATA}
       | )
       |""".stripMargin

  def createTemplateMetadata(templateMeta: TemplateMetadata): Unit = {
    val templateLogo = templateMeta.templateLogo
    val logoBytes: Array[Byte] = serializeTemplateLogo(templateLogo)
    val descriptionBytes: Array[Byte] = getDescriptionBytes(templateMeta.description)
    val targetFolderCiUid: CiUid = getTargetFolderCiUid(templateMeta.targetFolderId)

    val p = params(
      TEMPLATE_METADATA.RELEASE_UID -> templateMeta.releaseUid,
      TEMPLATE_METADATA.AUTHOR -> templateMeta.author,
      TEMPLATE_METADATA.TARGET_FOLDER_CI_UID -> targetFolderCiUid,
      TEMPLATE_METADATA.ALLOW_TARGET_FOLDER_OVERRIDE -> templateMeta.allowTargetFolderOverride.asInteger,
      TEMPLATE_METADATA.DESCRIPTION -> new SqlParameterValue(BLOB_TYPE, new SqlBinaryValue(descriptionBytes)),
      TEMPLATE_METADATA.LOGO_DATA -> new SqlParameterValue(BLOB_TYPE, new SqlBinaryValue(logoBytes))
    )
    try {
      namedTemplate.update(STMT_INSERT_TEMPLATE_META, p)
    } catch {
      case _: DuplicateKeyException =>
        throw new ItemAlreadyExistsException(s"Template metadata already exists for ${templateMeta.releaseUid}")
    }

    if (templateLogo != null) {
      val artifactId = templateLogo.getId
      val artifactFile = templateLogo.getFile
      // throw an exception here if file is not found
      Option(artifactFile) match {
        case Some(artifactFile) =>
          val artifactName = artifactFile.getName
          val content = artifactFile.getInputStream
          self.insertArtifact(templateMeta.releaseUid, artifactId, artifactName, content)
        case None => throw new NotFoundException(s"Artifact $artifactId file not found")
      }

    }
  }

  private val STMT_UPDATE_TEMPLATE_META =
    s"""
       | UPDATE ${TEMPLATE_METADATA.TABLE}
       | SET
       |  ${TEMPLATE_METADATA.AUTHOR} = :${TEMPLATE_METADATA.AUTHOR},
       |  ${TEMPLATE_METADATA.TARGET_FOLDER_CI_UID} = :${TEMPLATE_METADATA.TARGET_FOLDER_CI_UID},
       |  ${TEMPLATE_METADATA.ALLOW_TARGET_FOLDER_OVERRIDE} = :${TEMPLATE_METADATA.ALLOW_TARGET_FOLDER_OVERRIDE},
       |  ${TEMPLATE_METADATA.DESCRIPTION} = :${TEMPLATE_METADATA.DESCRIPTION}
       | WHERE ${TEMPLATE_METADATA.RELEASE_UID} = :${TEMPLATE_METADATA.RELEASE_UID}
       |""".stripMargin

  def updateTemplateMetadata(releaseUid: CiUid, templateMeta: TemplateMetadata): Boolean = {
    val descriptionBytes: Array[Byte] = getDescriptionBytes(templateMeta.description)
    val targetFolderCiUid: CiUid = getTargetFolderCiUid(templateMeta.targetFolderId)
    val p = Map[String, Any](
      TEMPLATE_METADATA.RELEASE_UID -> releaseUid,
      TEMPLATE_METADATA.AUTHOR -> templateMeta.author,
      TEMPLATE_METADATA.TARGET_FOLDER_CI_UID -> targetFolderCiUid,
      TEMPLATE_METADATA.ALLOW_TARGET_FOLDER_OVERRIDE -> templateMeta.allowTargetFolderOverride.asInteger,
      TEMPLATE_METADATA.DESCRIPTION -> new SqlParameterValue(BLOB_TYPE, new SqlBinaryValue(descriptionBytes))
    )
    val res = namedTemplate.update(STMT_UPDATE_TEMPLATE_META, p)
    res == 1
  }

  private val STMT_UPDATE_LOGO =
    s""" UPDATE ${TEMPLATE_METADATA.TABLE}
       | SET
       | ${TEMPLATE_METADATA.LOGO_DATA} = :${TEMPLATE_METADATA.LOGO_DATA}
       | WHERE ${TEMPLATE_METADATA.RELEASE_UID} = :${TEMPLATE_METADATA.RELEASE_UID}
       |""".stripMargin

  def updateLogo(releaseUid: CiUid, originalLogo: Option[TemplateLogo], updatedLogo: TemplateLogo): Boolean = {
    // delete the existing logo
    originalLogo.foreach { templateLogo =>
      self.deleteArtifactById(templateLogo.getId, releaseUid)
    }

    val logo = for {
      newLogo <- Option(updatedLogo)
      logoFile <- Option(newLogo.getFile)
      _ = self.insertArtifact(releaseUid, newLogo.getId, logoFile.getName, logoFile.getInputStream)
    } yield newLogo

    // update template metadata
    val logoBytes: Array[Byte] = serializeTemplateLogo(logo.orNull)
    val p = Map[String, Any](
      TEMPLATE_METADATA.RELEASE_UID -> releaseUid,
      TEMPLATE_METADATA.LOGO_DATA -> new SqlParameterValue(BLOB_TYPE, new SqlBinaryValue(logoBytes))
    )
    val res = namedTemplate.update(STMT_UPDATE_LOGO, p)
    res == 1
  }

  private val STMT_DELETE_TEMPLATE_META =
    s"DELETE FROM ${TEMPLATE_METADATA.TABLE} WHERE ${TEMPLATE_METADATA.RELEASE_UID} = :${TEMPLATE_METADATA.RELEASE_UID}"

  def deleteTemplateMetadata(releaseUid: CiUid): Boolean = {
    val res = namedTemplate.update(STMT_DELETE_TEMPLATE_META, params(TEMPLATE_METADATA.RELEASE_UID -> releaseUid))
    res == 1
  }

  private val STMT_FIND_BY_RELEASE_UID =
    s"""
       | SELECT
       | t.${TEMPLATE_METADATA.RELEASE_UID}
       | , t.${TEMPLATE_METADATA.AUTHOR}
       | , t.${TEMPLATE_METADATA.TARGET_FOLDER_CI_UID}
       | , t.${TEMPLATE_METADATA.ALLOW_TARGET_FOLDER_OVERRIDE}
       | , t.${TEMPLATE_METADATA.DESCRIPTION}
       | , t.${TEMPLATE_METADATA.LOGO_DATA}
       | , f.${FOLDERS.FOLDER_ID}
       | , f.${FOLDERS.FOLDER_PATH}
       | FROM ${TEMPLATE_METADATA.TABLE} t
       | LEFT JOIN ${FOLDERS.TABLE} f ON f.${FOLDERS.CI_UID} = t.${TEMPLATE_METADATA.TARGET_FOLDER_CI_UID}
       | WHERE t.${TEMPLATE_METADATA.RELEASE_UID} = :${TEMPLATE_METADATA.RELEASE_UID}
       |""".stripMargin

  @IsReadOnly
  def findTemplateMetadata(releaseUid: CiUid): Option[TemplateMetadata] = {
    try {
      Option(namedTemplate.queryForObject(STMT_FIND_BY_RELEASE_UID,
        params(TEMPLATE_METADATA.RELEASE_UID -> releaseUid),
        toTemplateMeta
      ))
    } catch {
      case _: EmptyResultDataAccessException =>
        None
    }
  }

  @IsReadOnly
  def findTemplateLogoById(logoId: CiId): Option[TemplateLogo] = {
    loadSourceArtifact(logoId.normalized)(classOf[TemplateLogo])
  }

  private val STMT_FIND_TEMPLATE_TILES_BY_DEFAULT_TARGET_FOLDER_ID =
    s"""
       | SELECT
       |  r.${RELEASES.RELEASE_ID},
       |  r.${RELEASES.RELEASE_TITLE}
       | FROM ${FOLDERS.TABLE} af
       |  JOIN ${PATHS.TABLE} p ON p.${PATHS.ANCESTOR_UID} = af.${FOLDERS.CI_UID}
       |  JOIN ${FOLDERS.TABLE} f ON f.${FOLDERS.CI_UID} = p.${PATHS.DESCENDANT_UID}
       |  JOIN ${TEMPLATE_METADATA.TABLE} t ON t.${TEMPLATE_METADATA.TARGET_FOLDER_CI_UID} = f.${FOLDERS.CI_UID}
       |  JOIN ${RELEASES.TABLE} r ON r.${RELEASES.CI_UID} = t.${TEMPLATE_METADATA.RELEASE_UID}
       | WHERE af.${FOLDERS.FOLDER_ID} = :${FOLDERS.FOLDER_ID}
       |""".stripMargin

  @IsReadOnly
  def getTemplatesWithDefaultTargetFolder(folderId: String): List[ReleaseBasicData] = {
    findMany(
      sqlQuery(STMT_FIND_TEMPLATE_TILES_BY_DEFAULT_TARGET_FOLDER_ID,
        Utils.params(FOLDERS.FOLDER_ID -> Ids.getName(folderId)),
        rs => ReleaseBasicData(rs.getString(RELEASES.RELEASE_ID), rs.getString(RELEASES.RELEASE_TITLE))
      )
    ).toList
  }

  private def serializeTemplateLogo(templateLogo: TemplateLogo): Array[Byte] = {
    if (templateLogo != null) {
      val copy = CiCloneHelper.cloneCi(templateLogo)
      // don't store folder info in the JSON so that it does not have to be updated on move operations
      rewriteWithNewId(copy, getFolderlessId(copy.getId))
      val logoJson = CiSerializerHelper.serialize(copy)
      logoJson.getBytes(StandardCharsets.UTF_8)
    } else {
      Array.emptyByteArray
    }
  }

  private def getDescriptionBytes(description: String) = {
    val descriptionBytes: Array[Byte] = if (description != null) {
      description.getBytes(StandardCharsets.UTF_8)
    } else {
      Array.emptyByteArray
    }
    descriptionBytes
  }

  private def getTargetFolderCiUid(targetFolderId: String): CiUid = {
    val targetFolderCiUid: CiUid = Option(targetFolderId)
      .map(folderPersistence.getUid)
      .orNull
    targetFolderCiUid
  }

  private val toTemplateMeta: RowMapper[TemplateMetadata] = {
    case (rs, _) =>
      val releaseUid = rs.getString(TEMPLATE_METADATA.RELEASE_UID)
      val author = rs.getString(TEMPLATE_METADATA.AUTHOR)
      val rawFolderId = rs.getString(FOLDERS.FOLDER_ID)
      val rawFolderPath = rs.getString(FOLDERS.FOLDER_PATH)
      val targetFolderId = if (rawFolderId != null) {
        FolderId(s"$rawFolderPath${Ids.SEPARATOR}$rawFolderId".dropWhile(_ == '/')).absolute
      } else {
        null
      }
      val allowTargetFolderOverride = rs.getInt(TEMPLATE_METADATA.ALLOW_TARGET_FOLDER_OVERRIDE).asBoolean
      val descriptionStream = rs.getBinaryStream(TEMPLATE_METADATA.DESCRIPTION)
      val description = if (descriptionStream == null) {
        null
      } else {
        Using.resource(descriptionStream) { is =>
          val logoBytes = StreamUtils.copyToByteArray(is)
          new String(logoBytes, StandardCharsets.UTF_8)
        }
      }

      val logoStream = rs.getBinaryStream(TEMPLATE_METADATA.LOGO_DATA)
      val templateLogo = if (logoStream == null) {
        null
      } else {
        Using.resource(logoStream) { is =>
          val logoBytes = StreamUtils.copyToByteArray(is)
          if (logoBytes.isEmpty) {
            null
          } else {
            val logoJson = new String(logoBytes, StandardCharsets.UTF_8)
            val deserializedLogo = CiSerializerHelper.deserialize(logoJson, null).asInstanceOf[TemplateLogo]
            val templateLogoWithLoadedArtifact = self.loadSourceArtifact(deserializedLogo.getId)(classOf[TemplateLogo])
            templateLogoWithLoadedArtifact match {
              case Some(v) =>
                v.setContentType(deserializedLogo.getContentType)
                v.setExportFilename(deserializedLogo.getExportFilename)
                v.setPortableFilename(deserializedLogo.getPortableFilename)
                v
              case None => null
            }
          }

        }
      }

      TemplateMetadata(releaseUid, author, description, targetFolderId, allowTargetFolderOverride, templateLogo)
  }

}
