package com.xebialabs.xlrelease.repository.sql.persistence

import com.xebialabs.deployit.checks.Checks.IncorrectArgumentException
import com.xebialabs.deployit.core.xml.PasswordEncryptingCiConverter
import com.xebialabs.deployit.exception.NotFoundException
import com.xebialabs.deployit.io.SourceArtifactFile
import com.xebialabs.deployit.plugin.api.reflect.Type
import com.xebialabs.deployit.plugin.api.udm.artifact.SourceArtifact
import com.xebialabs.deployit.security.PermissionEnforcer
import com.xebialabs.xlplatform.utils.ResourceManagement.using
import com.xebialabs.xlrelease.api.v1.forms.{ReleaseOrderMode, ReleasesFilters}
import com.xebialabs.xlrelease.db.sql.SqlBuilder.Dialect
import com.xebialabs.xlrelease.db.sql.transaction.IsTransactional
import com.xebialabs.xlrelease.db.sql.{DatabaseInfo, LimitOffset, Sql, SqlWithParameters}
import com.xebialabs.xlrelease.domain.id.{CiUid, IdCreator}
import com.xebialabs.xlrelease.domain.status.{PhaseStatus, ReleaseStatus, TaskStatus}
import com.xebialabs.xlrelease.domain.utils.syntax.ConfigurationItemOps
import com.xebialabs.xlrelease.domain.utils.{AdaptiveReleaseId, FullReleaseId, IdUtils}
import com.xebialabs.xlrelease.domain.{Attachment, Release, ReleaseKind, TemplateLogo}
import com.xebialabs.xlrelease.repository.Ids._
import com.xebialabs.xlrelease.repository._
import com.xebialabs.xlrelease.repository.sql.SqlRepository
import com.xebialabs.xlrelease.repository.sql.persistence.CiId._
import com.xebialabs.xlrelease.repository.sql.persistence.ReleasePersistence._
import com.xebialabs.xlrelease.repository.sql.persistence.ReleasesSqlBuilder.{CURRENT_PHASE_TITLE, alias, folderAlias, phaseAlias}
import com.xebialabs.xlrelease.repository.sql.persistence.Schema._
import com.xebialabs.xlrelease.repository.sql.persistence.Utils._
import com.xebialabs.xlrelease.repository.sql.persistence.configuration.ConfigurationPersistence
import com.xebialabs.xlrelease.repository.sql.persistence.data.FolderRow.Root
import com.xebialabs.xlrelease.repository.sql.persistence.data._
import com.xebialabs.xlrelease.security.SecuredCi
import com.xebialabs.xlrelease.serialization.json.utils.CiSerializerHelper.serialize
import com.xebialabs.xlrelease.service.{CiIdService, SqlReleasesFilterSupport, SqlTemplatesFilterSupport, WorkdirSupport}
import com.xebialabs.xlrelease.utils.CiHelper.{fixUpInternalReferences, rewriteWithNewId}
import com.xebialabs.xlrelease.utils.FolderId
import com.xebialabs.xlrelease.variable.VariablePersistenceHelper.fixUpVariableIds
import com.xebialabs.xlrelease.views.TemplateFilters
import grizzled.slf4j.Logging
import org.slf4j.{Logger, LoggerFactory}
import org.springframework.jdbc.core._
import org.springframework.jdbc.support.rowset.SqlRowSet
import org.springframework.transaction.annotation.Isolation.READ_COMMITTED
import org.springframework.transaction.annotation.Propagation.REQUIRED
import org.springframework.transaction.annotation.Transactional

import java.io.InputStream
import java.net.URLEncoder
import java.nio.charset.StandardCharsets.UTF_8
import java.sql.{PreparedStatement, ResultSet, Types}
import java.util.Collections.emptyList
import java.util.{Collections, Date}
import scala.collection.mutable.{ArrayBuffer, ListBuffer}
import scala.jdk.CollectionConverters._
import scala.util.Try


//scalastyle:off number.of.methods
@IsTransactional
class ReleasePersistence(val folderPersistence: FolderPersistence,
                         ciIdService: CiIdService,
                         releaseCacheService: ReleaseCacheService,
                         dbInfo: DatabaseInfo,
                         implicit val jdbcTemplate: JdbcTemplate,
                         implicit val dialect: Dialect)
  extends SqlRepository with PersistenceSupport with Logging with Utils with LimitOffset with TemplateMetadataPersistence with WorkdirSupport {

  private val FIND_UID_BY_RELEASE_ID = s"""SELECT ${RELEASES.CI_UID} FROM ${RELEASES.TABLE} WHERE ${RELEASES.RELEASE_ID} = ?"""
  private val FIND_UID_BY_RELEASE_ID_BATCH = s"""SELECT ${RELEASES.RELEASE_ID}, ${RELEASES.CI_UID} FROM ${RELEASES.TABLE} WHERE ${RELEASES.RELEASE_ID} IN (:releaseIds)"""
  private val FIND_UID_BY_RELEASE_ID_BATCH_SIZE = 512

  def findUidByReleaseId(releaseId: CiId): Option[CiUid] = {
    findOptional(_.queryForObject(
      FIND_UID_BY_RELEASE_ID,
      classOf[CiUid],
      getName(releaseId.normalized)
    ))
  }

  def insert(release: Release): CiUid = {
    logger.debug(s"Inserting release ${release.getId} into database")
    validateTaskSchedulingPropertiesForWorkflow(release)
    fixUpInternalReferences(release)
    fixUpVariableIds(release.getId, release.getVariables, ciIdService)
    val parentFolder = Try(getParentId(release.getId)).toOption
      .map(folderPersistence.findById(_, depth = 0).value)
      .getOrElse(Root)
    release.set$securedCi(parentFolder.securityUid)
    release.createCiAttributes()
    val ciUid = IdCreator.generateId
    insertRelease(ciUid, release, parentFolder)
    insertTags(ciUid)(distinctTags(release.getTags))
    insertArtifacts(ciUid, release.getAttachments.asScala.toSeq)
    insertReleaseJson(ciUid, release)
    insertTemplateMetadata(ciUid, release)
    ciUid
  }

  private val STMT_FIND_BY_RELEASE_ID =
    s"""|SELECT
        |  rel.${RELEASES.CI_UID},
        |  rel.${RELEASES.RELEASE_ID},
        |  rel.${RELEASES.SECURITY_UID},
        |  f.${FOLDERS.FOLDER_PATH},
        |  f.${FOLDERS.FOLDER_ID},
        |  rel.${RELEASES.RISK_SCORE},
        |  rel.${RELEASES.TOTAL_RISK_SCORE},
        |  rel.${RELEASES.SCM_DATA},
        |  rel.${RELEASES.PLANNED_DURATION},
        |  rel.${RELEASES.STATUS},
        |  rel.${RELEASES.REAL_FLAG_STATUS},
        |  rel.${RELEASES.RELEASE_TITLE},
        |  rel.${RELEASES.START_DATE},
        |  rel.${RELEASES.END_DATE},
        |  rel.${RELEASES.IS_OVERDUE_NOTIFIED},
        |  rel.${RELEASES.KIND}
        | FROM ${RELEASES.TABLE} rel
        | INNER JOIN ${FOLDERS.TABLE} f ON f.${FOLDERS.CI_UID} = rel.${RELEASES.FOLDER_CI_UID}
        | WHERE rel.${RELEASES.RELEASE_ID} = :releaseId""".stripMargin

  def findByReleaseId(releaseId: CiId): Option[ReleaseRow] = {
    logger.debug(s"Selecting release $releaseId from database")
    releaseCacheService.get(releaseId).orElse {
      findOne(sqlQuery(STMT_FIND_BY_RELEASE_ID, params("releaseId" -> getName(releaseId)), releaseRowMapper))
        .map(populateWithReleaseJson)
    }
  }

  private val SELECT_BASIC_ROW =
    s"""
       |SELECT
       |  rel.${RELEASES.CI_UID},
       |  rel.${RELEASES.RELEASE_ID},
       |  rel.${RELEASES.SECURITY_UID},
       |  f.${FOLDERS.FOLDER_PATH},
       |  f.${FOLDERS.FOLDER_ID},
       |  rel.${RELEASES.RISK_SCORE},
       |  rel.${RELEASES.TOTAL_RISK_SCORE},
       |  rel.${RELEASES.SCM_DATA},
       |  rel.${RELEASES.PLANNED_DURATION},
       |  rel.${RELEASES.STATUS},
       |  rel.${RELEASES.REAL_FLAG_STATUS},
       |  rel.${RELEASES.RELEASE_TITLE},
       |  rel.${RELEASES.START_DATE},
       |  rel.${RELEASES.END_DATE},
       |  rel.${RELEASES.IS_OVERDUE_NOTIFIED},
       |  rel.${RELEASES.KIND},
       |  rel.${RELEASES.RELEASE_OWNER}
       | FROM ${RELEASES.TABLE} rel
       | INNER JOIN ${FOLDERS.TABLE} f ON f.${FOLDERS.CI_UID} = rel.${RELEASES.FOLDER_CI_UID}
       |""".stripMargin

  private val STMT_GET_BASIC_ROW_BY_RELEASE_ID =
    s"""| $SELECT_BASIC_ROW
        | WHERE rel.${RELEASES.RELEASE_ID} = :${RELEASES.RELEASE_ID}""".stripMargin

  private val STMT_GET_BASIC_ROW_BY_PARENT_ID =
    s"""| $SELECT_BASIC_ROW
        | WHERE rel.${RELEASES.PARENT_RELEASE_ID} = :${RELEASES.PARENT_RELEASE_ID}
        | AND rel.${RELEASES.PRE_ARCHIVED} = 0
        | """.stripMargin

  def getBasicReleaseRow(releaseId: CiId): Option[BasicReleaseRow] = {
    logger.debug(s"Selecting release $releaseId from database")
    findOne(sqlQuery(STMT_GET_BASIC_ROW_BY_RELEASE_ID, params(RELEASES.RELEASE_ID -> getName(releaseId)), basicReleaseRowMapper))
  }

  def findBasicReleaseRowByQuery(parentReleaseId: CiId): Seq[BasicReleaseRow] = {
    logger.debug(s"Finding releases with parentReleaseId $parentReleaseId from database")
    findMany(sqlQuery(STMT_GET_BASIC_ROW_BY_PARENT_ID, params(RELEASES.PARENT_RELEASE_ID -> getName(parentReleaseId)), basicReleaseRowMapper))
  }

  def getReleaseJson(releaseId: CiId): Option[String] = {
    releaseCacheService.get(releaseId).map(_.json).orElse {
      findUidByReleaseId(releaseId).flatMap(releaseUid => fetchReleaseJsonByUid(releaseUid))
    }
  }

  private val STMT_FIND_FOLDER_ID_BY_RELEASE_ID =
    s"""|SELECT
        |  f.${FOLDERS.FOLDER_PATH},
        |  f.${FOLDERS.FOLDER_ID}
        | FROM ${RELEASES.TABLE} rel
        | INNER JOIN ${FOLDERS.TABLE} f ON f.${FOLDERS.CI_UID} = rel.${RELEASES.FOLDER_CI_UID}
        | WHERE rel.${RELEASES.RELEASE_ID} = :releaseId""".stripMargin

  def findFolderIdByReleaseId(releaseId: CiId): Option[String] = {
    logger.debug(s"Obtaining folder id for release $releaseId from database")
    findOne(sqlQuery(STMT_FIND_FOLDER_ID_BY_RELEASE_ID, params("releaseId" -> getName(releaseId)), rs => {
      (FolderId(rs.getString(FOLDERS.FOLDER_PATH)) / rs.getString(FOLDERS.FOLDER_ID)).absolute
    }))
  }

  private val STMT_FIND_BY_RELEASE_UID =
    s"""|SELECT
        |  rel.${RELEASES.CI_UID},
        |  rel.${RELEASES.RELEASE_ID},
        |  rel.${RELEASES.SECURITY_UID},
        |  f.${FOLDERS.FOLDER_PATH},
        |  f.${FOLDERS.FOLDER_ID},
        |  rel.${RELEASES.RISK_SCORE},
        |  rel.${RELEASES.TOTAL_RISK_SCORE},
        |  rel.${RELEASES.SCM_DATA},
        |  rel.${RELEASES.PLANNED_DURATION},
        |  rel.${RELEASES.STATUS},
        |  rel.${RELEASES.REAL_FLAG_STATUS},
        |  rel.${RELEASES.RELEASE_TITLE},
        |  rel.${RELEASES.START_DATE},
        |  rel.${RELEASES.END_DATE},
        |  rel.${RELEASES.IS_OVERDUE_NOTIFIED},
        |  rel.${RELEASES.KIND}
        | FROM ${RELEASES.TABLE} rel
        | INNER JOIN ${FOLDERS.TABLE} f ON f.${FOLDERS.CI_UID} = rel.${RELEASES.FOLDER_CI_UID}
        | WHERE rel.${RELEASES.CI_UID} = :releaseUid""".stripMargin

  def findByReleaseUid(releaseUid: Int): Option[ReleaseRow] = {
    logger.debug(s"Selecting release $releaseUid from database")
    findOne(sqlQuery(STMT_FIND_BY_RELEASE_UID, params("releaseUid" -> releaseUid), releaseRowMapper))
      .map(populateWithReleaseJson)
  }

  private val STMT_FIND_SCM_DATA_BY_RELEASE_ID =
    s"""
       | SELECT rel.${RELEASES.SCM_DATA}
       | FROM ${RELEASES.TABLE} rel
       | WHERE rel.${RELEASES.RELEASE_ID} = :releaseId
     """.stripMargin

  def findSCMDataById(releaseId: CiId): Option[Int] = {
    logger.debug(s"Obtaining scm data id from release $releaseId")
    findOne(sqlQuery(STMT_FIND_SCM_DATA_BY_RELEASE_ID, params("releaseId" -> getName(releaseId)), result => {
      result.getInt(RELEASES.SCM_DATA)
    }))
  }

  def deleteById(releaseId: CiId): Unit = {
    logger.debug(s"Deleting release $releaseId from database")
    findUidByReleaseId(releaseId) match {
      case Some(ciUid) =>
        if (deleteReleaseRow(ciUid)) {
          releaseCacheService.invalidate(releaseId)
        } else {
          throw new IllegalStateException(s"Could not delete release by ID $releaseId")
        }
      case None =>
        throw new NotFoundException(s"Release $releaseId not found")
    }
  }

  private val STMT_DELETE_RELEASE =
    s"""DELETE FROM ${RELEASES.TABLE} WHERE ${RELEASES.CI_UID} = :ciUid""".stripMargin

  private def deleteReleaseRow(ciUid: CiUid): Boolean = {
    sqlUpdate(STMT_DELETE_RELEASE, params("ciUid" -> ciUid), {
      case 0 => false
      case _ => true
    })
  }

  def update(release: Release): Unit = {
    update(None, release)
  }

  def update(original: Option[Release], updated: Release): Unit = {
    fixUpInternalReferences(updated)
    logger.debug(s"Updating release ${updated.getId} on database")
    validateTaskSchedulingPropertiesForWorkflow(updated)
    findUidByReleaseId(updated.getId) match {
      case Some(ciUid) =>
        updateRelease(ciUid, updated)
        updated.setCiUid(ciUid)
        updated.updateCiAttributes()
        updateReleaseJson(ciUid, updated)
        updateTags(ciUid, original, updated)
        updateTemplateMetadata(ciUid, original, updated)
      case None =>
        throw new ReleaseStoreException(s"""Release ${updated.getId} not found""")
    }
  }

  private val STMT_UPDATE_RELEASE =
    s"""|UPDATE ${RELEASES.TABLE}
        | SET
        |   ${RELEASES.STATUS} = :status,
        |   ${RELEASES.RELEASE_TITLE} = :title,
        |   ${RELEASES.START_DATE} = :startDate,
        |   ${RELEASES.END_DATE} = :endDate,
        |   ${RELEASES.RISK_SCORE} = :riskScore,
        |   ${RELEASES.TOTAL_RISK_SCORE} = :totalRiskScore,
        |   ${RELEASES.RELEASE_OWNER} = :owner,
        |   ${RELEASES.CALENDAR_TOKEN} = :calendarToken,
        |   ${RELEASES.AUTO_START} = :autoStart,
        |   ${RELEASES.REAL_FLAG_STATUS} = :realFlagStatus,
        |   ${RELEASES.SCM_DATA} = :scmData,
        |   ${RELEASES.PLANNED_DURATION} = :plannedDuration,
        |   ${RELEASES.IS_OVERDUE_NOTIFIED} = :overdueNotified,
        |   ${RELEASES.ORIGIN_TEMPLATE_ID} = :originTemplateId,
        |   ${RELEASES.KIND} = :${RELEASES.KIND}
        | WHERE ${RELEASES.CI_UID} = :ciUid""".stripMargin


  private def updateRelease(ciUid: CiUid, updated: Release): Unit = {

    sqlUpdate(STMT_UPDATE_RELEASE, params(
      "status" -> updated.getStatus.value,
      "title" -> updated.getTitle,
      "startDate" -> updated.getStartOrScheduledDate,
      "endDate" -> updated.getEndOrDueDate,
      "riskScore" -> Integer.valueOf(updated.getPropertyOption("riskScore").getOrElse("0")),
      "totalRiskScore" -> Integer.valueOf(updated.getPropertyOption("totalRiskScore").getOrElse("0")),
      "owner" -> updated.getOwner,
      "calendarToken" -> updated.getCalendarLinkToken,
      "autoStart" -> updated.isAutoStart.asInteger,
      "realFlagStatus" -> Integer.valueOf(updated.getRealFlagStatus.getRisk),
      "scmData" -> getSCMData(updated.get$ciAttributes().getScmTraceabilityDataId),
      "plannedDuration" -> updated.getPlannedDuration,
      "overdueNotified" -> updated.isOverdueNotified.asInteger,
      "originTemplateId" -> getName(updated.getOriginTemplateId),
      RELEASES.KIND -> updated.getKind.value(),
      "ciUid" -> ciUid
    ), _ => ())
  }

  private def getSCMData(scmData: CiUid): CiUid = Option(scmData).filter(_ != "0").orNull

  private def deleteAndCreateSetDiff(ciUid: CiUid,
                                     originalSet: Set[String],
                                     updatedSet: Set[String],
                                     delete: Set[String] => Unit,
                                     insert: Set[String] => Unit): Unit = {

    if (updatedSet != originalSet) {
      val deletedOnes = originalSet.diff(updatedSet)
      val createdOnes = updatedSet.diff(originalSet)
      delete(deletedOnes)
      insert(createdOnes)
    }
  }

  private def updateTags(ciUid: CiUid, original: Option[Release], updated: Release): Unit = {
    val originalTags: Set[String] = original match {
      case Some(originalRelease) => distinctTags(originalRelease.getTags)
      case None => selectTags(ciUid)
    }
    val updatedTags = distinctTags(updated.getTags)
    deleteAndCreateSetDiff(ciUid,
      originalTags,
      updatedTags, deleteTags(ciUid), insertTags(ciUid)
    )
  }

  private def insertTemplateMetadata(ciUid: CiUid, release: Release): Unit = {
    updateTemplateMetadata(ciUid, None, release)
  }

  // scalastyle:off cyclomatic.complexity
  private def updateTemplateMetadata(ciUid: CiUid, original: Option[Release], updated: Release): Unit = {
    // logo will be ignored and it will be updated separately
    if (updated.isTemplate) {
      val updatedAuthor = updated.getAuthor
      val updatedDescription = updated.getDescription
      val updatedTargetFolderId = updated.getDefaultTargetFolderId
      val updatedTargetFolderOverride = updated.getAllowTargetFolderOverride

      original match {
        case Some(_) =>
          val updatedTemplateMetadata = TemplateMetadata(ciUid, updatedAuthor, updatedDescription, updatedTargetFolderId, updatedTargetFolderOverride, null)
          updateTemplateMetadata(ciUid, updatedTemplateMetadata)
        case None =>

          val originalTemplateMetadata = findTemplateMetadata(ciUid)
          originalTemplateMetadata match {
            case Some(_) =>
              val updatedTemplateMetadata = TemplateMetadata(ciUid, updatedAuthor, updatedDescription, updatedTargetFolderId, updatedTargetFolderOverride, null)
              updateTemplateMetadata(ciUid, updatedTemplateMetadata)
            case None =>
              val updatedTemplateLogo = updated.getLogo
              // fix id for template logo
              fixLogoId(updated.getId, updatedTemplateLogo)
              val templateMetadata = TemplateMetadata(ciUid,
                updatedAuthor,
                updatedDescription,
                updatedTargetFolderId,
                updatedTargetFolderOverride,
                updatedTemplateLogo)
              createTemplateMetadata(templateMetadata)
          }

      }
    }
  }
  // scalastyle:on cyclomatic.complexity

  def fixLogoId(parentId: String, templateLogo: TemplateLogo): Unit = {
    val folderlessParentId = Ids.getFolderlessId(parentId)
    if (templateLogo != null) {
      val logoId = if (templateLogo.getId == null) {
        IdUtils.getUniqueId(Type.valueOf(classOf[TemplateLogo]), folderlessParentId)
      } else {
        s"$folderlessParentId/${getName(templateLogo.getId)}"
      }
      templateLogo.setId(logoId)
    }
  }

  private val STMT_UPDATE_FOLDER_CI_UID =
    s"""|UPDATE ${RELEASES.TABLE}
        | SET
        |   ${RELEASES.FOLDER_CI_UID} = :newFolderCiUid,
        |   ${RELEASES.SECURITY_UID} = :newSecurityUid
        | WHERE ${RELEASES.RELEASE_ID} = :releaseId""".stripMargin

  def move(originalReleaseId: CiId, newReleaseId: CiId): Unit = {
    val newFolder = folderPersistence.findById(getParentId(newReleaseId), 0).value

    sqlUpdate(STMT_UPDATE_FOLDER_CI_UID, params(
      "releaseId" -> getName(originalReleaseId),
      "newFolderCiUid" -> newFolder.uid,
      "newSecurityUid" -> newFolder.securityUid
    ), _ => ())
  }

  // visible for upgraders
  def updateReleaseJson(ciUid: CiUid, updated: Release): Unit = {
    val json = serializeRelease(updated)
    updateSerializedReleaseJson(ciUid, updated.getId, json)
    releaseCacheService.put(updated.getId, releaseRow(updated, json, updated.get$securedCi()))
  }

  private def releaseRow(release: Release, json: String, securityUid: CiUid): ReleaseRow = {
    val bareReleaseId = com.xebialabs.xlrelease.repository.Ids.getFolderlessId(release.getId)
    val folderId = if (isRoot(release.getId)) {
      Ids.ROOT_FOLDER_ID
    } else {
      com.xebialabs.xlrelease.repository.Ids.getParentId(release.getId)
    }
    ReleaseRow(
      json = json,
      releaseId = bareReleaseId,
      ciUid = release.getCiUid,
      folderId = folderId,
      securityUid = securityUid,
      status = release.getStatus.value(),
      realFlagStatus = release.getRealFlagStatus().getRisk,
      title = release.getTitle,
      startDate = release.getStartDate,
      endDate = release.getEndDate,
      riskScore = Integer.valueOf(release.getPropertyOption("riskScore").getOrElse("0")),
      totalRiskScore = Integer.valueOf(release.getPropertyOption("totalRiskScore").getOrElse("0")),
      scmDataUid = release.get$ciAttributes().getScmTraceabilityDataId,
      plannedDuration = release.getPlannedDuration,
      isOverdueNotified = release.isOverdueNotified,
      kind = release.getKind.value()
    )
  }

  // visible for testing
  def updateSerializedReleaseJson(ciUid: CiUid, releaseId: CiId, json: String): Unit = {
    sqlExecWithContent(STMT_UPDATE_RELEASE_CONTENT, params("ciUid" -> ciUid), "content" -> json, _ => checkCiUpdated(releaseId))
  }

  private def populateWithReleaseJson(row: ReleaseRow): ReleaseRow = {
    val releaseUid = row.ciUid
    val json = fetchReleaseJsonByUid(releaseUid)
    row.copy(json = json.get)
  }

  private def fetchReleaseJsonByUid(releaseUid: CiUid): Option[String] = {
    // we know insert happens ONLY via actor - fetch might be safely cacheable in cluster env
    findOne(sqlQuery(STMT_READ_RELEASE_CONTENT, params(RELEASES_DATA.CI_UID -> releaseUid), releaseBinaryStreamRowMapper))
  }

  @Transactional(value = "xlrRepositoryTransactionManager",
    propagation = REQUIRED,
    isolation = READ_COMMITTED,
    noRollbackFor = Array(classOf[RuntimeException]))
  def insertAttachments(releaseId: CiId, attachments: Seq[SourceArtifact]): Boolean = {
    if (attachments.nonEmpty) {
      logger.debug(s"Inserting ${attachments.size} attachments for release $releaseId")
      val pk = findUidByReleaseId(releaseId).get
      insertArtifacts(pk, attachments)
    }
    true
  }

  def findAttachmentsByCiUid(ciUid: CiUid): Seq[Attachment] = {
    loadSourceArtifacts(ciUid)(classOf[Attachment])
  }

  def findAttachmentById(attachmentId: CiId): Option[Attachment] = {
    loadSourceArtifact(attachmentId.normalized)(classOf[Attachment])
  }

  def deleteAttachmentById(attachmentId: CiId, releaseCiUid: CiUid): Unit = {
    deleteArtifactById(attachmentId.normalized, releaseCiUid)
  }

  def insertArtifact(pk: CiUid, artifactId: String, artifactName: String, content: InputStream): Boolean = {
    val res = jdbcTemplate.update(
      s"""INSERT INTO ${ARTIFACTS.TABLE}
         |(${ARTIFACTS.ARTIFACTS_ID}, ${ARTIFACTS.CI_UID}, ${ARTIFACTS.NAME}, ${ARTIFACTS.CONTENT})
         | VALUES (?,?,?,?)""".stripMargin,
      new PreparedStatementSetter with ParameterDisposer {
        override def setValues(ps: PreparedStatement): Unit = {
          ps.setString(1, getReleaselessChildId(artifactId))
          ps.setString(2, pk)
          ps.setString(3, artifactName.truncate(COLUMN_LENGTH_TITLE))
          ps.setBinaryStream(4, content)
        }

        override def cleanupParameters(): Unit = content.close()
      }
    )
    res == 1
  }

  private def insertArtifacts(pk: CiUid, artifacts: Seq[SourceArtifact]): Unit = {
    if (artifacts.nonEmpty) {
      jdbcTemplate.batchUpdate(
        s"""INSERT INTO ${ARTIFACTS.TABLE}
           |(${ARTIFACTS.ARTIFACTS_ID}, ${ARTIFACTS.CI_UID}, ${ARTIFACTS.NAME}, ${ARTIFACTS.CONTENT})
           | VALUES (?,?,?,?)""".stripMargin,
        new BatchPreparedStatementSetter with ParameterDisposer {
          private val streamsToClose = ListBuffer.empty[InputStream]

          override def getBatchSize: Int = artifacts.size

          override def setValues(ps: PreparedStatement, i: Int): Unit = {
            val artifact = artifacts(i)
            val artifactId = artifact.getId
            ps.setString(1, getReleaselessChildId(artifactId))
            ps.setString(2, pk)
            ps.setString(3, artifact.getFile.getName.truncate(COLUMN_LENGTH_TITLE))
            val stream = artifact.getFile.getInputStream
            streamsToClose += stream
            ps.setBinaryStream(4, stream)
            artifact.setProperty(SourceArtifact.FILE_URI_PROPERTY_NAME, getFileUri(artifactId))
          }

          override def cleanupParameters(): Unit = {
            streamsToClose.foreach(_.close())
          }
        }
      )
    }
  }

  private def getFileUri(artifactId: String) = {
    s"sql:${URLEncoder.encode(artifactId, UTF_8.name())}"
  }


  private def loadSourceArtifacts[T <: SourceArtifact](ciUid: CiUid)(clazz: Class[T]): Seq[T] = {
    val STMT_RELEASE_ARTIFACTS =
      s"""
         |SELECT ${ARTIFACTS.ARTIFACTS_ID}, ${ARTIFACTS.NAME}
         | FROM ${ARTIFACTS.TABLE}
         | WHERE ${ARTIFACTS.CI_UID} = :${ARTIFACTS.CI_UID}
       """.stripMargin
    logger.trace(s"Eager fetching artifacts of a ci_uid: $ciUid")
    sqlQuery(
      STMT_RELEASE_ARTIFACTS,
      params(ARTIFACTS.CI_UID -> ciUid),
      artifactRowMapper(clazz)
    ).toSeq
  }

  private def artifactRowMapper[T <: SourceArtifact](clazz: Class[T]): RowMapper[T] = {
    val artifactType: Type = Type.valueOf(clazz)
    (rs: ResultSet, _: Int) => {
      val artifactId = rs.getString(ARTIFACTS.ARTIFACTS_ID)
      val artifactName = rs.getString(ARTIFACTS.NAME)
      val artifact: T = artifactType.getDescriptor.newInstance(artifactId)
      artifact.setProperty(SourceArtifact.FILE_URI_PROPERTY_NAME, getFileUri(artifactId))
      // ENG-5971 We should not use WorkDirContext.get() here
      // If we use WorkDirContext.get() then it captures existing workDir A, and it memoizes it, so, when
      // time for cleanup will come it will clean A instead of current workDir at that time. A is
      // then overriden by some other workDir B by calling WorkDirContext.initWorkdir, then lazy
      // file loading happens into A that is captured now, and then B is deleted by those who initialized
      // it by calling WorkDirContext.initWorkdir, because they are good guys and clean after themselves
      // The problem is that in rather rare case A will exist, so at this point WorkDirContext will have some
      // workDir cleaned but left by some previous operation. Usually A does not exist, and as a result
      // SourceArtifactFile.withNullableWorkDir falls back to null which means "use current WorkDirContext workDir"
      // If we use null here, we explicitly use "current WorkDirContext workDir"
      val lazyFile = SourceArtifactFile.withNullableWorkDir(artifactName, artifact, null)
      artifact.setFile(lazyFile)
      artifact
    }
  }

  private[persistence] def loadSourceArtifact[T <: SourceArtifact](artifactId: CiId)(clazz: Class[T]): Option[T] = {
    val artifact: T = Type.valueOf(clazz).getDescriptor.newInstance(artifactId)
    logger.trace(s"Lazy loading artifact $artifactId")
    findOptional(_.queryForObject(
      s"""SELECT
         | ${ARTIFACTS.ARTIFACTS_ID},
         | ${ARTIFACTS.NAME}
         |  FROM ${ARTIFACTS.TABLE}
         |  WHERE ${ARTIFACTS.ARTIFACTS_ID} = ?
         |  AND ${ARTIFACTS.CI_UID} =
         |    (SELECT ${RELEASES.CI_UID}
         |    FROM ${RELEASES.TABLE}
         |    WHERE ${RELEASES.RELEASE_ID} = ?)""".stripMargin,
      (rs: ResultSet, _: Int) => {
        artifact.setId(artifactId)
        artifact.setProperty(SourceArtifact.FILE_URI_PROPERTY_NAME, getFileUri(getReleaselessChildId(artifactId)))
        val fileName = rs.getString(ARTIFACTS.NAME)
        artifact.setFile(SourceArtifactFile.withNullableWorkDir(fileName, artifact, null))
        artifact
      },
      getReleaselessChildId(artifactId),
      getName(releaseIdFrom(artifactId).normalized))
    )
  }

  private[persistence] def deleteArtifactById(artifactId: CiId, releaseCiUid: CiUid): Unit = {
    logger.debug(s"Deleting artifact $artifactId")
    jdbcTemplate.update(
      s"""DELETE FROM ${ARTIFACTS.TABLE}
         | WHERE ${ARTIFACTS.ARTIFACTS_ID} = ?
         | AND ${ARTIFACTS.CI_UID} = ?""".stripMargin,
      getReleaselessChildId(artifactId),
      releaseCiUid)
  }

  def existsRelease(releaseId: CiId): Boolean = {
    logger.debug(s"Searching release $releaseId on database")
    val count = jdbcTemplate.queryForObject(
      s"""SELECT COUNT(*)
         |FROM ${RELEASES.TABLE}
         |WHERE ${RELEASES.RELEASE_ID} = ?""".stripMargin,
      classOf[Integer],
      getName(releaseId.normalized)
    )
    count > 0
  }

  def findReleaseTitle(releaseId: CiId): Option[String] = {
    logger.debug(s"Searching release $releaseId title on database")
    findOptional(_.queryForObject(
      s"""SELECT ${RELEASES.RELEASE_TITLE}
         |FROM ${RELEASES.TABLE}
         |WHERE ${RELEASES.RELEASE_ID} = ?""".stripMargin,
      classOf[String],
      getName(releaseId.normalized))
    )
  }

  def findReleaseOwner(releaseId: CiId): Option[String] = {
    logger.debug(s"Searching release $releaseId owner on database")
    findOptional(_.queryForObject(
      s"""SELECT ${RELEASES.RELEASE_OWNER}
         |FROM ${RELEASES.TABLE}
         |WHERE ${RELEASES.RELEASE_ID} = ?""".stripMargin,
      classOf[String],
      getName(releaseId.normalized))
    )
  }

  def findReleaseKind(releaseId: CiId): Option[String] = {
    logger.debug(s"Searching release $releaseId kind on database")
    findOptional(_.queryForObject(
      s"""SELECT ${RELEASES.KIND}
         |FROM ${RELEASES.TABLE}
         |WHERE ${RELEASES.RELEASE_ID} = ?""".stripMargin,
      classOf[String],
      getName(releaseId.normalized))
    )
  }

  def findReleaseStatus(releaseId: CiId): Option[String] = {
    logger.debug(s"Searching release $releaseId status on database")
    findOptional(_.queryForObject(
      s"""SELECT ${RELEASES.STATUS}
         |FROM ${RELEASES.TABLE}
         |WHERE ${RELEASES.RELEASE_ID} = ?""".stripMargin,
      classOf[String],
      getName(releaseId.normalized))
    )
  }

  def findReleaseStatuses(releaseIds: Seq[CiId]): Seq[(String, String)] = {
    logger.debug(s"Searching release $releaseIds statuses on database")
    val normalizedIds = releaseIds.map(id => getName(id.normalized)).asJava
    val mapper: RowMapper[(String, String)] = (rs, _) => (rs.getString(1), rs.getString(2))

    sqlQuery(
      s"""SELECT
         | ${RELEASES.RELEASE_ID},
         | ${RELEASES.STATUS}
         |FROM ${RELEASES.TABLE}
         |WHERE ${RELEASES.RELEASE_ID} IN (:releaseIds)""".stripMargin,
      params("releaseIds" -> normalizedIds), mapper
    ).toSeq
  }

  def findReleaseRiskScores(releaseIds: Seq[String]): Seq[Int] = {
    logger.debug(s"Searching release $releaseIds risk scores on database")
    val normalizedIds = releaseIds.map(id => getName(id.normalized)).asJava
    sqlQuery(
      s"""SELECT ${RELEASES.RISK_SCORE}
         |FROM ${RELEASES.TABLE}
         |WHERE ${RELEASES.RELEASE_ID} IN (:releaseIds)""".stripMargin,
      params("releaseIds" -> normalizedIds),
      _.getInt(1)
    ).toSeq
  }

  private val STMT_UPDATE_RISK_SCORES: String =
    s"""|UPDATE ${RELEASES.TABLE}
        |SET
        |   ${RELEASES.RISK_SCORE} = :score,
        |   ${RELEASES.TOTAL_RISK_SCORE} = :totalScore
        |WHERE ${RELEASES.RELEASE_ID} = :releaseId
     """.stripMargin

  def updateReleaseRiskScores(releaseId: String, score: Int, totalScore: Int): Int = {
    sqlUpdate(STMT_UPDATE_RISK_SCORES,
      params(
        "releaseId" -> getName(releaseId.normalized),
        "score" -> score,
        "totalScore" -> totalScore
      ),
      identity
    )
  }

  def countReleasesByStatus(sqlWithParameters: SqlWithParameters): Map[ReleaseStatus, Int] = {
    findReleaseColumnsByQuery(sqlWithParameters, rs => {
      ReleaseStatus.valueOf(rs.getString(1).toUpperCase()) -> rs.getInt(2)
    }).toMap
  }

  def dateRange(sqlWithParameters: SqlWithParameters): (Date, Date) = {
    val rowMapper: RowMapper[(Date, Date)] = (rs, _) => {
      rs.getTimestamp(1) -> rs.getTimestamp(2)
    }
    jdbcTemplate.queryForObject(sqlWithParameters._1, sqlWithParameters._2.toArray, rowMapper)
  }

  def findReleaseIdsByQuery(sqlWithParameters: SqlWithParameters): ArrayBuffer[CiId] = {
    findReleaseColumnsByQuery(sqlWithParameters, rs => FolderId(rs.getString(1)).absolute)
  }

  def findShortReleaseIdsWithFolderNameAndOrderCriterionByQuery(releaseOrderMode: Option[ReleaseOrderMode],
                                                                sqlWithParameters: SqlWithParameters): Seq[(AdaptiveReleaseId, Any)] = {
    val (sql, params) = sqlWithParameters
    val rowMapper: RowMapper[(AdaptiveReleaseId, Any)] = (rs, _) => (
      FullReleaseId(rs.getString(1), rs.getString(2)),
      releaseOrderMode match {
        case None => null
        case Some(ReleaseOrderMode.risk) => rs.getInt(3)
        case Some(ReleaseOrderMode.end_date) |
             Some(ReleaseOrderMode.start_date) => rs.getTimestamp(3)
        case Some(ReleaseOrderMode.title) => rs.getString(3)
      })
    jdbcTemplate.query(sql, rowMapper, params: _*).asScala.toSeq
  }

  def findReleaseIdsAndTitlesByQuery(sqlWithParameters: SqlWithParameters): ArrayBuffer[(CiId, String)] = {
    findReleaseColumnsByQuery(sqlWithParameters, rs => FolderId(rs.getString(1)).absolute -> rs.getString(2))
  }

  private def findReleaseColumnsByQuery[R](sqlWithParameters: SqlWithParameters, toModel: (SqlRowSet => R)): ArrayBuffer[R] = {
    val (sql, params) = sqlWithParameters
    val rowSet = jdbcTemplate.queryForRowSet(sql, params: _*)
    val seq = ArrayBuffer.empty[R]
    while (rowSet.next()) {
      seq += toModel(rowSet)
    }
    seq
  }

  def findReleaseDatasByQuery(sqlWithParameters: SqlWithParameters): Seq[ReleaseRow] = {
    // TODO: Performance bottleneck: check usage of this method - in some cases content might not be needed?
    val (sql, params) = sqlWithParameters
    val releaseRowMapper: RowMapper[ReleaseRow] = (rs: ResultSet, _: Int) => {
      // Do NOT rs.get* anything before having completely read the input stream ("object is already closed" on derby)
      // see: https://docs.oracle.com/javase/6/docs/api/java/sql/ResultSet.html#getBinaryStream(int)
      val data = using(rs.getBinaryStream(RELEASES_DATA.CONTENT))(decompress)
      buildReleaseRow(rs, data)
    }
    jdbcTemplate.query[ReleaseRow](sql, params.toArray, releaseRowMapper).asScala.toSeq
  }

  def findReleaseDataWithTemplateDataByQuery(sqlWithParameters: SqlWithParameters): Seq[(ReleaseRow, OriginTemplateDataRow)] = {
    val (sql, params) = sqlWithParameters
    jdbcTemplate.query[(ReleaseRow, OriginTemplateDataRow)](sql, params.toArray, releaseBinaryStreamAndTemplateDataRowMapper).asScala.toSeq
  }

  private val STMT_GET_EFFECTIVE_SECURED_CI =
    s"""|SELECT
        |   r.${RELEASES.CI_UID},
        |   r.${RELEASES.SECURITY_UID},
        |   f.${FOLDERS.FOLDER_PATH},
        |   f.${FOLDERS.FOLDER_ID}
        | FROM ${RELEASES.TABLE} r
        | LEFT JOIN ${FOLDERS.TABLE} f ON f.${FOLDERS.CI_UID} = r.${RELEASES.SECURITY_UID}
        | WHERE ${RELEASES.RELEASE_ID} = :releaseName
     """.stripMargin

  def getEffectiveSecuredCi(releaseId: CiId): SecuredCi = {
    sqlQuery(STMT_GET_EFFECTIVE_SECURED_CI, params(
      "releaseName" -> getName(releaseId.normalized)
    ), rs => {
      val releaseUid = rs.getString(RELEASES.CI_UID)
      val releaseSecurityUid = rs.getString(RELEASES.SECURITY_UID)
      val folderIdOpt = ConfigurationPersistence.mapFolderPath(rs)
      if (releaseUid == releaseSecurityUid) {
        // release has own security
        new SecuredCi(releaseId, releaseUid)
      } else {
        // release inherits security from a folder
        val folderId = folderIdOpt.getOrElse(
          throw new IllegalStateException(s"Effective security of release $releaseId points to a folder $releaseSecurityUid which does not exist")
        )
        new SecuredCi(folderId.absolute, releaseSecurityUid)
      }
    }).headOption.getOrElse {
      throw new NotFoundException(s"Release $releaseId not found")
    }
  }

  private val STMT_INHERIT_SECURITY_FROM_FOLDER =
    s"""|UPDATE ${RELEASES.TABLE}
        | SET ${RELEASES.SECURITY_UID} = :folderUid
        | WHERE ${RELEASES.RELEASE_ID} = :releaseName""".stripMargin

  def inheritSecurityFromFolder(releaseId: String, effectiveSecurityFolderUid: CiUid): Unit = {
    sqlUpdate(STMT_INHERIT_SECURITY_FROM_FOLDER, params(
      "releaseName" -> getName(releaseId.normalized),
      "folderUid" -> effectiveSecurityFolderUid
    ), _ => ())
  }

  private val STMT_SET_AS_EFFECTIVE_SECURED_CI =
    s"""|UPDATE ${RELEASES.TABLE}
        | SET ${RELEASES.SECURITY_UID} = ${RELEASES.CI_UID}
        | WHERE ${RELEASES.RELEASE_ID} = :releaseName""".stripMargin

  def setAsEffectiveSecuredCi(releaseId: String): Unit = {
    sqlUpdate(STMT_SET_AS_EFFECTIVE_SECURED_CI, params(
      "releaseName" -> getName(releaseId.normalized)
    ), _ => ())
  }

  private val STMT_REPLACE_SECURITY_UID =
    s"""|UPDATE ${RELEASES.TABLE}
        | SET ${RELEASES.SECURITY_UID} = :newSecurityUid
        | WHERE ${RELEASES.SECURITY_UID} = :oldSecurityUid
        |   AND ${RELEASES.FOLDER_CI_UID} IN (
        |     SELECT v2.ciUid FROM (
        |       SELECT
        |           v.${VIEW.DESCENDANT_UID} AS ciUid
        |         FROM ${VIEW.TABLE} v
        |         WHERE v.${VIEW.ANCESTOR_UID} = :folderUid
        |     ) v2
        |   )""".stripMargin

  def replaceSecurityUid(forReleasesUnderFolderUid: CiUid, oldSecurityUid: CiUid, newSecurityUid: CiUid): Unit = {
    if (oldSecurityUid != newSecurityUid) {
      sqlUpdate(STMT_REPLACE_SECURITY_UID, params(
        "folderUid" -> forReleasesUnderFolderUid,
        "newSecurityUid" -> newSecurityUid,
        "oldSecurityUid" -> oldSecurityUid
      ), _ => ())
    }
  }

  private val STMT_FIND_ALL_TAGS =
    s"""|SELECT DISTINCT ${TAGS.VALUE}
        | FROM ${TAGS.TABLE}
        | ORDER BY ${TAGS.VALUE}""".stripMargin

  def findAllTags(limitNumber: Int): Set[String] = {
    sqlQuery(addLimitAndOffset(STMT_FIND_ALL_TAGS, Some(limitNumber)), params(), rs => rs.getString(1)).toSet
  }

  def setPreArchived(releaseId: CiId, preArchived: Boolean): Boolean = {
    sqlUpdate(STMT_SET_PRE_ARCHIVED, params("releaseId" -> getName(releaseId.normalized), "preArchived" -> preArchived.asInteger), _ == 1)
  }

  def getPreArchived(releaseId: CiId): Option[Boolean] = {
    findOne(sqlQuery(STMT_GET_PRE_ARCHIVED, params("releaseId" -> getName(releaseId.normalized)), _.getInt(RELEASES.PRE_ARCHIVED).asBoolean))
  }

  private val STMT_INSERT_RELEASE =
    s"""INSERT INTO ${RELEASES.TABLE} (
       |  ${RELEASES.CI_UID},
       |  ${RELEASES.RELEASE_ID},
       |  ${RELEASES.ROOT_RELEASE_ID},
       |  ${RELEASES.FOLDER_CI_UID},
       |  ${RELEASES.STATUS},
       |  ${RELEASES.RELEASE_TITLE},
       |  ${RELEASES.START_DATE},
       |  ${RELEASES.END_DATE},
       |  ${RELEASES.RISK_SCORE},
       |  ${RELEASES.TOTAL_RISK_SCORE},
       |  ${RELEASES.RELEASE_OWNER},
       |  ${RELEASES.CALENDAR_TOKEN},
       |  ${RELEASES.AUTO_START},
       |  ${RELEASES.REAL_FLAG_STATUS},
       |  ${RELEASES.SECURITY_UID},
       |  ${RELEASES.SCM_DATA},
       |  ${RELEASES.PLANNED_DURATION},
       |  ${RELEASES.IS_OVERDUE_NOTIFIED},
       |  ${RELEASES.ORIGIN_TEMPLATE_ID},
       |  ${RELEASES.KIND},
       |  ${RELEASES.PARENT_RELEASE_ID})
       |VALUES(?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?)
     """.stripMargin

  private def setTraceabilityId(idx: Int, release: Release, ps: PreparedStatement): Unit = {
    if (getSCMData(release.get$ciAttributes().getScmTraceabilityDataId) != null) {
      ps.setString(idx, release.get$ciAttributes().getScmTraceabilityDataId)
    } else {
      ps.setNull(idx, java.sql.Types.INTEGER)
    }
  }

  private def insertRelease(ciUid: CiUid, release: Release, parentFolder: FolderRow) = {
    jdbcTemplate.execute(STMT_INSERT_RELEASE,
      new PreparedStatementCallback[Boolean] {

        override def doInPreparedStatement(ps: PreparedStatement): Boolean = {
          try {
            ps.setString(1, ciUid)
            ps.setString(2, getName(release.getId))
            ps.setString(3, getName(release.getRootReleaseId))
            ps.setString(4, parentFolder.uid)
            ps.setString(5, release.getStatus.value())
            ps.setString(6, release.getTitle.truncate(COLUMN_LENGTH_TITLE))
            ps.setTimestamp(7, release.getStartOrScheduledDate.asTimestamp)
            ps.setTimestamp(8, release.getEndOrDueDate.asTimestamp)
            ps.setInt(9, Integer.valueOf(release.getPropertyOption("riskScore").getOrElse("0")))
            ps.setInt(10, Integer.valueOf(release.getPropertyOption("totalRiskScore").getOrElse("0")))
            ps.setString(11, release.getOwner.truncate(COLUMN_LENGTH_AUTHORITY_NAME))
            ps.setString(12, release.getCalendarLinkToken)
            ps.setInt(13, release.isAutoStart.asInteger)
            ps.setInt(14, release.getRealFlagStatus.getRisk)
            // inherit security by default, this may be overridden later when saving release teams
            ps.setString(15, parentFolder.securityUid)
            setTraceabilityId(16, release, ps)
            if (release.getPlannedDuration == null) {
              ps.setNull(17, Types.INTEGER)
            } else {
              ps.setInt(17, release.getPlannedDuration)
            }
            ps.setInt(18, release.isOverdueNotified.asInteger)
            ps.setString(19, getName(release.getOriginTemplateId))
            ps.setString(20, release.getKind.value())
            ps.setString(21, getName(release.getParentReleaseId))
            release.setCiUid(ciUid)
            ps.execute()
          } catch {
            case e: Throwable =>
              throw new ReleaseStoreException(s"Failed to store release ${release.getId}", e)
          }
        }
      })
  }

  private val STMT_INSERT_RELEASE_CONTENT =
    s"""INSERT INTO ${RELEASES_DATA.TABLE}
       |  ( ${RELEASES_DATA.CI_UID}
       |  , ${RELEASES_DATA.CONTENT}
       |  )
       |  VALUES
       |  ( :ciUid
       |  , :content
       |  )""".stripMargin

  private def insertReleaseJson(ciUid: CiUid, release: Release): CiUid = {
    val json = serializeRelease(release)
    sqlExecWithContent(STMT_INSERT_RELEASE_CONTENT, params("ciUid" -> ciUid), "content" -> json, identity)
    // we want release object (original method parameter) to be passed into the releaseRow
    releaseCacheService.put(release.getId, releaseRow(release, json, release.get$securedCi()))
    ciUid
  }

  private def serializeRelease(release: Release): String = {
    val copy = CiCloneHelper.cloneCi(release)
    // don't store folder info in the JSON so that it does not have to be updated on move operations
    rewriteWithNewId(copy, getName(copy.getId))
    // don't store teams in JSON as they are stored in security tables and loaded through decorations
    copy.setTeams(emptyList())
    copy.setExtensions(null)
    copy.setTags(distinctTags(release.getTags, preserveCase = true).toList.asJavaMutable())
    copy.getAllTasks.forEach { t =>
      t.getComments.clear()
      t.setFacets(null)
      t.setTags(distinctTags(t.getTags, preserveCase = true).toList.asJavaMutable())
    }
    copy.setAuthor(null)
    copy.setLogo(null)
    copy.setCategories(Collections.emptySet())
    copy.setDefaultTargetFolderId(null)

    serialize(copy, new PasswordEncryptingCiConverter())
  }

  private def insertTags(ciUid: CiUid)(tags: Set[String]): Unit = {
    val listTags = tags.map(_.truncate(COLUMN_LENGTH_TITLE)).toList
    jdbcTemplate.batch(s"INSERT INTO ${TAGS.TABLE} (${TAGS.CI_UID}, ${TAGS.VALUE}) VALUES(?, ?)", ciUid, listTags)
  }

  private def selectTags(pk: CiUid): Set[String] =
    jdbcTemplate.queryForList(
      s"SELECT ${TAGS.VALUE} FROM ${TAGS.TABLE} WHERE ${TAGS.CI_UID} = ?",
      classOf[String],
      pk
    ).asScala.toSet

  private def deleteTags(ciUid: CiUid)(deletedTags: Set[String]): Unit = {
    val deletedTagsList = deletedTags.toList
    jdbcTemplate.batch(s"DELETE FROM ${TAGS.TABLE} WHERE ${TAGS.CI_UID} = ? AND ${TAGS.VALUE} = ?", ciUid, deletedTagsList)
  }

  private def distinctTags(tags: java.util.List[String], preserveCase: Boolean = false): Set[String] =
    Option(tags)
      .map(_.asScala.toSet)
      .map(tagsList => if (preserveCase) tagsList else tagsList.map(_.toLowerCase()))
      .getOrElse(Set.empty[String])

  private val releaseRowMapper: RowMapper[ReleaseRow] = (rs: ResultSet, _: Int) => {
    buildReleaseRow(rs, "empty")
  }

  private val basicReleaseRowMapper: RowMapper[BasicReleaseRow] = (rs, _) => BasicReleaseRow(
    releaseId = rs.getString(RELEASES.RELEASE_ID),
    ciUid = rs.getString(RELEASES.CI_UID),
    folderId = (FolderId(rs.getString(FOLDERS.FOLDER_PATH)) / rs.getString(FOLDERS.FOLDER_ID)).absolute,
    securityUid = rs.getString(RELEASES.SECURITY_UID),
    riskScore = rs.getInt(RELEASES.RISK_SCORE),
    totalRiskScore = rs.getInt(RELEASES.TOTAL_RISK_SCORE),
    scmDataUid = rs.getString(RELEASES.SCM_DATA),
    plannedDuration = rs.getObject(RELEASES.PLANNED_DURATION, classOf[Integer]),
    status = rs.getString(RELEASES.STATUS),
    realFlagStatus = rs.getInt(RELEASES.REAL_FLAG_STATUS),
    title = rs.getString(RELEASES.RELEASE_TITLE),
    startDate = rs.getTimestamp(RELEASES.START_DATE),
    endDate = rs.getTimestamp(RELEASES.END_DATE),
    isOverdueNotified = rs.getInt(RELEASES.IS_OVERDUE_NOTIFIED).asBoolean,
    kind = rs.getString(RELEASES.KIND),
    owner = rs.getString(RELEASES.RELEASE_OWNER)
  )

  private val releaseOverviewDataMapper: RowMapper[ReleaseOverviewDataRow] = (rs, _) => ReleaseOverviewDataRow(
    releaseId = rs.getString(RELEASES.RELEASE_ID),
    ciUid = rs.getString(RELEASES.CI_UID),
    folderId = (FolderId(rs.getString(FOLDERS.FOLDER_PATH)) / rs.getString(FOLDERS.FOLDER_ID)).absolute,
    securityUid = rs.getString(RELEASES.SECURITY_UID),
    riskScore = rs.getInt(RELEASES.RISK_SCORE),
    totalRiskScore = rs.getInt(RELEASES.TOTAL_RISK_SCORE),
    scmDataUid = rs.getString(RELEASES.SCM_DATA),
    plannedDuration = rs.getObject(RELEASES.PLANNED_DURATION, classOf[Integer]),
    status = rs.getString(RELEASES.STATUS),
    realFlagStatus = rs.getInt(RELEASES.REAL_FLAG_STATUS),
    title = rs.getString(RELEASES.RELEASE_TITLE),
    startDate = rs.getTimestamp(RELEASES.START_DATE),
    endDate = rs.getTimestamp(RELEASES.END_DATE),
    kind = rs.getString(RELEASES.KIND),
    currentPhaseTitle = rs.getString(CURRENT_PHASE_TITLE),
    owner = rs.getString(RELEASES.RELEASE_OWNER)
  )

  private val releaseBinaryStreamRowMapper: RowMapper[String] = (rs: ResultSet, _: Int) => {
    // Do NOT rs.get* anything before having completely read the input stream ("object is already closed" on derby)
    // see: https://docs.oracle.com/javase/6/docs/api/java/sql/ResultSet.html#getBinaryStream(int)
    using(rs.getBinaryStream(RELEASES_DATA.CONTENT))(decompress)
  }

  private val releaseBinaryStreamAndTemplateDataRowMapper: RowMapper[(ReleaseRow, OriginTemplateDataRow)] = (rs: ResultSet, _: Int) => {
    val data = using(rs.getBinaryStream(RELEASES_DATA.CONTENT))(decompress)
    buildReleaseRow(rs, data) ->
      OriginTemplateDataRow(
        originTemplateId = rs.getString(TEMP_ORIGIN_TEMPLATE_ID),
        originTemplateTitle = rs.getString(TEMP_ORIGIN_TEMPLATE_TITLE)
      )
  }

  private def buildReleaseRow(rs: ResultSet, releaseData: String): ReleaseRow = {
    ReleaseRow(
      json = releaseData,
      releaseId = rs.getString(RELEASES.RELEASE_ID),
      ciUid = rs.getString(RELEASES.CI_UID),
      folderId = (FolderId(rs.getString(FOLDERS.FOLDER_PATH)) / rs.getString(FOLDERS.FOLDER_ID)).absolute,
      securityUid = rs.getString(RELEASES.SECURITY_UID),
      riskScore = rs.getInt(RELEASES.RISK_SCORE),
      totalRiskScore = rs.getInt(RELEASES.TOTAL_RISK_SCORE),
      scmDataUid = rs.getString(RELEASES.SCM_DATA),
      plannedDuration = rs.getObject(RELEASES.PLANNED_DURATION, classOf[Integer]),
      status = rs.getString(RELEASES.STATUS),
      realFlagStatus = rs.getInt(RELEASES.REAL_FLAG_STATUS),
      title = rs.getString(RELEASES.RELEASE_TITLE),
      startDate = rs.getDate(RELEASES.START_DATE),
      endDate = rs.getDate(RELEASES.END_DATE),
      isOverdueNotified = rs.getInt(RELEASES.IS_OVERDUE_NOTIFIED).asBoolean,
      kind = rs.getString(RELEASES.KIND)
    )
  }

  val overdueReleasesQuery = new NotificationQueries.OverdueReleasesQuery(dbInfo, namedTemplate)

  def findOverdueReleaseIds(): Seq[String] = {
    overdueReleasesQuery.execute
  }

  private val STMT_FIND_RELEASE_IDS_BY_FOLDER_CI_UID =
    s"""SELECT ${RELEASES.RELEASE_ID}
       |FROM ${RELEASES.TABLE}
       |WHERE ${RELEASES.FOLDER_CI_UID} = :folderCiUid
       |""".stripMargin

  def findReleaseIdsByFolderCiUid(folderCiUid: CiUid): Seq[String] = {
    sqlQuery(
      STMT_FIND_RELEASE_IDS_BY_FOLDER_CI_UID,
      params("folderCiUid" -> folderCiUid),
      _.getString(1)
    ).toSeq
  }

  val SOURCE_RELEASE_ID_COLUMN = "SOURCE_RELEASE_ID"
  val SOURCE_TASK_ID_COLUMN = "SOURCE_TASK_ID"
  val SOURCE_TASK_UID_COLUMN = "SOURCE_TASK_UID"
  val TARGET_RELEASE_ID_COLUMN = "TARGET_RELEASE_ID"
  val TARGET_RELEASE_SUBOBJECT_ID_COLUMN = "TARGET_RELEASE_SUBOBJECT_ID"

  private val dependencySubQuery: String =
    s"""
       |SELECT
       |  targetRels.RELEASE_ID as $TARGET_RELEASE_ID_COLUMN,
       |  dep.GATE_TASK_UID as $SOURCE_TASK_UID_COLUMN,
       |  dep.TARGET_ID as $TARGET_RELEASE_SUBOBJECT_ID_COLUMN
       |FROM XLR_DEPENDENCIES dep
       |LEFT JOIN XLR_RELEASES targetRels ON dep.TARGET_RELEASE_UID = targetRels.CI_UID
       |""".stripMargin

  private def releaseDependencyQuery(ids: List[String]): Sql = {
    val questionMarks = ids.map(_ => "?").mkString(",")
    val query =
      s"""
         |SELECT
         |  sourceRels.RELEASE_ID as $SOURCE_RELEASE_ID_COLUMN,
         |  sourceTasks.TASK_ID as $SOURCE_TASK_ID_COLUMN,
         |  deps.$TARGET_RELEASE_ID_COLUMN as $TARGET_RELEASE_ID_COLUMN,
         |  deps.$TARGET_RELEASE_SUBOBJECT_ID_COLUMN as $TARGET_RELEASE_SUBOBJECT_ID_COLUMN
         |FROM XLR_RELEASES sourceRels
         |LEFT JOIN XLR_TASKS sourceTasks ON sourceRels.CI_UID = sourceTasks.RELEASE_UID
         |RIGHT JOIN ($dependencySubQuery) deps ON sourceTasks.CI_UID = deps.$SOURCE_TASK_UID_COLUMN
         |WHERE
         |  sourceRels.RELEASE_ID IN ($questionMarks)
         |AND
         |  sourceTasks.TASK_TYPE='xlrelease.GateTask'
         |""".stripMargin
    Sql(query, ids)
  }

  private def queryRowMapper: RowMapper[ReleaseDependency] = (rs: ResultSet, _: Int) => {
    ReleaseDependency(
      rs.getString(SOURCE_RELEASE_ID_COLUMN),
      rs.getString(SOURCE_TASK_ID_COLUMN),
      rs.getString(TARGET_RELEASE_ID_COLUMN),
      rs.getString(TARGET_RELEASE_SUBOBJECT_ID_COLUMN)
    )
  }

  def getDependencies(releaseIds: List[String]): List[ReleaseDependency] = {
    val query = releaseDependencyQuery(releaseIds)
    jdbcTemplate.query[ReleaseDependency](query.sql, queryRowMapper, query.parameters.toSeq: _*).asScala.toList
  }

  private def validateTaskSchedulingPropertiesForWorkflow(release: Release): Unit = {
    if (release.isWorkflow) {
      val hasDelayedTasksDuringBlackout = release.getAllTasks.asScala.exists(_.isDelayDuringBlackout)
      if (hasDelayedTasksDuringBlackout) {
        throw new IncorrectArgumentException("Postpone during blackout (delayDuringBlackout property) is not supported for workflow tasks")
      }
      val hasCheckAttributes = release.getAllTasks.asScala.exists(_.isCheckAttributes)
      if (hasCheckAttributes) {
        throw new IncorrectArgumentException("Check environment availability (checkAttributes property) is not supported for workflow tasks")
      }
    }
  }

  def findAbortableReleaseIds(date: Date, pageSize: Option[Long]): Seq[ReleaseIdWithCiUid] = {
    sqlQuery(
      addLimitAndOffset(STMT_GET_ABORTABLE_IDS, pageSize),
      params(RELEASES.START_DATE -> date),
      rs => ReleaseIdWithCiUid(rs.getString(RELEASES.RELEASE_ID), rs.getString(RELEASES.CI_UID))
    ).toSeq
  }

  def findAbortableTaskIdHashes(releaseCiUid: CiUid): Seq[Hash] = {
    sqlQuery(
      STMT_GET_ABORTABLE_TASK_DATA,
      params(TASKS.RELEASE_UID -> releaseCiUid),
      _.getString(TASKS.TASK_ID)
    ).toSeq
  }

  def countTemplatesByKind(releaseKind: ReleaseKind): Int = {
    namedTemplate.queryForObject(STMT_COUNT_NO_OF_TEMPLATES, paramSource(RELEASES.KIND -> releaseKind.value()), classOf[Int])
  }

  def releasesOverview(releasesFilters: ReleasesFilters,
                       pageNum: Long,
                       numberByPage: Long,
                       currentPrincipals: Iterable[String],
                       currentRoleIds: Iterable[String],
                       permissionEnforcer: PermissionEnforcer
                      ): List[ReleaseOverviewDataRow] = {
    if (releasesFilters.getOrderBy == null) {
      releasesFilters.setOrderBy(ReleaseOrderMode.risk)
    }
    val sqlBuilder = SqlReleasesFilterSupport
      .sqlBuilderByFilters(releasesFilters, currentPrincipals, currentRoleIds)(permissionEnforcer, dialect)
    searchOverview(sqlBuilder, pageNum, numberByPage)
  }

  def templatesOverview(templateFilters: TemplateFilters,
                        pageNum: Long,
                        numberByPage: Long,
                        currentPrincipals: Iterable[String],
                        currentRoleIds: Iterable[String],
                        permissionEnforcer: PermissionEnforcer
                       ): List[ReleaseOverviewDataRow] = {
    val sqlBuilder = SqlTemplatesFilterSupport
      .sqlBuilderByFilters(templateFilters, currentPrincipals, currentRoleIds)(permissionEnforcer, dialect)
    searchOverview(sqlBuilder, pageNum, numberByPage)
  }

  private def searchOverview(releasesSqlBuilder: ReleasesSqlBuilder,
                             pageNum: Long,
                             numberByPage: Long): List[ReleaseOverviewDataRow] = {
    val query = releasesSqlBuilder
      .addFolderJoin()
      .addPhaseJoinWithAnyStatus(Set(PhaseStatus.IN_PROGRESS, PhaseStatus.FAILED, PhaseStatus.FAILING))
      .customSelect(columns =
        s"$alias.${RELEASES.RELEASE_ID}",
        s"$alias.${RELEASES.CI_UID}",
        s"$folderAlias.${FOLDERS.FOLDER_PATH}",
        s"$folderAlias.${FOLDERS.FOLDER_ID}",
        s"$alias.${RELEASES.SECURITY_UID}",
        s"$alias.${RELEASES.RISK_SCORE}",
        s"$alias.${RELEASES.TOTAL_RISK_SCORE}",
        s"$alias.${RELEASES.SCM_DATA}",
        s"$alias.${RELEASES.PLANNED_DURATION}",
        s"$alias.${RELEASES.STATUS}",
        s"$alias.${RELEASES.REAL_FLAG_STATUS}",
        s"$alias.${RELEASES.RELEASE_TITLE}",
        s"$alias.${RELEASES.START_DATE}",
        s"$alias.${RELEASES.END_DATE}",
        s"$alias.${RELEASES.IS_OVERDUE_NOTIFIED}",
        s"$alias.${RELEASES.KIND}",
        s"$alias.${RELEASES.RELEASE_OWNER}",
        s"$phaseAlias.${PHASES.PHASE_TITLE} AS $CURRENT_PHASE_TITLE"
      )
    val page = Page(pageNum, numberByPage, 0)
    val (sql, params) = query.withPage(page).build()
    val rowList = jdbcTemplate.query[ReleaseOverviewDataRow](sql, releaseOverviewDataMapper, params: _*).asScala
    logger.debug(s"Searching for a batch of releases by SQL query: $sql")
    val releaseRows = rowList.map(row => {
      val fullReleaseId = formatWithFolderId(row.folderId, row.releaseId)
      row.copy(releaseId = fullReleaseId)
    })
    releaseRows.toList
  }
}

//scalastyle:on number.of.methods

object ReleasePersistence {
  private lazy val STMT_READ_RELEASE_CONTENT: String =
    s"""|SELECT ${RELEASES_DATA.CONTENT}
        | FROM ${RELEASES_DATA.TABLE}
        | WHERE ${RELEASES_DATA.CI_UID} = :${RELEASES_DATA.CI_UID}
     """.stripMargin

  lazy val STMT_UPDATE_RELEASE_CONTENT: String =
    s"""|UPDATE ${RELEASES_DATA.TABLE}
        | SET ${RELEASES_DATA.CONTENT} = :content
        | WHERE ${RELEASES_DATA.CI_UID} = :ciUid
     """.stripMargin

  private lazy val STMT_SET_PRE_ARCHIVED: String =
    s"""|UPDATE ${RELEASES.TABLE}
        | SET ${RELEASES.PRE_ARCHIVED} = :preArchived
        | WHERE ${RELEASES.RELEASE_ID} = :releaseId
     """.stripMargin

  private lazy val STMT_GET_PRE_ARCHIVED: String =
    s"""|SELECT ${RELEASES.PRE_ARCHIVED}
        | FROM ${RELEASES.TABLE}
        | WHERE ${RELEASES.RELEASE_ID} = :releaseId
     """.stripMargin

  private lazy val STMT_COUNT_NO_OF_TEMPLATES: String =
    s"""|SELECT COUNT(1)
        | FROM ${RELEASES.TABLE}
        | WHERE ${RELEASES.STATUS} = '${ReleaseStatus.TEMPLATE.value()}'
        | AND ${RELEASES.KIND} = :${RELEASES.KIND}
        |""".stripMargin

  private lazy val STMT_GET_ABORTABLE_IDS: String =
    s"""|
        |SELECT ${RELEASES.RELEASE_ID}, ${RELEASES.CI_UID}
        | FROM ${RELEASES.TABLE}
        | WHERE ${RELEASES.STATUS} IN ($ABORTABLE_STATUSES)
        | AND ${RELEASES.START_DATE} < :${RELEASES.START_DATE}
    """.stripMargin

  private lazy val STMT_GET_ABORTABLE_TASK_DATA: String =
    s"""|
          |SELECT ${TASKS.TASK_ID}
        | FROM ${TASKS.TABLE}
        | WHERE ${TASKS.RELEASE_UID} = :${TASKS.RELEASE_UID}
        | AND ${TASKS.IS_AUTOMATED} = 1
        | AND ${TASKS.STATUS} IN ($ABORTABLE_TASK_STATUSES)
      """.stripMargin

  private val ABORTABLE_STATUSES: String = ReleaseStatus.ACTIVE_STATUSES.toSet[ReleaseStatus].map(status => s"'${status.value()}'").mkString(",")
  private val ABORTABLE_TASK_STATUSES: String = TaskStatus.ACTIVE_STATUSES.toSet[TaskStatus].map(status => s"'${status.value()}'").mkString(",")
}

class ReleaseStoreException(msg: String, cause: Throwable) extends RuntimeException(msg) {
  val logger: Logger = LoggerFactory.getLogger(getClass)
  // notice that we did not init cause, but we would like to log this ALWAYS
  if (cause != null) logger.error(msg, cause) else logger.error(msg)

  def this(msg: String) = this(msg, null)

}

case class ReleaseDependency(sourceReleaseId: String, sourceTaskId: String, targetReleaseId: String, targetSubObjectId: String)
