package com.xebialabs.xlrelease.repository.sql.persistence

import com.xebialabs.xlrelease.db.sql.SqlBuilder.Dialect
import com.xebialabs.xlrelease.db.sql.transaction.IsTransactional
import com.xebialabs.xlrelease.domain.id.CiUid
import com.xebialabs.xlrelease.repository.sql.SqlRepository
import com.xebialabs.xlrelease.repository.sql.persistence.CiId.CiId
import com.xebialabs.xlrelease.repository.sql.persistence.Schema.{RELEASES, RELEASES_EXT}
import com.xebialabs.xlrelease.repository.sql.persistence.Utils.params
import com.xebialabs.xlrelease.repository.sql.persistence.data.ReleaseExtensionRow
import grizzled.slf4j.Logging
import org.springframework.jdbc.core.{JdbcTemplate, RowMapper}

import scala.jdk.CollectionConverters._

@IsTransactional
class ReleaseExtensionPersistence(val jdbcTemplate: JdbcTemplate, val dialect: Dialect)
  extends SqlRepository with PersistenceSupport with Logging with Utils {

  private val STMT_EXISTS_BY_ID: String =
    s"""|SELECT ${RELEASES.CI_UID}
        |FROM ${RELEASES_EXT.TABLE} ext
        |JOIN ${RELEASES.TABLE} rel ON rel.${RELEASES.CI_UID} = ext.${RELEASES_EXT.RELEASE_UID}
        |WHERE
        |   rel.${RELEASES.RELEASE_ID} = :releaseId AND
        |   ext.${RELEASES_EXT.EXTENSION_ID} = :extensionId""".stripMargin

  def exists(extensionId: CiId): Option[CiUid] = {
    logger.debug(s"exists($extensionId)")
    findOne(
      sqlQuery(STMT_EXISTS_BY_ID,
        params(
          "releaseId" -> ReleaseExtensionRow.toInternalReleaseId(extensionId),
          "extensionId" -> ReleaseExtensionRow.toInternalId(extensionId)
        ),
        rs => CiUid(rs.getString(RELEASES.CI_UID))
      )
    )
  }

  private val STMT_READ: String =
    s"""|SELECT
        |   ${RELEASES_EXT.EXTENSION_ID},
        |   ${RELEASES_EXT.EXTENSION_TYPE},
        |   ${RELEASES_EXT.CONTENT}
        |FROM ${RELEASES_EXT.TABLE}
        |WHERE
        |   ${RELEASES_EXT.RELEASE_UID} = :releaseUid AND
        |   ${RELEASES_EXT.EXTENSION_ID} = :extensionId
     """.stripMargin

  def read(releaseUid: CiUid, extensionId: CiId): Option[ReleaseExtensionRow] = {
    logger.debug(s"findById($releaseUid, $extensionId)")
    findOne(
      sqlQuery(STMT_READ,
        params(
          "releaseUid" -> releaseUid,
          "extensionId" -> ReleaseExtensionRow.toInternalId(extensionId)
        ),
        releaseExtensionRowMapper
      )
    )
  }

  private val STMT_READ_ALL: String =
    s"""|SELECT
        |   ${RELEASES_EXT.EXTENSION_ID},
        |   ${RELEASES_EXT.EXTENSION_TYPE},
        |   ${RELEASES_EXT.CONTENT}
        |FROM ${RELEASES_EXT.TABLE}
        |WHERE
        |   ${RELEASES_EXT.RELEASE_UID} = :releaseUid
        |ORDER BY ${RELEASES_EXT.EXTENSION_ID} ASC""".stripMargin

  def readAll(releaseUid: CiUid): Seq[ReleaseExtensionRow] = {
    logger.debug(s"findByReleaseId($releaseUid)")
    sqlQuery(STMT_READ_ALL, params("releaseUid" -> releaseUid), releaseExtensionRowMapper).toSeq
  }

  private val STMT_CREATE: String =
    s"""|INSERT INTO ${RELEASES_EXT.TABLE} (
        |   ${RELEASES_EXT.RELEASE_UID},
        |   ${RELEASES_EXT.EXTENSION_ID},
        |   ${RELEASES_EXT.EXTENSION_TYPE},
        |   ${RELEASES_EXT.CONTENT}
        |) VALUES (:releaseUid, :extensionId, :extensionType, :content)
     """.stripMargin


  def create(releaseUid: CiUid, releaseExtension: ReleaseExtensionRow): Unit = {
    logger.debug(s"create($releaseUid, ${releaseExtension.id})")
    sqlExecWithContent(STMT_CREATE,
      params(
        "releaseUid" -> releaseUid,
        "extensionId" -> releaseExtension.id,
        "extensionType" -> releaseExtension.ciType
      ),
      "content" -> releaseExtension.json,
      _ => ()
    )
  }

  def createAll(releaseUid: CiUid, releaseExtensions: Seq[ReleaseExtensionRow]): Int = {
    logger.debug(s"createAll($releaseUid, ${releaseExtensions.map(_.id)})")
    sqlBatchWithContent(STMT_CREATE,
      releaseExtensions.map(row => params(
        "releaseUid" -> releaseUid,
        "extensionId" -> row.id,
        "extensionType" -> row.ciType
      ) -> ("content" -> row.json))
    ).sum
  }

  private val STMT_UPDATE_EXTENSION: String =
    s"""|UPDATE ${RELEASES_EXT.TABLE}
        | SET
        |   ${RELEASES_EXT.EXTENSION_TYPE} = :extensionType,
        |   ${RELEASES_EXT.CONTENT} = :content
        | WHERE
        |   ${RELEASES_EXT.RELEASE_UID} = :releaseUid AND
        |   ${RELEASES_EXT.EXTENSION_ID} = :extensionId
     """.stripMargin

  def update(releaseUid: CiUid, releaseExtension: ReleaseExtensionRow): Int = {
    logger.debug(s"update($releaseUid, ${releaseExtension.id})")
    sqlExecWithContent(STMT_UPDATE_EXTENSION,
      params(
        "releaseUid" -> releaseUid,
        "extensionId" -> releaseExtension.id,
        "extensionType" -> releaseExtension.ciType
      ),
      "content" -> releaseExtension.json,
      identity
    )
  }

  private val STMT_DELETE_EXTENSION: String =
    s"""|DELETE FROM ${RELEASES_EXT.TABLE}
        |WHERE
        |   ${RELEASES_EXT.RELEASE_UID} = :releaseUid AND
        |   ${RELEASES_EXT.EXTENSION_ID} = :extensionId
     """.stripMargin

  def delete(releaseUid: CiUid, extensionId: String): Int = {
    logger.debug(s"delete($releaseUid, $extensionId)")
    sqlUpdate(STMT_DELETE_EXTENSION,
      params(
        "releaseUid" -> releaseUid,
        "extensionId" -> ReleaseExtensionRow.toInternalId(extensionId)
      ),
      identity
    )
  }

  private val STMT_DELETE_ALL: String =
    s"""|DELETE FROM ${RELEASES_EXT.TABLE}
        |WHERE
        |   ${RELEASES_EXT.RELEASE_UID} = :releaseUid
     """.stripMargin

  def deleteAll(releaseUid: CiUid): Int = {
    logger.debug(s"deleteByReleaseId($releaseUid")
    sqlUpdate(STMT_DELETE_ALL, params("releaseUid" -> releaseUid), identity)
  }


  private val STMT_FIND_BY_RELEASE_UIDS_AND_EXTENSION_ID: String =
    s"""|SELECT
        |   ${RELEASES_EXT.EXTENSION_ID},
        |   ${RELEASES_EXT.EXTENSION_TYPE},
        |   ${RELEASES_EXT.CONTENT}
        |FROM ${RELEASES_EXT.TABLE}
        |WHERE
        |   ${RELEASES_EXT.RELEASE_UID} IN ( :releaseUids )
        |   AND ${RELEASES_EXT.EXTENSION_ID} = :extensionId
        |ORDER BY ${RELEASES_EXT.EXTENSION_ID} ASC""".stripMargin

  def findByExtensionIdAndReleaseUids(extensionId: String, releaseUids: Seq[String]): Seq[ReleaseExtensionRow] = {
    if (releaseUids.isEmpty) {
      Seq.empty[ReleaseExtensionRow]
    } else {
      logger.debug(s"findByExtensionIdAndReleaseUids($extensionId, ${releaseUids.mkString(",")})")
      val pars = params(
        "releaseUids" -> releaseUids.asJava,
        "extensionId" -> extensionId
      )
      sqlQuery(STMT_FIND_BY_RELEASE_UIDS_AND_EXTENSION_ID, pars, releaseExtensionRowMapper).toSeq
    }
  }


  val releaseExtensionRowMapper: RowMapper[ReleaseExtensionRow] =
    (rs, _) => ReleaseExtensionRow.fromDatabase(rs)
}
