package com.xebialabs.xlrelease.repository.sql.persistence

import com.xebialabs.xlrelease.db.sql.SqlBuilder.Dialect
import com.xebialabs.xlrelease.db.sql.transaction.IsTransactional
import com.xebialabs.xlrelease.domain.id.IdCreator
import com.xebialabs.xlrelease.repository.sql.SqlRepository
import com.xebialabs.xlrelease.repository.sql.persistence.Schema.JOURNAL_ROW
import grizzled.slf4j.Logging
import org.apache.pekko.persistence.jdbc.journal.dao.legacy.JournalRow
import org.springframework.jdbc.core.JdbcTemplate
import org.springframework.jdbc.core.namedparam.{MapSqlParameterSource, SqlParameterSource}
import org.springframework.jdbc.core.support.SqlLobValue

import java.sql.Types

@IsTransactional
class PekkoJournalPersistence(implicit val jdbcTemplate: JdbcTemplate,
                              implicit val dialect: Dialect)
  extends SqlRepository with PersistenceSupport with Logging with Utils {

  def buildInsertQuery(tableName: String) =
    s"""INSERT INTO $tableName (
       |  ${JOURNAL_ROW.ORDERING},
       |  ${JOURNAL_ROW.PERSISTENCE_ID},
       |  ${JOURNAL_ROW.SEQUENCE_NUMBER},
       |  ${JOURNAL_ROW.DELETED},
       |  ${JOURNAL_ROW.TAGS},
       |  ${JOURNAL_ROW.MESSAGE}
       |  ) VALUES (
       |  :${JOURNAL_ROW.ORDERING},
       |  :${JOURNAL_ROW.PERSISTENCE_ID},
       |  :${JOURNAL_ROW.SEQUENCE_NUMBER},
       |  :${JOURNAL_ROW.DELETED},
       |  :${JOURNAL_ROW.TAGS},
       |  :${JOURNAL_ROW.MESSAGE}
       |  )
       |  """.stripMargin

  def batchInsert(journalRows: Seq[JournalRow], tableName: String): Unit = {
    //Has to be long or whole persistence layer has to be changed
    val id: Long = IdCreator.generateLongId
    val batchArgs: Array[SqlParameterSource] = journalRows.collect {
      case JournalRow(_, deleted, persistenceId, sequenceNumber, message, tags) =>
        val params = new MapSqlParameterSource()
        params.addValue(JOURNAL_ROW.ORDERING, id)
        params.addValue(JOURNAL_ROW.PERSISTENCE_ID, persistenceId)
        params.addValue(JOURNAL_ROW.SEQUENCE_NUMBER, sequenceNumber)
        params.addValue(JOURNAL_ROW.DELETED, deleted)
        params.addValue(JOURNAL_ROW.TAGS, tags.orNull)
        params.addValue(JOURNAL_ROW.MESSAGE, new SqlLobValue(message), Types.BLOB)
        params
    }.toArray

    namedTemplate.batchUpdate(buildInsertQuery(tableName), batchArgs)
  }

}
