package com.xebialabs.xlrelease.repository.sql.persistence

import com.xebialabs.xlrelease.db.sql.SqlBuilder.{Dialect, MSSQLDialect}
import com.xebialabs.xlrelease.db.sql.transaction.IsTransactional
import com.xebialabs.xlrelease.domain.id.{CiUid, IdCreator}
import com.xebialabs.xlrelease.domain.variables.Variable
import com.xebialabs.xlrelease.repository.Ids.getName
import com.xebialabs.xlrelease.repository.sql.SqlRepositoryAdapter
import com.xebialabs.xlrelease.repository.sql.persistence.CiId.CiId
import com.xebialabs.xlrelease.repository.sql.persistence.Schema.{FOLDERS, FOLDER_VARIABLES}
import com.xebialabs.xlrelease.repository.sql.persistence.Utils.params
import com.xebialabs.xlrelease.repository.sql.persistence.data.FolderVariableRow
import com.xebialabs.xlrelease.serialization.json.utils.CiSerializerHelper.serialize
import com.xebialabs.xlrelease.service.CiIdService
import com.xebialabs.xlrelease.utils.FolderId
import com.xebialabs.xlrelease.variable.VariablePersistenceHelper
import org.springframework.beans.factory.annotation.{Autowired, Qualifier}
import org.springframework.dao.DuplicateKeyException
import org.springframework.jdbc.core.{JdbcTemplate, RowMapper}
import org.springframework.stereotype.Repository

import scala.collection.mutable
import scala.jdk.CollectionConverters._
import scala.util.Try

@Repository
@IsTransactional
class FolderVariablePersistence @Autowired()(@Qualifier("xlrRepositoryJdbcTemplate") implicit val jdbcTemplate: JdbcTemplate,
                                             @Qualifier("xlrRepositorySqlDialect") implicit val dialect: Dialect,
                                             folderPersistence: FolderPersistence,
                                             repositoryAdapter: SqlRepositoryAdapter,
                                             implicit val ciIdService: CiIdService) extends PersistenceSupport {


  private val alias = "fv"
  private val select =
    s"""SELECT
       | $alias.${FOLDER_VARIABLES.CI_UID},
       | $alias.${FOLDER_VARIABLES.ID},
       | $alias.${FOLDER_VARIABLES.CI_TYPE},
       | $alias.${FOLDER_VARIABLES.KEY},
       | $alias.${FOLDER_VARIABLES.CONTENT},
       | folder.${FOLDERS.FOLDER_PATH},
       | folder.${FOLDERS.FOLDER_ID}
       | FROM ${FOLDER_VARIABLES.TABLE} $alias
       | INNER JOIN ${FOLDERS.TABLE} folder ON folder.${FOLDERS.CI_UID} = $alias.${FOLDER_VARIABLES.FOLDER_CI_UID}
    """.stripMargin


  private val STMT_EXISTS_VARIABLE_BY_ID: String = s"SELECT COUNT(*) FROM ${FOLDER_VARIABLES.TABLE} WHERE ${FOLDER_VARIABLES.ID} = :variableId"

  def exists(variableId: CiId): Boolean = sqlQuery(STMT_EXISTS_VARIABLE_BY_ID, params("variableId" -> getName(variableId)), _.getInt(1) > 0).head

  private val STMT_FIND_VARIABLE_BY_ID = select ++ s"WHERE $alias.${FOLDER_VARIABLES.ID} = :variableId"

  def findById(variableId: CiId): Option[FolderVariableRow] = {
    sqlQuery(STMT_FIND_VARIABLE_BY_ID, params("variableId" -> variableId), variableRowMapper).headOption
  }

  private val STMT_FIND_BY_CI_UID = select ++ s"WHERE $alias.${FOLDER_VARIABLES.CI_UID} = :ciUid"

  def findByCiUid(ciUid: CiUid): Option[FolderVariableRow] = {
    sqlQuery(STMT_FIND_BY_CI_UID, params("ciUid" -> ciUid), variableRowMapper).headOption
  }

  private val STMT_FIND_VARIABLE_BY_KEY =
    select ++
      s"""WHERE $alias.${FOLDER_VARIABLES.KEY} = :variableKey
         |AND $alias.${FOLDER_VARIABLES.FOLDER_CI_UID} IN (:ciUid)
         |ORDER BY ${length(s"folder.${FOLDERS.FOLDER_PATH}")} DESC
         |""".stripMargin

  def findVariableByKey(variableKey: String, folderId: CiId): Option[FolderVariableRow] = {
    val folderPath: Array[String] = FolderId(folderId).getPath
    val idToUid = folderPersistence.getUidsByIds(folderPath)
    sqlQuery(STMT_FIND_VARIABLE_BY_KEY, params("variableKey" -> variableKey, "ciUid" -> idToUid.values.asJava), variableRowMapper)
      .headOption
  }

  private val STMT_FIND_VARIABLES_BY_PARENT_ID = select ++ s"WHERE $alias.${FOLDER_VARIABLES.FOLDER_CI_UID} = :folderCiUid"

  def findByFolderId(folderId: CiId): mutable.Buffer[FolderVariableRow] = {
    sqlQuery(STMT_FIND_VARIABLES_BY_PARENT_ID, params("folderCiUid" -> folderPersistence.getUid(folderId)), variableRowMapper)
  }

  private val STMT_FIND_ALL_IN_TREE = select ++ s"WHERE $alias.${FOLDER_VARIABLES.FOLDER_CI_UID} IN (:folderCiUids)"

  def findByParentIdIncludingAncestors(parentId: String): mutable.Buffer[FolderVariableRow] = {
    val folderPath = FolderId(parentId).getPath
    val idToUid = folderPersistence.getUidsByIds(folderPath)
    if (idToUid.nonEmpty) {
      val slqResult = sqlQuery(STMT_FIND_ALL_IN_TREE, params("folderCiUids" -> idToUid.values.asJava), variableRowMapper)
        .groupBy(_.folderId.id)

      folderPath.foldLeft(Map.empty[String, FolderVariableRow]) { case (map, folderId) =>
        map ++ slqResult
          .getOrElse(folderId, mutable.Buffer.empty)
          .map(fv => fv.key -> fv)
      }.values.toBuffer
    } else {
      mutable.Buffer.empty[FolderVariableRow]
    }
  }

  private val STMT_INSERT_VARIABLE =
    s"""INSERT INTO ${FOLDER_VARIABLES.TABLE}
       | ( ${FOLDER_VARIABLES.ID}
       | , ${FOLDER_VARIABLES.CI_UID}
       | , ${FOLDER_VARIABLES.CI_TYPE}
       | , ${FOLDER_VARIABLES.FOLDER_CI_UID}
       | , ${FOLDER_VARIABLES.KEY}
       | , ${FOLDER_VARIABLES.CONTENT}
       | ) VALUES
       | ( :variableId
       | , :ciUid
       | , :ciType
       | , :folderCiUid
       | , :variableKey
       | , :content
       | )
     """.stripMargin

  def create(variable: Variable): CiUid = {
    val folderId = variable.getFolderId
    create(variable, folderPersistence.getUid(folderId))
  }

  def create(variable: Variable, folderCiUid: CiUid): CiUid = {
    variable.setId(createPersistedId[Variable]())
    VariablePersistenceHelper.fixUpFolderVariable(variable, ciIdService)
    val content = serialize(variable)
    val id = IdCreator.generateId
    
    try {
      sqlExecWithContent(STMT_INSERT_VARIABLE, params(
        "variableId" -> variable.getId,
        "ciUid" -> id,
        "ciType" -> variable.getType.toString,
        "folderCiUid" -> folderCiUid,
        "variableKey" -> variable.getKey
      ), "content" -> content)
      id
    } catch {
      case ex: DuplicateKeyException =>
        throw new IllegalArgumentException(s"Variable with ID='${variable.getId}' or key='${variable.getKey}' already exists", ex)
    }
  }

  private val STMT_UPDATE_VARIABLE =
    s"""UPDATE ${FOLDER_VARIABLES.TABLE}
       | SET
       |  ${FOLDER_VARIABLES.FOLDER_CI_UID} = :folderCiUid,
       |  ${FOLDER_VARIABLES.CONTENT} = :content
       | WHERE ${FOLDER_VARIABLES.ID} = :variableId
     """.stripMargin

  def update(variable: Variable): Unit = {
    update(variable, folderPersistence.getUid(variable.getFolderId))
  }

  def update(variable: Variable, folderCiUid: CiUid): Unit = {
    VariablePersistenceHelper.fixUpFolderVariable(variable, ciIdService)
    val content = serialize(variable)
    try {
      sqlExecWithContent(STMT_UPDATE_VARIABLE, params(
        "folderCiUid" -> folderCiUid,
        "variableId" -> variable.getId
      ), "content" -> content, checkCiUpdated(variable.getId))
    } catch {
      case ex: DuplicateKeyException =>
        throw new IllegalArgumentException(s"Variable with ID='${variable.getId}' or key='${variable.getKey}' already exists", ex)
    }
  }

  private val STMT_DELETE_VARIABLE =
    s"""DELETE FROM ${FOLDER_VARIABLES.TABLE}
       | WHERE
       | ${FOLDER_VARIABLES.ID} = :variableId
     """.stripMargin

  def delete(variableId: String): Try[Boolean] = {
    sqlExec(STMT_DELETE_VARIABLE, params("variableId" -> variableId), ps => Try(ps.execute()))
  }

  private val variableRowMapper: RowMapper[FolderVariableRow] = (rs, _) => FolderVariableRow.fromDatabase(rs)

  private def length(column: String): String = {
    dialect match {
      case MSSQLDialect(_) => s"LEN($column)"
      case _ => s"LENGTH($column)"
    }
  }
}
