package com.xebialabs.xlrelease.repository.sql.persistence

import com.xebialabs.deployit.exception.NotFoundException
import com.xebialabs.deployit.plugin.api.reflect.Type
import com.xebialabs.xlrelease.db.sql.SqlBuilder.Dialect
import com.xebialabs.xlrelease.db.sql.transaction.IsTransactional
import com.xebialabs.xlrelease.repository.sql.persistence.CiId.CiId
import com.xebialabs.xlrelease.repository.sql.persistence.Schema.CALENDAR_ENTRIES
import com.xebialabs.xlrelease.repository.sql.persistence.Utils.{params, rowMapper}
import com.xebialabs.xlrelease.repository.sql.persistence.data.CalendarEntryRow
import grizzled.slf4j.Logging
import org.springframework.jdbc.core.{JdbcTemplate, RowMapper}

import java.util.Date
import scala.jdk.CollectionConverters._

@IsTransactional
class CalendarEntryPersistence(implicit val jdbcTemplate: JdbcTemplate, implicit val dialect: Dialect) extends PersistenceSupport with Logging {

  private val STMT_INSERT_ENTRY =
    s"""INSERT INTO ${CALENDAR_ENTRIES.TABLE} (
       |  ${CALENDAR_ENTRIES.CI_UID},
       |  ${CALENDAR_ENTRIES.ENTRY_ID},
       |  ${CALENDAR_ENTRIES.LABEL},
       |  ${CALENDAR_ENTRIES.START_DATE},
       |  ${CALENDAR_ENTRIES.END_DATE},
       |  ${CALENDAR_ENTRIES.CI_TYPE},
       |  ${CALENDAR_ENTRIES.CONTENT}
       |) VALUES (:ciUid, :ciId, :label, :start, :end, :ciType, :content)
     """.stripMargin

  def insert(row: CalendarEntryRow): Unit = {
    try {
      sqlExecWithContent(STMT_INSERT_ENTRY, params(
        "ciUid" -> row.ciUid,
        "ciId" -> row.entryId,
        "label" -> row.label,
        "start" -> row.start,
        "end" -> row.end,
        "ciType" -> row.ciType.toString
      ), "content" -> row.content,
        _ => ())
    } catch {
      case ex: Exception => {
        logger.error(ex)
        throw CalendarEntryPersistenceException("Unable to insert calendar entry, check logs for more information")
      }
    }
  }

  private val STMT_UPDATE_ENTRY =
    s"""UPDATE ${CALENDAR_ENTRIES.TABLE}
       |  SET
       |    ${CALENDAR_ENTRIES.LABEL} = :label,
       |    ${CALENDAR_ENTRIES.START_DATE} = :start,
       |    ${CALENDAR_ENTRIES.END_DATE} = :end,
       |    ${CALENDAR_ENTRIES.CI_TYPE} = :ciType,
       |    ${CALENDAR_ENTRIES.CONTENT} = :content
       |WHERE ${CALENDAR_ENTRIES.ENTRY_ID} = :ciId
     """.stripMargin

  def update(updated: CalendarEntryRow): Unit = {
    try {
      sqlExecWithContent(STMT_UPDATE_ENTRY, params(
        "ciId" -> updated.entryId,
        "label" -> updated.label,
        "start" -> updated.start,
        "end" -> updated.end,
        "ciType" -> updated.ciType.toString
      ), "content" -> updated.content,
        checkCiUpdated(updated.entryId)
      )
    } catch {
      case ex: Exception => {
        logger.error(ex)
        throw CalendarEntryPersistenceException("Unable to Update calendar entry, check logs for more information")
      }
    }
  }

  private val STMT_DELETE_ENTRY =
    s"""| DELETE FROM ${CALENDAR_ENTRIES.TABLE}
        | WHERE
        |   ${CALENDAR_ENTRIES.ENTRY_ID} = :ciId
      """.stripMargin

  def deleteById(ciId: CiId): Unit = {
    sqlUpdate(STMT_DELETE_ENTRY, params("ciId" -> ciId),
      checkCiUpdated(ciId)
    )
  }

  private val SELECT_ENTRY =
    s"""SELECT
       |  ce.${CALENDAR_ENTRIES.CI_UID},
       |  ce.${CALENDAR_ENTRIES.CI_TYPE},
       |  ce.${CALENDAR_ENTRIES.ENTRY_ID},
       |  ce.${CALENDAR_ENTRIES.LABEL},
       |  ce.${CALENDAR_ENTRIES.START_DATE},
       |  ce.${CALENDAR_ENTRIES.END_DATE},
       |  ce.${CALENDAR_ENTRIES.CONTENT}
       |FROM ${CALENDAR_ENTRIES.TABLE} ce
     """.stripMargin

  private val STMT_FIND_BY_ID =
    s"""$SELECT_ENTRY
       |WHERE ce.${CALENDAR_ENTRIES.ENTRY_ID} = :ciId
     """.stripMargin

  def findById(ciId: CiId): CalendarEntryRow = {
    sqlQuery(STMT_FIND_BY_ID, params("ciId" -> ciId), calendarEntryMapper).headOption.getOrElse {
      throw new NotFoundException(s"Calendar row [$ciId] not found")
    }
  }

  private val STMT_ENTRY_EXISTS: String = s"SELECT COUNT(*) FROM ${CALENDAR_ENTRIES.TABLE} WHERE ${CALENDAR_ENTRIES.ENTRY_ID} = :ciId"

  def exists(ciId: CiId): Boolean = {
    sqlQuery(STMT_ENTRY_EXISTS, params("ciId" -> ciId), _.getInt(1) > 0).head
  }

  private val ORDER_BY_START_END_LABEL =
    s"""ORDER BY
       |   ce.${CALENDAR_ENTRIES.START_DATE} ASC,
       |   ce.${CALENDAR_ENTRIES.END_DATE} ASC,
       |   ce.${CALENDAR_ENTRIES.LABEL} ASC,
       |   ce.${CALENDAR_ENTRIES.ENTRY_ID} ASC
     """.stripMargin

  private val COND_BY_TYPES =
    s"""ce.${CALENDAR_ENTRIES.CI_TYPE} IN (:ciTypes)""".stripMargin

  private val COND_BY_RANGE =
    s"""ce.${CALENDAR_ENTRIES.START_DATE} <= :to AND ce.${CALENDAR_ENTRIES.END_DATE} >= :from""".stripMargin

  private val COND_BY_END_DATE =
    s"""ce.${CALENDAR_ENTRIES.END_DATE} > :from""".stripMargin

  private val STMT_SEARCH_ENTRIES_BY_TYPE =
    s"""$SELECT_ENTRY
       |WHERE $COND_BY_TYPES
       |$ORDER_BY_START_END_LABEL
     """.stripMargin

  private val STMT_SEARCH_ENTRIES_IN_RANGE =
    s"""$SELECT_ENTRY
       |WHERE
       |  $COND_BY_TYPES AND
       |  $COND_BY_RANGE
       |$ORDER_BY_START_END_LABEL
     """.stripMargin

  private val STMT_SEARCH_ENTRIES_BY_END_DATE =
    s"""$SELECT_ENTRY
       |WHERE
       |  $COND_BY_TYPES AND
       |  $COND_BY_END_DATE
       |$ORDER_BY_START_END_LABEL
     """.stripMargin

  def search(ciTypes: Seq[Type], from: Date = null, to: Date = null): Seq[CalendarEntryRow] = {
    val ciTypesList = ciTypes.map(_.toString).asJava
    if (from == null && to == null) {
      sqlQuery(STMT_SEARCH_ENTRIES_BY_TYPE, params("ciTypes" -> ciTypesList), calendarEntryMapper).toSeq
    } else if (to == null) {
      sqlQuery(STMT_SEARCH_ENTRIES_BY_END_DATE, params("ciTypes" -> ciTypesList, "from" -> from), calendarEntryMapper).toSeq
    } else {
      sqlQuery(STMT_SEARCH_ENTRIES_IN_RANGE, params("ciTypes" -> ciTypesList, "from" -> from, "to" -> to), calendarEntryMapper).toSeq
    }
  }

  private val STMT_SELECT_ENTRIES =
    s"""SELECT ce.${CALENDAR_ENTRIES.CI_UID}
       |      FROM ${CALENDAR_ENTRIES.TABLE} ce""".stripMargin

  private val STMT_EXISTS_BY_TYPE =
    s"""SELECT
       |  CASE
       |    WHEN EXISTS (
       |      $STMT_SELECT_ENTRIES
       |      WHERE $COND_BY_TYPES
       |    ) THEN  1
       |    ELSE    0
       |  END
       |FROM ${CALENDAR_ENTRIES.TABLE}
     """.stripMargin

  private val STMT_EXISTS_BY_TYPE_IN_RANGE =
    s"""SELECT
       |  CASE
       |    WHEN EXISTS (
       |      $STMT_SELECT_ENTRIES
       |      WHERE $COND_BY_TYPES AND
       |            $COND_BY_RANGE
       |    ) THEN  1
       |    ELSE    0
       |  END
       |FROM ${CALENDAR_ENTRIES.TABLE}
     """.stripMargin

  def existsByTypeInRange(ciTypes: Seq[Type], from: Date, to: Date): Boolean = {
    val ciTypesList = ciTypes.map(_.toString).asJava
    val rows = if (from == null && to == null) {
      sqlQuery(STMT_EXISTS_BY_TYPE, params("ciTypes" -> ciTypesList), _.getInt(1) == 1)
    } else {
      sqlQuery(STMT_EXISTS_BY_TYPE_IN_RANGE, params("ciTypes" -> ciTypesList, "from" -> from, "to" -> to), _.getInt(1) == 1)
    }
    rows.headOption.getOrElse(false)
  }

  private val calendarEntryMapper: RowMapper[CalendarEntryRow] = rowMapper { rs =>
    val contentStream = rs.getBinaryStream(CALENDAR_ENTRIES.CONTENT)
    try {
      val content = decompress(contentStream)
      CalendarEntryRow(
        ciUid = rs.getString(CALENDAR_ENTRIES.CI_UID),
        entryId = rs.getString(CALENDAR_ENTRIES.ENTRY_ID),
        ciType = Type.valueOf(rs.getString(CALENDAR_ENTRIES.CI_TYPE)),
        label = rs.getString(CALENDAR_ENTRIES.LABEL),
        start = rs.getTimestamp(CALENDAR_ENTRIES.START_DATE),
        end = rs.getTimestamp(CALENDAR_ENTRIES.END_DATE),
        content = content
      )
    } finally {
      contentStream.close()
    }
  }

  case class CalendarEntryPersistenceException(message: String) extends Exception(message)

}
