package com.xebialabs.xlrelease.repository.sql

import com.xebialabs.xlrelease.db.sql.transaction.IsTransactional
import com.xebialabs.xlrelease.domain.id.CiUid
import com.xebialabs.xlrelease.repository.Ids.{getParentId, isFolderId, isReleaseId}
import com.xebialabs.xlrelease.repository.SecuredCis
import com.xebialabs.xlrelease.repository.sql.persistence.{FolderPersistence, ReleasePersistence}
import com.xebialabs.xlrelease.security.SecuredCi
import io.micrometer.core.annotation.Timed

@IsTransactional
class SqlSecuredCis(folderPersistence: FolderPersistence,
                    releasePersistence: ReleasePersistence) extends SecuredCis {

  @Timed
  override def getEffectiveSecuredCi(containerId: String): SecuredCi = containerId match {
    case id if isFolderId(id) => folderPersistence.getEffectiveSecuredCi(id)
    case id if isReleaseId(id) => releasePersistence.getEffectiveSecuredCi(id)
    case id => getSecuredCi(id)
  }

  @Timed
  override def getSecuredCi(containerId: String): SecuredCi = containerId match {
    case id if isFolderId(id) => folderPersistence.getSecuredCi(containerId)
    case id if isReleaseId(id) => asSecuredCi(id, releasePersistence.findUidByReleaseId(id))
    case _ => throw new IllegalArgumentException(s"Unable to get UID for unsupported ID format [$containerId]")
  }

  private def asSecuredCi(containerId: String, uid: Option[CiUid]) =
    new SecuredCi(containerId, uid.getOrElse(throw new IllegalArgumentException(s"Unable to find release by ID [$containerId]")).toString)

  @Timed
  override def setAsEffectiveSecuredCi(containerId: String, isEffective: Boolean): Unit = {
    if (isFolderId(containerId)) {
      val oldSecurityUid = folderPersistence.getEffectiveSecuredCi(containerId).getSecurityUid
      val newFolderData = if (isEffective) {
        folderPersistence.setAsEffectiveSecuredCi(containerId)
      } else {
        folderPersistence.inheritEffectiveSecuredCi(containerId)
      }
      releasePersistence.replaceSecurityUid(newFolderData.uid, oldSecurityUid, newFolderData.securityUid)

    } else {
      if (isEffective) {
        releasePersistence.setAsEffectiveSecuredCi(containerId)
      } else {
        val effectiveFolder = folderPersistence.getEffectiveSecuredCi(getParentId(containerId))
        releasePersistence.inheritSecurityFromFolder(containerId, effectiveFolder.getSecurityUid)
      }
    }
  }

}
