package com.xebialabs.xlrelease.upgrade.db

import com.google.common.base.Strings
import com.xebialabs.deployit.server.api.upgrade.{Upgrade, Version}
import com.xebialabs.xlrelease.config.XlrConfig
import com.xebialabs.xlrelease.db.sql.SqlBuilder.Dialect
import com.xebialabs.xlrelease.domain.distributed.events.EvictAllEvent
import com.xebialabs.xlrelease.repository.sql.persistence.PersistenceSupport
import com.xebialabs.xlrelease.repository.sql.persistence.Schema.{FOLDERS, RELEASES}
import com.xebialabs.xlrelease.repository.sql.persistence.Utils._
import com.xebialabs.xlrelease.repository.{Ids, TeamRepository}
import com.xebialabs.xlrelease.security.SecuredCi
import com.xebialabs.xlrelease.security.sql.SecurityCacheEviction
import com.xebialabs.xlrelease.security.sql.db.SecuritySchema.ROLES
import com.xebialabs.xlrelease.upgrade.Components.XL_RELEASE_COMPONENT
import com.xebialabs.xlrelease.upgrade.UpgradeSupport.{BatchSupport, TransactionSupport}
import grizzled.slf4j.Logging
import org.springframework.beans.factory.annotation.{Autowired, Qualifier}
import org.springframework.jdbc.core.JdbcTemplate
import org.springframework.stereotype.Component
import org.springframework.transaction.support.TransactionTemplate

import java.util.Optional
import scala.collection.mutable.ListBuffer
import scala.collection.parallel.CollectionConverters._
import scala.jdk.CollectionConverters._


@Component
class XLRelease980TeamsDeduplication @Autowired()(@Qualifier("xlrRepositoryJdbcTemplate") val jdbcTemplate: JdbcTemplate,
                                                  @Qualifier("xlrRepositorySqlDialect") implicit val dialect: Dialect,
                                                  @Qualifier("xlrRepositoryTransactionTemplate") val transactionTemplate: TransactionTemplate,
                                                  teamRepository: TeamRepository,
                                                  xlrConfig: XlrConfig,
                                                  securityCacheEviction: Optional[SecurityCacheEviction])
  extends Upgrade
    with PersistenceSupport
    with Logging
    with BatchSupport
    with TransactionSupport {

  private val batchSize: Int = Math.max(1, (xlrConfig.database.maxPoolSize - 2) / 2)

  override def upgradeVersion(): Version = Version.valueOf(XL_RELEASE_COMPONENT, "9.8.0#2")

  override def doUpgrade(): Boolean = {
    logger.info("Starting team deduplication on folders and templates")

    val duplicatingTeams = fetchCisWithDups().map {
      case (i, f, r) if Strings.isNullOrEmpty(f) => (i, r)
      case (i, f, r) if Strings.isNullOrEmpty(r) => (i, f)
    }

    doInBatch(duplicatingTeams, batchSize = batchSize) { batch =>
      val items = if (xlrConfig.upgrader.parallel) {
        batch.items.par
      } else {
        batch.items
      }
      items.iterator.map(tuple => new SecuredCi(tuple._2, tuple._1)).foreach { securedCi =>
        doInTransaction {
          deduplicate(securedCi)
        }
      }
    }
    logger.info("Finished team deduplication on folders and templates")
    securityCacheEviction.map(_.onEvictAll(EvictAllEvent()))
    true
  }

  def deduplicate(securedCi: SecuredCi): Unit = {
    val teams = teamRepository.getTeams(securedCi).asScala
    val renamedTeams = ListBuffer.empty[String]
    teams
      .groupBy(_.getTeamName.toUpperCase)
      .flatMap { case (_, list) => list.zipWithIndex }
      .foreach {
        case (team, i) if (i > 0) =>
          var offset = i;

          def newName = s"${team.getTeamName} ($offset)"

          while ((teams.map(_.getTeamName) ++ renamedTeams).exists(_.toLowerCase == newName.toLowerCase)) {
            offset = offset + 1
          }
          updateTeamName(Ids.getName(team.getId), newName)
          logger.warn(s"Team name for $team is updated to '$newName'")
          renamedTeams.append(newName)
        case _ =>
      }
  }

  private val STMT_GET_ALL_CIS_WITH_DUPLICATED_ROLES =
    s"""|SELECT DISTINCT(dups.${ROLES.ciId}) CIUID, f.${FOLDERS.FOLDER_ID} FOLDERID, t.${RELEASES.RELEASE_ID} RELEASEID FROM
        |        (SELECT
        |        COUNT(lower(r.${ROLES.name})) CNT,
        |        r.${ROLES.ciId}
        |        FROM ${ROLES.TABLE} r
        |        GROUP BY r.${ROLES.ciId}, lower(r.${ROLES.name})) dups
        |         LEFT JOIN ${FOLDERS.TABLE} f on f.${FOLDERS.CI_UID} = dups.${ROLES.ciId}
        |         LEFT JOIN ${RELEASES.TABLE} t on t.${RELEASES.CI_UID} = dups.${ROLES.ciId}
        |WHERE dups.CNT > 1
        |""".stripMargin

  private def fetchCisWithDups(): List[(Long, String, String)] = {
    sqlQuery(STMT_GET_ALL_CIS_WITH_DUPLICATED_ROLES,
      params(),
      rs => (rs.getLong("CIUID"), rs.getString("FOLDERID"), rs.getString("RELEASEID"))
    ).toList
  }

  private val STMT_UPDATE_TEAM_NAME =
    s"""UPDATE ${ROLES.TABLE} SET ${ROLES.name} = :name
       | WHERE ${ROLES.id} = :id
       |""".stripMargin

  private def updateTeamName(id: String, name: String): Unit = {
    sqlUpdate(STMT_UPDATE_TEAM_NAME,
      params(
        "id" -> id,
        "name" -> name
      ),
      identity
    )
  }
}
