package com.xebialabs.xlrelease.repository.sql.persistence.data

import com.xebialabs.xlrelease.domain.folder.Folder
import com.xebialabs.xlrelease.domain.id.CiUid
import com.xebialabs.xlrelease.repository.Ids.ROOT_FOLDER_ID
import com.xebialabs.xlrelease.repository.sql.persistence.Token
import com.xebialabs.xlrelease.utils.FolderId
import com.xebialabs.xlrelease.utils.Tree.Node

import scala.jdk.CollectionConverters._

sealed trait FolderRow {
  def uid: CiUid

  def parentUid: CiUid

  def folderId: FolderId

  def securityUid: CiUid

  def name: String

  def depth: Int

  def hasSecurity: Boolean = securityUid == uid

  def inheritsSecurity: Boolean = !hasSecurity
}

object FolderRow {

  case object Root extends FolderRow {
    val uid: CiUid = 1L
    val parentUid: CiUid = uid
    val folderId: FolderId = FolderId(ROOT_FOLDER_ID)
    val securityUid: CiUid = uid
    val name: String = "/"
    val depth: Int = 0
  }

  case class FolderEntry(uid: CiUid,
                         token: Token,
                         securityUid: CiUid,
                         folderId: FolderId,
                         name: String,
                         parentUid: CiUid,
                         depth: Int) extends FolderRow

  implicit class FolderDataOps(val data: FolderRow) extends AnyVal {
    def toOption: Option[FolderEntry] = fold(Option.empty[FolderEntry])(Some(_))

    def asFolder: Folder = {
      val folder = new Folder()
      folder.setId(data.folderId.absolute)
      folder.set$internalId(data.uid)
      folder.set$securedCi(data.securityUid)
      data.fold(())(v => folder.set$token(v.token.toString))
      folder.setTitle(data.name)
      folder
    }

    def fold[A](root: => A)(entry: FolderEntry => A): A = data match {
      case Root => root
      case e: FolderEntry => entry(e)
    }

    def setFolderId(newFolderId: FolderId): FolderRow = data match {
      case Root => Root
      case entry: FolderEntry => entry.copy(folderId = newFolderId)
    }
  }

  implicit class NodeFolderDataOps(val node: Node[FolderRow]) extends AnyVal {
    def toFolder: Folder = node.unsafeMap { case Node(value, children) =>
      val folders: List[Folder] = children.map(_.toFolder)
      val folder = value.asFolder
      folder.setChildren(folders.toSet.asJava)
      folder
    }.value
  }

}
