package com.xebialabs.xlrelease.repository.sql.persistence

import com.xebialabs.xlrelease.db.sql.LimitOffset
import com.xebialabs.xlrelease.db.sql.SqlBuilder.Dialect
import com.xebialabs.xlrelease.db.sql.transaction.{IsReadOnly, IsTransactional}
import com.xebialabs.xlrelease.domain.Phase
import com.xebialabs.xlrelease.domain.id.{CiUid, IdCreator}
import com.xebialabs.xlrelease.domain.status.PhaseStatus
import com.xebialabs.xlrelease.repository.Ids
import com.xebialabs.xlrelease.repository.Ids.getFolderlessId
import com.xebialabs.xlrelease.repository.sql.SqlRepository
import com.xebialabs.xlrelease.repository.sql.persistence.CiId.CiId
import com.xebialabs.xlrelease.repository.sql.persistence.Schema.PHASES
import com.xebialabs.xlrelease.repository.sql.persistence.Utils.{RichStringAsTruncatable, params}
import grizzled.slf4j.Logging
import org.springframework.jdbc.core.JdbcTemplate

@IsTransactional
class PhasePersistence(implicit val jdbcTemplate: JdbcTemplate,
                       implicit val dialect: Dialect)
  extends SqlRepository
    with PersistenceSupport
    with LimitOffset
    with Logging {

  private val STMT_INSERT_PHASE =
    s"""| INSERT INTO ${PHASES.TABLE}
        |   ( ${PHASES.RELEASE_UID}
        |   , ${PHASES.CI_UID}
        |   , ${PHASES.PHASE_ID}
        |   , ${PHASES.PHASE_TITLE}
        |   , ${PHASES.STATUS}
        |   , ${PHASES.COLOR}
        |   , ${PHASES.START_DATE}
        |   , ${PHASES.END_DATE}
        |   )
        | VALUES
        |   ( :releaseUid
        |   , :ciUid
        |   , :phaseId
        |   , :title
        |   , :status
        |   , :color
        |   , :startDate
        |   , :endDate
        |   )
      """.stripMargin

  def insert(phase: Phase, releaseUid: CiUid): CiUid = {
    val id = IdCreator.generateId
    sqlInsert(STMT_INSERT_PHASE, params("releaseUid" -> releaseUid, "ciUid" -> id) ++ commonPhaseParams(phase))
    id
  }

  def batchInsert(phases: Set[Phase], releaseUid: CiUid): Unit = {
    sqlBatch(STMT_INSERT_PHASE, phases.collect {
      case phase: Phase => params("releaseUid" -> releaseUid, "ciUid" -> IdCreator.generateId) ++ commonPhaseParams(phase)
    })
  }

  private val STMT_UPDATE_PHASE =
    s"""| UPDATE ${PHASES.TABLE}
        |  SET
        |   ${PHASES.PHASE_TITLE} = :title,
        |   ${PHASES.STATUS} = :status,
        |   ${PHASES.COLOR} = :color,
        |   ${PHASES.START_DATE} = :startDate,
        |   ${PHASES.END_DATE} = :endDate
        | WHERE
        |   ${PHASES.PHASE_ID} = :phaseId
      """.stripMargin

  def update(phase: Phase): Boolean = {
    sqlUpdate(STMT_UPDATE_PHASE, commonPhaseParams(phase), _ == 1)
  }

  def batchUpdate(phases: Set[Phase]): Int = {
    sqlBatch(STMT_UPDATE_PHASE, phases.collect {
      case phase: Phase => commonPhaseParams(phase)
    }).sum
  }

  private val STMT_DELETE_PHASE =
    s"""|DELETE FROM ${PHASES.TABLE}
        | WHERE
        |   ${PHASES.PHASE_ID} = :phaseId
      """.stripMargin

  def delete(phase: Phase): Unit = {
    sqlUpdate(STMT_DELETE_PHASE, commonPhaseParams(phase), _ => ())
  }

  private val STMT_DELETE_PHASES_BY_RELEASE_UID =
    s"""|DELETE FROM ${PHASES.TABLE}
        | WHERE
        |   ${PHASES.RELEASE_UID} = :releaseUid
      """.stripMargin

  def deletePhasesByReleaseUid(releaseUid: CiUid): Unit = {
    sqlUpdate(STMT_DELETE_PHASES_BY_RELEASE_UID, params("releaseUid" -> releaseUid), _ => ())
  }

  private val STMT_GET_TITLE_BY_ID =
    s"""|SELECT ${PHASES.PHASE_TITLE}
        | FROM ${PHASES.TABLE}
        | WHERE
        |   ${PHASES.PHASE_ID} = :phaseId
        |""".stripMargin

  @IsReadOnly
  def getTitle(phaseId: CiId): Option[String] = {
    sqlQuery(STMT_GET_TITLE_BY_ID, params(
      "phaseId" -> getFolderlessId(phaseId)
    ), _.getString(PHASES.PHASE_TITLE)).headOption
  }

  private val STMT_GET_STATUS =
    s"""|SELECT ${PHASES.STATUS} FROM ${PHASES.TABLE}
        | WHERE
        |   ${PHASES.PHASE_ID} = :phaseId
        |""".stripMargin

  @IsReadOnly
  def getStatus(phaseId: String): Option[PhaseStatus] = {
    sqlQuery(STMT_GET_STATUS,
      params("phaseId" -> Ids.getFolderlessId(phaseId)),
      rs => PhaseStatus.valueOf(rs.getString(PHASES.STATUS).toUpperCase)).headOption
  }

  private[persistence] def commonPhaseParams(phase: Phase): Map[String, Any] = {
    val commonParams = params(
      "phaseId" -> getFolderlessId(phase.getId),
      "title" -> phase.getTitle.truncate(Schema.COLUMN_LENGTH_TITLE),
      "status" -> phase.getStatus.value(),
      "color" -> phase.getColor(),
      "startDate" -> phase.getStartOrScheduledDate(),
      "endDate" -> phase.getEndOrDueDate(),
    )
    commonParams
  }
}
