package com.xebialabs.xlrelease.repository.sql

import com.xebialabs.deployit.checks.Checks.checkArgument
import com.xebialabs.deployit.plugin.api.reflect.Type
import com.xebialabs.deployit.security.permission.Permission
import com.xebialabs.deployit.security.{PermissionEditor, Role}
import com.xebialabs.xlrelease.domain.Team
import com.xebialabs.xlrelease.repository.Ids._
import com.xebialabs.xlrelease.repository.TeamRepository._
import com.xebialabs.xlrelease.repository._
import com.xebialabs.xlrelease.repository.sql.persistence.Schema.FOLDERS
import com.xebialabs.xlrelease.repository.sql.persistence._
import com.xebialabs.xlrelease.security.SecuredCi
import com.xebialabs.xlrelease.security.sql.SqlRoleService
import grizzled.slf4j.Logging
import io.micrometer.core.annotation.Timed
import org.springframework.jdbc.core.namedparam.NamedParameterJdbcTemplate
import org.springframework.jdbc.core.{JdbcTemplate, RowMapper}
import org.springframework.util.StringUtils

import java.util
import scala.jdk.CollectionConverters._


class SqlTeamRepository(val jdbcTemplate: JdbcTemplate,
                        val permissionEditor: PermissionEditor,
                        val roleService: SqlRoleService,
                        val securedCis: SecuredCis) extends TeamRepository with Logging {

  private val namedParameterJdbcTemplate = new NamedParameterJdbcTemplate(jdbcTemplate)

  @Timed
  override def create(containerId: String, team: Team): Team = {
    removeMissingRoles(team)

    val role = toPlatformRole(team)
    val securityUid = securedCis.getSecuredCi(containerId).getSecurityUid

    roleService.create(securityUid, role)
    updatePermissions(securityUid, role -> team.getPermissions)
    updateInheritanceFlag(containerId, Seq(team))
    team
  }

  @Timed
  override def update(team: Team): Team = {
    val containerId = getParentId(team.getId)
    removeMissingRoles(team)

    val role = toPlatformRole(team)
    val securityUid = securedCis.getSecuredCi(containerId).getSecurityUid

    roleService.update(securityUid, role)
    updatePermissions(securityUid, role -> team.getPermissions)
    team
  }

  @Timed
  override def getTeams(containerId: String): util.List[Team] = getTeams(securedCis.getSecuredCi(containerId))

  @Timed
  override def getTeams(ci: SecuredCi): util.List[Team] = {
    val roles = roleService.readRoleAssignments(ci.getSecurityUid).asScala.toList

    (roles match {
      case Nil => Seq.empty
      case _ =>
        val permissionsByRole = permissionEditor.readPermissions(ci.getSecurityUid)
        roles.map(role => fromPlatformRole(ci.getId, role, permissionsByRole.getOrDefault(role, new util.HashSet[Permission]()).asScala.toSeq))
    }).asJavaMutable()
  }


  private val STMT_GET_TEAM_PERMISSION_ROWS =
    s"""
       |SELECT
       |    teams.ID        teamId,
       |    teams.NAME      teamName,
       |    team_permissions.PERMISSION_NAME permissionName
       |FROM
       |    XL_ROLES teams
       |    JOIN XL_ROLE_PERMISSIONS team_permissions on teams.ID = team_permissions.ROLE_ID
       |WHERE teams.CI_ID = :${FOLDERS.CI_UID}
       |ORDER BY teamName, teamId
       |""".stripMargin

  override def getTeamPermissionRows(container: SecuredCi): Seq[TeamPermissionRow] = {
    val params = Map(FOLDERS.CI_UID -> container.getSecurityUid.toLong).asJava
    val mapper: RowMapper[TeamPermissionRow] = (rs, _) => {
      val teamId = rs.getString("teamId")
      val teamName = rs.getString("teamName")
      val permissionName = rs.getString("permissionName")
      TeamPermissionRow(TeamRow(teamId, teamName), permissionName)
    }
    val rows = namedParameterJdbcTemplate.query[TeamPermissionRow](STMT_GET_TEAM_PERMISSION_ROWS, params, mapper).asScala.toSeq
    rows
  }


  private val STMT_GET_PRINCIPAL_TEAM_MEMBER_VIEW_ROWS =
    s"""
       |SELECT teams.ID                              teamId,
       |       teams.NAME                            teamName,
       |       team_principal_members.PRINCIPAL_NAME memberName,
       |       team_principal_profile.FULL_NAME      fullName
       |FROM XL_ROLES teams
       |         LEFT JOIN XL_ROLE_PRINCIPALS team_principal_members on teams.ID = team_principal_members.ROLE_ID
       |         LEFT JOIN XLR_USER_PROFILES team_principal_profile on team_principal_profile.USERNAME = team_principal_members.PRINCIPAL_NAME
       |WHERE teams.CI_ID = :${FOLDERS.CI_UID}
       |ORDER BY teamName, teamId
       |""".stripMargin
  private val STMT_GET_ROLE_TEAM_MEMBER_VIEW_ROWS =
    s"""
       |SELECT teams.ID        teamId,
       |       teams.NAME      teamName,
       |       team_roles.NAME memberName,
       |       team_roles.ID   roleId
       |FROM XL_ROLES teams
       |         LEFT JOIN XL_ROLE_ROLES team_role_members on teams.ID = team_role_members.ROLE_ID
       |         LEFT JOIN XL_ROLES team_roles on team_role_members.MEMBER_ROLE_ID = team_roles.ID
       |WHERE teams.CI_ID = :${FOLDERS.CI_UID}
       |ORDER BY teamName, teamId
       |""".stripMargin

  override def getMembersPerTeam(container: SecuredCi): Seq[TeamMemberRow] = {
    val params = Map(FOLDERS.CI_UID -> container.getSecurityUid.toLong).asJava
    val principalMapper: RowMapper[TeamMemberRow] = (rs, _) => {
      val teamId = rs.getString("teamId")
      val teamName = rs.getString("teamName")
      val memberName = rs.getString("memberName")
      val fullName = rs.getString("fullName")
      val memberRow = if (memberName != null) {
        Some(MemberRow(memberName, fullName, null))
      } else {
        None
      }
      TeamMemberRow(TeamRow(teamId, teamName), memberRow)
    }
    val principalRows = namedParameterJdbcTemplate.query[TeamMemberRow](STMT_GET_PRINCIPAL_TEAM_MEMBER_VIEW_ROWS, params, principalMapper).asScala.toSeq

    val roleMapper: RowMapper[TeamMemberRow] = (rs, _) => {
      val teamId = rs.getString("teamId")
      val teamName = rs.getString("teamName")
      val memberName = rs.getString("memberName")
      val roleId = rs.getString("roleId")
      val memberRow = if (roleId != null && memberName != null) {
        Some(MemberRow(memberName, null, roleId))
      } else {
        None
      }
      TeamMemberRow(TeamRow(teamId, teamName), memberRow)
    }
    val roleRows = namedParameterJdbcTemplate.query[TeamMemberRow](STMT_GET_ROLE_TEAM_MEMBER_VIEW_ROWS, params, roleMapper).asScala.toSeq

    principalRows ++ roleRows
  }

  @Timed
  override def delete(teamId: String): Unit = {
    val containerId = getParentId(teamId)
    val securityUid = securedCis.getSecuredCi(containerId).getSecurityUid
    roleService.deleteRoles(securityUid, toPlatformId(teamId))
    updateInheritanceFlag(containerId, getTeams(containerId).asScala.toSeq)
  }

  @Timed
  override def deleteTeamsFromPlatform(containerId: String): Unit = deleteTeamsFromPlatform(securedCis.getSecuredCi(containerId))

  @Timed
  override def deleteTeamsFromPlatform(container: SecuredCi): Unit = {
    val roles = roleService.getRoles(container.getSecurityUid).asScala.toSeq
    roleService.deleteRoles(container.getSecurityUid, roles.map(_.getId): _*)
    if (isFolderId(container.getId)) {
      securedCis.setAsEffectiveSecuredCi(container.getId, isEffective = false)
    }
  }

  @Timed
  override def saveTeamsToPlatform(containerId: String, teams: util.List[Team]): util.List[Team] =
    saveTeamsToPlatform(containerId, teams.asScala.toSeq).asJavaMutable()

  private def saveTeamsToPlatform(containerId: String, teams: Seq[Team]): Seq[Team] = {
    removeMissingRoles(teams: _*)

    val securityUid = securedCis.getSecuredCi(containerId).getSecurityUid

    val roles = teams.map(team => toPlatformRole(team) -> team.getPermissions)
    roleService.writeRoleAssignments(securityUid, roles.map(_._1).asJava)
    updatePermissions(securityUid, roles: _*)

    updateInheritanceFlag(containerId, teams)

    teams
  }

  private def updatePermissions(securityUid: String, rolesWithPermissions: (Role, util.List[String])*): Unit = {
    val permissions = permissionEditor.readPermissions(securityUid)
    rolesWithPermissions.foreach {
      case (role, rolePermissions) => permissions.put(role, rolePermissions.asScala.map(Permission.find).toSet.asJava)
    }
    permissionEditor.editPermissions(securityUid, permissions)
  }

  private def updateInheritanceFlag(containerId: String, teams: Seq[Team]): Unit = {
    securedCis.setAsEffectiveSecuredCi(containerId, teams.nonEmpty)
  }

  private def removeMissingRoles(teams: Team*): Unit = {
    lazy val availableRoles = roleService.getRoles().asScala

    teams.foreach { team =>
      val nonExistingRoles = team.getRoles.asScala.filter(roleName => !availableRoles.exists(_.getName == roleName))
      if (nonExistingRoles.nonEmpty) {
        logger.warn(s"The following role(s) do not exist: $nonExistingRoles. They were removed from the team ${team.getTeamName} [${team.getId}].")
        team.getRoles.removeAll(nonExistingRoles.asJava)
      }
    }
  }

  private def fromPlatformRole(containerId: String, role: Role, permissions: Seq[Permission]): Team = {
    val team: Team = Type.valueOf(classOf[Team]).getDescriptor.newInstance(fromPlatformId(containerId, role.getId))
    team.setRoles(role.getRoles)
    team.setMembers(role.getPrincipals)
    team.setTeamName(role.getName)
    val permissionNames = permissions.collect { case a: Permission => a.getPermissionName }
    team.setPermissions(permissionNames.asJavaMutable())
    team
  }

  private def toPlatformRole(team: Team): Role = {
    checkArgument(StringUtils.hasText(team.getTeamName), s"Team name is missing in team ${team.getId}")
    val role = new Role(toPlatformId(team.getId), team.getTeamName)
    role.setRoles(team.getRoles)
    role.setPrincipals(team.getMembers)
    role
  }

  private def toPlatformId(id: String): String = Ids.getName(id)

  private def fromPlatformId(containerId: String, id: String): String = containerId + "/" + id
}

