package com.xebialabs.xlrelease.repository.sql

import com.github.benmanes.caffeine.cache.Cache
import com.xebialabs.deployit.plugin.api.reflect.Type
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem
import com.xebialabs.deployit.plumbing.serialization.ResolutionContext
import com.xebialabs.xlrelease.domain.BaseConfiguration
import com.xebialabs.xlrelease.domain.distributed.events.{EvictAllConfigurationCacheEvent, EvictConfigurationCacheEvent}
import com.xebialabs.xlrelease.domain.id.CiUid
import com.xebialabs.xlrelease.domain.status.ReleaseStatus
import com.xebialabs.xlrelease.events.EventListener
import com.xebialabs.xlrelease.repository.query.ReleaseBasicData
import com.xebialabs.xlrelease.repository.sql.ConfigurationCacheConstants._
import com.xebialabs.xlrelease.repository.sql.persistence.configuration.ReleaseConfigurationReferencePersistence
import com.xebialabs.xlrelease.repository.{CiCloneHelper, ConfigurationRepository, PersistenceInterceptor}
import com.xebialabs.xlrelease.service.BroadcastService
import com.xebialabs.xlrelease.support.cache.caffeine.spring.XlrCaffeineCacheManager
import com.xebialabs.xlrelease.utils.CiHelper

import java.util
import java.util.{Optional, List => JList}
import scala.jdk.CollectionConverters._


@EventListener
class CachingSqlConfigurationRepository(delegate: ConfigurationRepository,
                                        cacheManager: XlrCaffeineCacheManager,
                                        releaseConfigurationReferencePersistence: ReleaseConfigurationReferencePersistence,
                                        broadcastService: BroadcastService) extends ConfigurationRepository {

  override def registerPersistenceInterceptor(persistenceInterceptor: PersistenceInterceptor[BaseConfiguration]): Unit =
    delegate.registerPersistenceInterceptor(persistenceInterceptor)

  private def configurationsCache[T <: BaseConfiguration]: Cache[String, T] = {
    cacheManager.getOrCreateCache(CONFIGURATION_BY_ID_CACHE)
  }

  private def referencedConfigurationsCache[T <: BaseConfiguration]: Cache[String, Set[String]] = {
    cacheManager.getOrCreateCache(REFERENCED_CONFIGURATIONS_CACHE)
  }

  private def putToCache[T <: BaseConfiguration](configuration: T): T = {
    configurationsCache.put(configuration.getId, configuration)

    val references = collectReferences(configuration, Set.empty)
    references.foreach(ref => {
      val refUsage = Option(referencedConfigurationsCache.getIfPresent(ref.getId))
        .map(refs => refs + configuration.getId)
        .getOrElse(Set(configuration.getId))

      referencedConfigurationsCache.put(ref.getId, refUsage)
    })

    configuration
  }

  private def collectReferences[T <: BaseConfiguration](configuration: ConfigurationItem, seen: Set[ConfigurationItem]): Set[ConfigurationItem] = {
    val references = CiHelper.getExternalReferences(configuration).asScala
    val nestedRefs = references.flatMap(ref => {
      // Avoid infinite recursion
      if (!seen.contains(ref)) {
        collectReferences(ref, seen ++ references)
      } else {
        Set.empty
      }
    })

    references.toSet ++ nestedRefs
  }

  private def getFromCache[T <: BaseConfiguration](configurationId: String): Option[T] = {
    Option(configurationsCache.getIfPresent(configurationId))
  }

  private def emitEvictCacheEvent(): Unit = {
    this.invalidateAll()
    broadcastService.broadcast(EvictAllConfigurationCacheEvent(), publishEventOnSelf = false)
  }

  private def emitEvictCacheEvent(configurationId: String): Unit = {
    this.invalidate(configurationId)
    broadcastService.broadcast(EvictConfigurationCacheEvent(configurationId), publishEventOnSelf = false)
  }

  def invalidateAll(): Unit = {
    configurationsCache.invalidateAll()
    referencedConfigurationsCache.invalidateAll()
  }

  def invalidate(configurationId: String): Unit = {
    configurationsCache.invalidate(configurationId)
    Option(referencedConfigurationsCache.getIfPresent(configurationId)).foreach(refs => {
      refs.foreach(ref => {
        configurationsCache.invalidate(ref)
      })
      referencedConfigurationsCache.invalidate(configurationId)
    })
  }

  override def read[T <: BaseConfiguration](configurationId: String): T = {
    val result = getFromCache(configurationId).getOrElse({
      val repositoryValue: T = delegate.read[T](configurationId)
      putToCache(repositoryValue)
    })

    CiCloneHelper.cloneCi[T](result)
  }

  override def find[T <: BaseConfiguration](configurationId: String): Option[T] = {
    getFromCache(configurationId).orElse {
      delegate.find[T](configurationId).map(putToCache)
    }.map(CiCloneHelper.cloneCi[T])
  }

  def findByIds[T <: BaseConfiguration](configurationIds: JList[String]): JList[T] = delegate.findByIds[T](configurationIds)
    .asScala
    .map(putToCache)
    .asJava

  override def update[T <: BaseConfiguration](updated: T): T = {
    val result = delegate.update(updated)
    emitEvictCacheEvent(result.getId)
    result
  }

  override def update[T <: BaseConfiguration](updated: T, folderCiUid: CiUid): T = {
    val result = delegate.update(updated, folderCiUid)
    emitEvictCacheEvent(result.getId)
    result
  }

  override def delete(configurationId: String): Unit = {
    delegate.delete(configurationId)
    emitEvictCacheEvent(configurationId)
  }

  override def deleteByTypes(ciTypes: Seq[String]): Unit = {
    delegate.deleteByTypes(ciTypes)
    emitEvictCacheEvent()
  }

  override def getReleaseConfigurations[T <: BaseConfiguration](releaseUid: CiUid): Seq[T] = {
    val references: List[T] = releaseConfigurationReferencePersistence.getReferencesByUid(releaseUid)
      .map(cId => configurationsCache.getIfPresent(cId).asInstanceOf[T])
    val allMatch = !references.contains(null)

    if (allMatch) {
      references.map(CiCloneHelper.cloneCi[T])
    } else {
      delegate.getReleaseConfigurations[T](releaseUid)
        .map(putToCache)
    }
  }

  override def create[T <: BaseConfiguration](configuration: T): T = delegate.create(configuration)

  override def create[T <: BaseConfiguration](configuration: T, folderCiUid: CiUid): T = delegate.create(configuration, folderCiUid)

  override def findConfigurationTitleById(configId: String): String = {
    getFromCache[BaseConfiguration](configId).map(_.getTitle).getOrElse(delegate.findConfigurationTitleById(configId))
  }

  override def findAllByType[T <: BaseConfiguration](ciType: Type): JList[T] = delegate.findAllByType(ciType)

  override def existsByTypeAndTitle[T <: BaseConfiguration](ciType: Type, title: String): Boolean = delegate.existsByTypeAndTitle(ciType, title)

  override def findAllByTypeAndTitle[T <: BaseConfiguration](ciType: Type, title: String): util.List[T] = delegate.findAllByTypeAndTitle(ciType, title)

  override def findAllByTypeAndTitle[T <: BaseConfiguration](ciType: Type, title: String, folderId: String, folderOnly: Boolean): util.List[T] =
    delegate.findAllByTypeAndTitle(ciType, title, folderId, folderOnly)

  override def findFirstByType[T <: BaseConfiguration](ciType: Type): Optional[T] = delegate.findFirstByType(ciType)

  override def findFirstByType[T <: BaseConfiguration](ciType: Type, folderId: ResolutionContext): Optional[T] = delegate.findFirstByType(ciType, folderId)

  override def getReferenceReleases(configurationId: String): util.List[ReleaseBasicData] = delegate.getReferenceReleases(configurationId)

  override def getAllTypes: Seq[String] = delegate.getAllTypes

  override def exists(configurationId: String): Boolean = delegate.exists(configurationId)

  override def findAllNonInheritedReleaseReferences(folderId: String, releaseStatus: Seq[ReleaseStatus]): Seq[String] =
    delegate.findAllNonInheritedReleaseReferences(folderId, releaseStatus)

  override def findAllNonInheritedTriggerReferences(folderId: String): Seq[String] =
    delegate.findAllNonInheritedTriggerReferences(folderId)
}
