package com.xebialabs.xlrelease.service

import com.xebialabs.deployit.plugin.api.reflect.Type
import com.xebialabs.deployit.security.Permissions.{authenticationToPrincipals, getAuthentication}
import com.xebialabs.deployit.security.{PermissionEnforcer, Permissions, RoleService}
import com.xebialabs.xlrelease.api.internal.EffectiveSecurityDecorator.EFFECTIVE_SECURITY
import com.xebialabs.xlrelease.api.internal.{EffectiveSecurity, InternalMetadataDecoratorService}
import com.xebialabs.xlrelease.builder.ReleaseBuilder
import com.xebialabs.xlrelease.domain.CustomScriptTask
import com.xebialabs.xlrelease.domain.status.{ReleaseStatus, TaskStatus}
import com.xebialabs.xlrelease.repository.sql.persistence.TaskPersistence
import com.xebialabs.xlrelease.repository.sql.persistence.TaskPersistence.TaskRowsQueryParams
import com.xebialabs.xlrelease.repository.sql.persistence.data.{ReleaseBasicInfo, ReleasePhaseTaskRow}
import com.xebialabs.xlrelease.views.tasks._
import com.xebialabs.xlrelease.views.{TasksFilters, UserView}
import grizzled.slf4j.Logging
import io.micrometer.core.annotation.Timed
import org.springframework.data.domain.{Page, PageImpl}
import org.springframework.util.StringUtils._

import java.util.{List => JList}
import _root_.scala.jdk.CollectionConverters._

class SqlTaskSearchService(roleService: RoleService,
                           taskAccessService: TaskAccessService,
                           decoratorService: InternalMetadataDecoratorService,
                           taskPersistence: TaskPersistence,
                           permissionEnforcer: PermissionEnforcer
                          )
  extends TaskSearchService
    with Logging {

  @Timed
  def getTasksByRelease(tasksFilters: TasksFilters, page: Int, numberByPage: Int): Page[TaskListRelease] = {
    val releaseTaskRows: Page[ReleasePhaseTaskRow] = time("finding list of ReleasePhaseTaskRow") {
      searchReleaseTaskRows(tasksFilters, page, numberByPage)
    }

    val pageable = releaseTaskRows.getPageable
    val total = releaseTaskRows.getTotalElements

    val tasksByRelease = releaseTaskRows.getContent.asScala.toSeq.groupBy { row => row.releaseInfo }
    // it IS faster to get effective security for all releases at once
    val effectiveSecurityByReleaseId = getEffectiveSecurityByReleaseId(tasksByRelease)
    val releasesTasks = tasksByRelease.collect {
      case (releaseInfo, taskRows) =>
        val taskFullViews: JList[TaskListReleaseTask] = taskRows.map(rowToTask).asJava
        val taskListRelease = new TaskListRelease()
        taskListRelease.id = releaseInfo.releaseId
        taskListRelease.title = releaseInfo.releaseTitle
        taskListRelease.status = ReleaseStatus.valueOf(releaseInfo.releaseStatus.toUpperCase)
        taskListRelease.tasks = taskFullViews
        taskListRelease.security = effectiveSecurityByReleaseId.get(releaseInfo.releaseId).orNull

        taskListRelease
    }.toSeq
    // TODO: proposal - instead of decorating with effective security move actions behind 3 dot button menu
    //  and load effective security on-demand once user clicks on it (and then cache this information somehow on the UI)

    val sortedTasks = sortByReleaseTitle(releasesTasks)

    // TODO: Current UI expects tasks which are group by releases. For that, content object on Pageable is changed
    //  however we want to keep total count to be total no of tasks. Using reflection to update the count for now.
    //  API and UI both should be change in future to only pass tasks and UI should be migrated to React
    //  and do whatever grouping/sorting requires on UI side.
    val result = new PageImpl[TaskListRelease](sortedTasks, pageable, total)
    val totalField = result.getClass.getDeclaredField("total")
    totalField.setAccessible(true)
    totalField.set(result, total)

    result
  }

  type ReleaseId = String

  private def getEffectiveSecurityByReleaseId(tasksByRelease: Map[ReleaseBasicInfo, Seq[ReleasePhaseTaskRow]]): Map[ReleaseId, EffectiveSecurity] = {
    logger.trace(s"decorateWithEffectiveSecurity called for ${tasksByRelease.size}")
    val cis = tasksByRelease.collect {
      case (info, _) =>
        val r = ReleaseBuilder.newTemplate().withId(info.releaseId).build()
        r.set$securedCi(info.releaseSecurityUid)
        r
    }.toList.asJava
    decoratorService.decorate(cis, List(EFFECTIVE_SECURITY).asJava)
    cis.asScala.map { r =>
      r.getId -> r.get$metadata().get(EFFECTIVE_SECURITY).asInstanceOf[EffectiveSecurity]
    }.toMap
  }

  private def getTaskTypePropertyValue(taskType: Type, propertyName: String): String = {
    Option(taskType.getDescriptor.getPropertyDescriptor(propertyName)).map(pd => pd.getDefaultValue.asInstanceOf[String]).orNull
  }

  private def rowToTask(row: ReleasePhaseTaskRow): TaskListReleaseTask = {
    val taskRow = row.taskRow

    val taskStatus = TaskStatus.valueOf(taskRow.status.toUpperCase)
    val taskType = if (Type.valueOf(taskRow.taskType).exists()) {
      Type.valueOf(taskRow.taskType)
    } else {
      CustomScriptTask.UNKNOWN_TYPE //It does not matter if we return UNKNOWN_TYPE or UNKNOWN_TASK_TYPE
    }

    val taskColor = getTaskTypePropertyValue(taskType, "taskColor")
    val taskIconLocation = getTaskTypePropertyValue(taskType, "iconLocation")
    val taskIconClass = getTaskTypePropertyValue(taskType, "iconClass")
    val taskUpdatable = taskStatus != TaskStatus.ABORTED && !taskStatus.isDone && !taskStatus.isDoneInAdvance

    val taskListReleaseTask = new TaskListReleaseTask()
    taskListReleaseTask.id = taskRow.taskId
    taskListReleaseTask.title = taskRow.title

    if (taskStatus.isOneOf(TaskStatus.PLANNED, TaskStatus.PENDING)) {
      taskListReleaseTask.scheduledStartDate = taskRow.startDate
    } else {
      taskListReleaseTask.startDate = taskRow.startDate
    }
    if (!taskStatus.isDone && !taskStatus.isDoneInAdvance) {
      taskListReleaseTask.dueDate = taskRow.endDate
    } else {
      taskListReleaseTask.endDate = taskRow.endDate
    }

    taskListReleaseTask.`type` = taskType.toString
    taskListReleaseTask.hasBeenStarted = taskStatus.hasBeenStarted // Need this?
    taskListReleaseTask.updatable = taskUpdatable
    taskListReleaseTask.active = taskStatus.isActive // Need this?
    taskListReleaseTask.done = taskStatus.isDone // Need this?
    taskListReleaseTask.plannedDuration = taskRow.plannedDuration
    if (hasText(taskRow.owner)) {
      taskListReleaseTask.owner = new UserView(taskRow.owner)
    }
    taskListReleaseTask.team = taskRow.team
    taskListReleaseTask.status = taskStatus
    taskListReleaseTask.locked = taskRow.locked
    taskListReleaseTask.color = taskColor
    taskListReleaseTask.customIconLocation = taskIconLocation
    taskListReleaseTask.customIconClass = taskIconClass
    taskListReleaseTask.editable = taskUpdatable
    taskListReleaseTask.releaseId = row.releaseInfo.releaseId
    taskListReleaseTask.phaseTitle = row.phaseBasicInfo.phaseTitle

    taskListReleaseTask
  }

  private def time[A](desc: String)(block: => A): A = {
    val startTime = System.currentTimeMillis()
    val res = block
    val endTime = System.currentTimeMillis()
    val total = endTime - startTime
    logger.trace(s"TIME: $desc took $total")
    res
  }

  private def sortByReleaseTitle(releasesTasks: Seq[TaskListRelease]): JList[TaskListRelease] = {
    releasesTasks.sortBy(releaseTasks => Option(releaseTasks.title).getOrElse("")).asJava
  }

  //  @VisibleForTesting
  def searchReleaseTaskRows(tasksFilters: TasksFilters, pageNum: Int, numberByPage: Int): Page[ReleasePhaseTaskRow] = {
    val allowedTaskTypes = taskAccessService.getAllowedTaskTypesForAuthenticatedUser.asScala.toList
    val allTaskTypes = taskAccessService.getAllTaskTypes().asScala.toList
    val principals = authenticationToPrincipals(getAuthentication).asScala.toSeq
    val roleIds = roleService.getRolesFor(getAuthentication).asScala.map(_.getId).toSeq
    val username = Permissions.getAuthenticatedUserName
    // TODO isAdmin loads principals and roleIds
    val isAdmin = permissionEnforcer.isCurrentUserAdmin
    val queryParams = TaskRowsQueryParams(
      tasksFilters, allowedTaskTypes, allTaskTypes, principals, roleIds, pageNum, numberByPage, username, isAdmin
    )
    val rows = taskPersistence.findReleaseTaskRowsByQuery(queryParams)
    rows
  }
}
