package com.xebialabs.xlrelease.service

import com.xebialabs.deployit.plugin.api.reflect.{PropertyDescriptor, PropertyKind, Type}
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem
import com.xebialabs.xlrelease.domain.PythonScript.CUSTOM_SCRIPT_TASK_PROPERTY
import com.xebialabs.xlrelease.domain._
import com.xebialabs.xlrelease.repository.{CiProperty, Ids}
import org.springframework.stereotype.Component

import java.util
import scala.jdk.CollectionConverters._

private case class PropertyNameAliasesPair(name: String, aliases: Set[String])

@Component
class TaskTypeConverterService(serverConnectionMigrator: ServerConnectionMigrator) {
  def copyPythonScriptProperties(source: Task, target: CustomScriptTask): Unit = {
    var propertiesSource: ConfigurationItem = null
    val targetScript: ConfigurationItem = target.getPythonScript
    var excludedProperties: Seq[String] = Seq.empty

    source match {
      case task: CustomScriptTask =>
        propertiesSource = task.getPythonScript
        excludedProperties = Seq(CUSTOM_SCRIPT_TASK_PROPERTY) ++ PythonScript.UPDATEABLE_PROPERTIES.asScala
      case _ => propertiesSource = source
    }

    copyNonContainmentProperties(propertiesSource, targetScript, excludedProperties)
  }

  def copyContainerTaskProperties(source: Task, target: ContainerTask): Unit = {
    val propertiesSource = source match {
      case task: CustomScriptTask => task.getPythonScript
      case task: ScriptTask =>
        forceCopySelectedNonContainmentProperties(task, target, selectedProperties = Seq("script"))
        task
      case _ => source
    }

    copyNonContainmentProperties(propertiesSource, target, excludedProperties = Seq.empty)
  }

  def copyProperties(source: Task, target: Task, excludedProperties: Seq[String]): Unit = {
    target.setId(source.getId)
    copyNonContainmentProperties(source, target, excludedProperties)
    serverConnectionMigrator.migrateProperties(source, target)
    removeVariableMappingsForNonExistingProperties(source, target)
    moveContainedCollectionProperties(source, target)
    updateLinksInParallelGroup(source, target)
    updateInContainer(source, target)
    target.setTaskFailureHandlerEnabled(source.isTaskFailureHandlerEnabled)
  }

  private def copyNonContainmentProperties(from: ConfigurationItem, to: ConfigurationItem, excludedProperties: Seq[String]): Unit = {
    for (newPd <- to.getType.getDescriptor.getPropertyDescriptors.asScala) {
      if (!excludedProperties.contains(newPd.getName) && !newPd.isHidden) {
        val oldPd = from.getType.getDescriptor.getPropertyDescriptor(newPd.getName)
        copyNonContainmentProperty(from, to, oldPd, newPd)
      }
    }
  }

  private def copyNonContainmentProperty(from: ConfigurationItem, to: ConfigurationItem,
                                         oldPd: PropertyDescriptor, newPd: PropertyDescriptor,
                                         forced: Boolean = false): Unit = {
    if (isSameKindOfProperty(newPd, oldPd) && (hasSameCategory(newPd, oldPd) || forced)) {
      if (oldPd.getKind.isSimple || !oldPd.isAsContainment && hasCopyableReferencedType(oldPd, newPd)) {
        newPd.set(to, oldPd.get(from))
      }
    }
  }

  def forceCopySelectedNonContainmentProperties(from: ConfigurationItem, to: ConfigurationItem, selectedProperties: Seq[String]): Unit = {
    for (newPd <- to.getType.getDescriptor.getPropertyDescriptors.asScala) {
      if (selectedProperties.contains(newPd.getName) && !newPd.isHidden) {
        val oldPd = from.getType.getDescriptor.getPropertyDescriptor(newPd.getName)
        copyNonContainmentProperty(from, to, oldPd, newPd, forced = true)
      }
    }
  }

  private def removeVariableMappingsForNonExistingProperties(oldTask: Task, updatedTask: Task) = {
    updatedTask.getVariableMapping.keySet.removeIf((fqPropertyName: String) => {
      val newProperty = CiProperty.of(updatedTask, fqPropertyName)
      val oldProperty = CiProperty.of(oldTask, fqPropertyName)
      !oldProperty.isPresent || !newProperty.isPresent || !isSameKindOfProperty(oldProperty.get.getDescriptor, newProperty.get.getDescriptor)
    })
  }

  private def moveContainedCollectionProperties(from: ConfigurationItem, to: ConfigurationItem): Unit = {
    for (newPd <- to.getType.getDescriptor.getPropertyDescriptors.asScala) {
      val oldPd = from.getType.getDescriptor.getPropertyDescriptor(newPd.getName)
      if (isSameKindOfProperty(newPd, oldPd)) {
        val isContainedCollectionProperty = oldPd.isAsContainment && (oldPd.getKind ne PropertyKind.CI)
        if (isContainedCollectionProperty && hasCopyableReferencedType(oldPd, newPd)) {
          val referencedType: Type = oldPd.getReferencedType
          if (null != referencedType) {
            val oldCollection = oldPd.get(from).asInstanceOf[util.Collection[ConfigurationItem]]
            oldCollection.forEach({ ci =>
              val newId: String = to.getId + "/" + Ids.getName(ci.getId)
              ci.setId(newId)
            })
            newPd.set(to, oldCollection)
          }
        }
      }
    }
  }

  private def isSameKindOfProperty(newPd: PropertyDescriptor, oldPd: PropertyDescriptor) = {
    oldPd != null && oldPd.getKind == newPd.getKind && oldPd.isPassword == newPd.isPassword && oldPd.isAsContainment == newPd.isAsContainment
  }

  private def hasSameCategory(newPd: PropertyDescriptor, oldPd: PropertyDescriptor) = {
    oldPd != null && oldPd.getCategory == newPd.getCategory
  }

  private def hasCopyableReferencedType(oldPd: PropertyDescriptor, newPd: PropertyDescriptor) = {
    val oldReferencedType = oldPd.getReferencedType
    val newReferencedType = newPd.getReferencedType
    (oldReferencedType == null && newReferencedType == null) || (oldReferencedType != null && oldReferencedType.instanceOf(newReferencedType))
  }

  private def updateLinksInParallelGroup(oldTask: Task, newTask: Task): Unit = {
    oldTask.getContainer match {
      case group: ParallelGroup =>
        for (link <- group.getLinksOf(oldTask).asScala) {
          if (link.hasTarget(oldTask)) {
            link.setTarget(newTask)
          }
          if (link.hasSource(oldTask)) {
            link.setSource(newTask)
          }
        }
      case _ =>
    }
  }

  private def updateInContainer(oldTask: Task, newTask: Task): Unit = {
    oldTask.getContainer.replaceTask(oldTask, newTask)
  }
}

