package com.xebialabs.xlrelease.repository.sql

import com.xebialabs.xlrelease.db.sql.transaction.{IsReadOnly, IsTransactional}
import com.xebialabs.xlrelease.domain.{Datacenter, DatacenterTargetState}
import com.xebialabs.xlrelease.repository.DatacenterRepository
import com.xebialabs.xlrelease.repository.sql.jpa.JpaDatacenterRepository
import com.xebialabs.xlrelease.repository.sql.jpa.entities.ClusterDatacenter

import java.time.Instant
import jakarta.persistence.OptimisticLockException
import scala.jdk.CollectionConverters._
import scala.jdk.OptionConverters._

@IsTransactional
class SqlDatacenterRepository(jpaDatacenterRepository: JpaDatacenterRepository) extends DatacenterRepository {

  @IsReadOnly
  override def find(datacenter: String): Option[Datacenter] = {
    jpaDatacenterRepository.findByDatacenterIgnoreCase(datacenter)
      .map(fromRow)
      .toScala
  }

  @IsReadOnly
  override def findAll(): Seq[Datacenter] = {
    jpaDatacenterRepository.findAll().asScala.map(fromRow).toSeq
  }

  @IsReadOnly
  override def count(): Long = {
    jpaDatacenterRepository.count()
  }

  override def delete(datacenter: String): Unit = {
    jpaDatacenterRepository.deleteByDatacenterIgnoreCase(datacenter)
  }

  override def create(datacenter: String, state: DatacenterTargetState, createdBy: String): Unit = {
    val newClusterDatacenterRow = new ClusterDatacenter()
    newClusterDatacenterRow.setDatacenter(datacenter)
    newClusterDatacenterRow.setTargetState(state.value())
    newClusterDatacenterRow.setUpdatedBy(createdBy)
    newClusterDatacenterRow.setUpdatedDate(Instant.now())
    jpaDatacenterRepository.save(newClusterDatacenterRow)
  }

  override def update(datacenter: String, state: DatacenterTargetState, updatedBy: String): Unit = {
    val maybeClusterDatacenterRow = jpaDatacenterRepository.findByDatacenterIgnoreCase(datacenter)
    if (maybeClusterDatacenterRow.isPresent) {
      val existingClusterDatacenterRow = maybeClusterDatacenterRow.get()
      existingClusterDatacenterRow.setTargetState(state.value())
      existingClusterDatacenterRow.setUpdatedBy(updatedBy)
      existingClusterDatacenterRow.setUpdatedDate(Instant.now())
      try {
        jpaDatacenterRepository.save(existingClusterDatacenterRow)
      } catch {
        case e: OptimisticLockException =>
          throw new RuntimeException(s"Failed to update cluster state for datacenter $datacenter", e)
      }
    } else {
      throw new IllegalArgumentException(s"Datacenter $datacenter does not exist")
    }
  }

  private def fromRow(row: ClusterDatacenter): Datacenter = {
    Datacenter(row.getDatacenter, DatacenterTargetState.fromString(row.getTargetState), row.getUpdatedBy, row.getUpdatedDate)
  }

}
