package com.xebialabs.xlrelease.scheduler.logs

import com.xebialabs.xlrelease.db.sql.SqlBuilder.Dialect
import com.xebialabs.xlrelease.db.sql.transaction.{IsReadOnly, IsTransactional}
import com.xebialabs.xlrelease.repository.sql.persistence.Schema.TASK_EXECUTIONS
import com.xebialabs.xlrelease.repository.sql.persistence.TaskPersistence.hash
import com.xebialabs.xlrelease.repository.sql.persistence.Utils._
import com.xebialabs.xlrelease.repository.sql.persistence.{PersistenceSupport, Utils}
import com.xebialabs.xlrelease.scheduler.logs.TaskExecutionRepository._
import com.xebialabs.xlrelease.storage.domain.LogEntry
import grizzled.slf4j.Logging
import io.micrometer.core.annotation.Timed
import org.springframework.data.domain.{Page, Pageable}
import org.springframework.jdbc.core.JdbcTemplate

import java.time.Instant
import scala.util.{Failure, Success, Try}


@IsTransactional
class SqlTaskExecutionRepository(implicit val jdbcTemplate: JdbcTemplate, val dialect: Dialect)
  extends TaskExecutionRepository
    with PersistenceSupport
    with TaskExecutionRowMapper
    with Logging
    with Utils {

  @Timed
  @IsReadOnly
  override def read(taskId: String, executionId: String): Option[TaskExecutionEntry] = {
    findOptional(_.queryForObject(STMT_FIND_TASK_EXECUTION, taskExecutionRowMapper, hash(taskId), executionId))
  }

  @Timed
  override def update(operation: UpdateTaskExecutionOperation): Option[TaskExecutionEntry] = {
    operation match {
      case TaskExecutionRepository.FinishExecution(taskId, executionId, endDate) =>
        val maybeRow = read(taskId, executionId)
        maybeRow match {
          case Some(row) => Some(updateRow(row.copy(endDate = endDate)))
          case None => None // Must be for a non container based task so do nothing
        }
      case TaskExecutionRepository.UpdateWithLogEntry(logEntry, maybeEndDate) =>
        val taskExecutionEntry = logEntryToTaskExecutionEntry(logEntry)
        val maybeRow = read(logEntry.taskId, logEntry.executionId)
        val row = maybeRow match {
          case Some(row) =>
            updateRow(row, taskExecutionEntry, maybeEndDate.orNull)
          case None =>
            Try(create(taskExecutionEntry)) match {
              case Failure(exception) =>
                val maybeRow = read(logEntry.taskId, logEntry.executionId)
                maybeRow.map(row => updateRow(row, taskExecutionEntry, maybeEndDate.orNull)).getOrElse(throw exception)
              case Success(row) => row
            }
        }
        Some(row)
    }
  }

  private def updateRow(row: TaskExecutionEntry, taskExecutionEntry: TaskExecutionEntry, endDate: Instant): TaskExecutionEntry = {
    val updated = row.copy(
      // Always choose max values in case log entries are received out of order
      lastJob = Integer.max(taskExecutionEntry.lastJob, row.lastJob),
      lastChunk = Integer.max(taskExecutionEntry.lastChunk, row.lastChunk),
      lastModifiedDate = maxInstant(taskExecutionEntry.lastModifiedDate, row.lastModifiedDate),
      endDate = endDate
    )
    updateRow(updated)
  }

  @Timed
  override def delete(operation: DeleteOperation): Unit = {
    operation match {
      case Delete(taskId, executionId) => deleteExecution(hash(taskId), executionId)
      case DeleteWithTaskExecutionEntry(entry) => deleteExecution(entry.taskIdHash, entry.executionId)
    }
  }

  @Timed
  @IsReadOnly
  override def find(findOperation: FindOperation, pageable: Pageable): Page[TaskExecutionEntry] = {
    SqlTaskExecutionQueryBuilder(dialect, namedTemplate)
      .from(findOperation)
      .withPageable(pageable)
      .build()
      .execute()
  }

  private def maxInstant(a: Instant, b: Instant): Instant = {
    if (a.isAfter(b)) a else b
  }

  private def logEntryToTaskExecutionEntry(logEntry: LogEntry): TaskExecutionEntry = {
    TaskExecutionEntry(
      hash(logEntry.taskId),
      logEntry.executionId,
      logEntry.jobId.toInt,
      logEntry.chunk.toInt,
      Instant.parse(logEntry.lastEntryTimestamp),
      null
    )
  }

  private def createSqlParamMap(row: TaskExecutionEntry): Map[String, Any] = {
    Map(
      TASK_EXECUTIONS.TASK_ID_HASH -> row.taskIdHash,
      TASK_EXECUTIONS.EXECUTION_ID -> row.executionId,
      TASK_EXECUTIONS.LAST_JOB -> row.lastJob,
      TASK_EXECUTIONS.LAST_CHUNK -> row.lastChunk,
      TASK_EXECUTIONS.LAST_MODIFIED_DATE -> row.lastModifiedDate.asTimestamp,
      TASK_EXECUTIONS.END_DATE -> row.endDate.asTimestamp
    )
  }

  private def create(row: TaskExecutionEntry): TaskExecutionEntry = {
    namedTemplate.update(STMT_INSERT_ROW, createSqlParamMap(row))
    row
  }

  private def updateRow(row: TaskExecutionEntry): TaskExecutionEntry = {
    namedTemplate.update(STMT_UPDATE_ROW, createSqlParamMap(row))
    row
  }

  private def deleteExecution(taskIdHash: String, executionId: String): Unit = {
    val params = Map(
      TASK_EXECUTIONS.TASK_ID_HASH -> taskIdHash,
      TASK_EXECUTIONS.EXECUTION_ID -> executionId
    )
    namedTemplate.update(STMT_DELETE_ROW, params)
  }

  private val STMT_INSERT_ROW =
    s"""INSERT INTO ${TASK_EXECUTIONS.TABLE} (
       |  ${TASK_EXECUTIONS.TASK_ID_HASH},
       |  ${TASK_EXECUTIONS.EXECUTION_ID},
       |  ${TASK_EXECUTIONS.LAST_JOB},
       |  ${TASK_EXECUTIONS.LAST_CHUNK},
       |  ${TASK_EXECUTIONS.LAST_MODIFIED_DATE},
       |  ${TASK_EXECUTIONS.END_DATE}
       | ) VALUES (
       |  :${TASK_EXECUTIONS.TASK_ID_HASH},
       |  :${TASK_EXECUTIONS.EXECUTION_ID},
       |  :${TASK_EXECUTIONS.LAST_JOB},
       |  :${TASK_EXECUTIONS.LAST_CHUNK},
       |  :${TASK_EXECUTIONS.LAST_MODIFIED_DATE},
       |  :${TASK_EXECUTIONS.END_DATE}
       | )
       |""".stripMargin

  private val STMT_UPDATE_ROW =
    s"""UPDATE ${TASK_EXECUTIONS.TABLE}
       |  SET
       |    ${TASK_EXECUTIONS.LAST_JOB} = :${TASK_EXECUTIONS.LAST_JOB},
       |    ${TASK_EXECUTIONS.LAST_CHUNK} = :${TASK_EXECUTIONS.LAST_CHUNK},
       |    ${TASK_EXECUTIONS.LAST_MODIFIED_DATE} = :${TASK_EXECUTIONS.LAST_MODIFIED_DATE},
       |    ${TASK_EXECUTIONS.END_DATE} = :${TASK_EXECUTIONS.END_DATE}
       |  WHERE
       |    ${TASK_EXECUTIONS.TASK_ID_HASH} = :${TASK_EXECUTIONS.TASK_ID_HASH}
       |  AND
       |    ${TASK_EXECUTIONS.EXECUTION_ID} = :${TASK_EXECUTIONS.EXECUTION_ID}
       |""".stripMargin

  private val STMT_DELETE_ROW =
    s"""DELETE FROM ${TASK_EXECUTIONS.TABLE}
       |  WHERE
       |    ${TASK_EXECUTIONS.TASK_ID_HASH} = :${TASK_EXECUTIONS.TASK_ID_HASH}
       |  AND
       |    ${TASK_EXECUTIONS.EXECUTION_ID} = :${TASK_EXECUTIONS.EXECUTION_ID}
       |""".stripMargin

  private val STMT_FIND_TASK_EXECUTION =
    s"""SELECT * FROM ${TASK_EXECUTIONS.TABLE}
       |  WHERE ${TASK_EXECUTIONS.TASK_ID_HASH} = ?
       |  AND ${TASK_EXECUTIONS.EXECUTION_ID} = ?
       |""".stripMargin


}
