package com.xebialabs.xlrelease.spring.config

import com.xebialabs.xlplatform.repository.sql.{Database, DatabaseConfig, DatabaseType}
import com.xebialabs.xlplatform.upgrade.sql.{SqlInitializationStrategy, SqlRepositoryVersionService, SqlUpgradeStrategy}
import com.xebialabs.xlrelease.config.XlrConfig
import com.xebialabs.xlrelease.db.XLReleaseDbInitializer
import com.xebialabs.xlrelease.db.sql.SqlBuilder._
import com.xebialabs.xlrelease.db.sql.{DatabaseInfo, SqlDialectDetector}
import com.xebialabs.xlrelease.features.DatabaseProxyFeature
import com.xebialabs.xlrelease.repository.sql.JdbcTemplateHolder
import com.xebialabs.xlrelease.repository.sql.persistence.ReleaseCacheService
import com.xebialabs.xlrelease.security.XlRoleRepository
import com.xebialabs.xlrelease.security.entities.XlRole
import com.xebialabs.xlrelease.service.SqlCiIdService
import com.xebialabs.xlrelease.spring.config.SqlConfiguration._
import com.zaxxer.hikari.metrics.MetricsTrackerFactory
import com.zaxxer.hikari.{HikariConfig, HikariDataSource}
import grizzled.slf4j.Logging
import org.hibernate.cfg.AvailableSettings
import org.springframework.context.annotation._
import org.springframework.data.jpa.repository.config.EnableJpaRepositories
import org.springframework.jdbc.core.JdbcTemplate
import org.springframework.jdbc.datasource.{DataSourceTransactionManager, TransactionAwareDataSourceProxy}
import org.springframework.orm.jpa.vendor.HibernateJpaVendorAdapter
import org.springframework.orm.jpa.{JpaTransactionManager, LocalContainerEntityManagerFactoryBean}
import org.springframework.transaction.PlatformTransactionManager
import org.springframework.transaction.annotation.TransactionManagementConfigurer
import org.springframework.transaction.support.TransactionTemplate

import java.util.Properties
import java.util.concurrent.{Executors, ThreadFactory}
import javax.sql.DataSource
import scala.jdk.CollectionConverters._

//noinspection ScalaStyle
@Configuration
@EnableJpaRepositories(
  basePackageClasses = Array(classOf[XlRoleRepository]),
  entityManagerFactoryRef = SECURITY_ENTITY_MANAGER,
  transactionManagerRef = XLR_TRANSACTION_MANAGER
)
class SqlConfiguration(xlrConfig: XlrConfig,
                       releaseCacheConfiguration: ReleaseCacheConfiguration,
                       databaseProxyFeature: DatabaseProxyFeature,
                       metricsTrackerFactory: MetricsTrackerFactory)
  extends TransactionManagementConfigurer with Logging {

  @Bean
  def jdbcTemplateHolder(): JdbcTemplateHolder.type = {
    JdbcTemplateHolder.init(xlrRepositoryJdbcTemplate())
    JdbcTemplateHolder
  }

  @Bean
  def ciIdService() = new SqlCiIdService()

  @Bean
  def releaseCacheService(): ReleaseCacheService = {
    new ReleaseCacheService(releaseCacheConfiguration.releaseCacheManager())
  }

  @Bean
  def upgradeStrategy = new SqlUpgradeStrategy()

  @Bean
  def initializationStrategy = new SqlInitializationStrategy()

  @Bean
  def repositoryVersionService() = new SqlRepositoryVersionService(xlrRepositoryDatabase())

  @Bean
  def xlrMigrationsDbInitializer(): XLReleaseDbInitializer = {
    new XLReleaseDbInitializer("com/xebialabs/xlplatform/upgrade/sql/db/xl-migrations-schema.yaml", xlrRepositoryDataSourceProxy())
  }

  @Bean
  def xlrRepositoryDbInitializer(): XLReleaseDbInitializer = {
    new XLReleaseDbInitializer("com/xebialabs/xlrelease/db/xl-release-schema-initializer.yaml", xlrRepositoryDataSourceProxy())
  }

  @Bean
  def rawRepositoryDataSource(): HikariDataSource = {
    new HikariDataSource(xlrRepositoryDataSourceConfig(xlrConfig))
  }

  @Bean(destroyMethod = "close", name = Array(XLR_REPOSITORY_DATA_SOURCE))
  def xlrRepositoryDataSource(): CloseableDataSource = {
    new XlrDelegatingDataSource(rawRepositoryDataSource(), databaseProxyFeature, xlrConfig)
  }

  @Bean
  def xlrRepositoryDataSourceProxy(): DataSource = {
    new TransactionAwareDataSourceProxy(xlrRepositoryDataSource())
  }

  @Bean
  def xlrRepositoryDatabase(): Database = {
    val databaseConfig = DatabaseConfig(
      getDatabaseType(xlrRepositorySqlDialect()),
      "xlr-ds",
      minThreads = xlrConfig.database.maxPoolSize,
      maxThreads = xlrConfig.database.maxPoolSize,
      maxConnections = xlrConfig.database.maxPoolSize,
      queryTimeout = xlrConfig.timeouts.releaseActionResponse
    )

    Database(xlrRepositoryDataSourceProxy(), databaseConfig)
  }

  override def annotationDrivenTransactionManager(): PlatformTransactionManager = xlrRepositoryTransactionManager()

  @Bean
  def securityEntityManager: LocalContainerEntityManagerFactoryBean = {
    val em = new LocalContainerEntityManagerFactoryBean()
    em.setDataSource(xlrRepositoryDataSource())
    em.setPackagesToScan(classOf[XlRole].getPackageName)
    val vendorAdapter = new HibernateJpaVendorAdapter()
    em.setJpaVendorAdapter(vendorAdapter)
    // vendor adapter properties will be added to properties we specify
    val properties = hibernateProperties.asScala.toMap.asJava
    em.setJpaPropertyMap(properties)
    em
  }

  private def hibernateProperties: Properties = {
    val hibernateProperties = new Properties()
    val dialect = xlrDbInfo() match {
      case DatabaseInfo.TestDatabaseInfo(metadata) => throw new IllegalStateException("Unsupported database type: TestDatabaseInfo")
      case DatabaseInfo.Unknown(metadata) => throw new IllegalStateException("Unsupported database type: Unknown")
      case DatabaseInfo.Db2(metadata) => "org.hibernate.dialect.DB2Dialect"
      case DatabaseInfo.Derby(metadata) => "org.hibernate.dialect.DerbyDialect"
      case DatabaseInfo.H2(metadata) => "org.hibernate.dialect.H2Dialect"
      case DatabaseInfo.MsSqlServer(metadata) => "org.hibernate.dialect.SQLServerDialect"
      case DatabaseInfo.MySql(metadata) => "org.hibernate.dialect.MySQLDialect"
      case DatabaseInfo.Oracle(metadata) => "org.hibernate.dialect.OracleDialect"
      case DatabaseInfo.PostgreSql(metadata) => "org.hibernate.dialect.PostgreSQLDialect"
    }
    // if not specified STATEMENT_BATCH_SIZE is defined on database Dialect (can be 0, 1, 15)
    // additionally JDBC drivers can rewrite batch statements
    hibernateProperties.setProperty(AvailableSettings.STATEMENT_BATCH_SIZE, "15")
    hibernateProperties.setProperty(AvailableSettings.BATCH_VERSIONED_DATA, "true")
    hibernateProperties.setProperty(AvailableSettings.ORDER_INSERTS, "true")
    hibernateProperties.setProperty(AvailableSettings.ORDER_UPDATES, "true")
    hibernateProperties.setProperty(AvailableSettings.DIALECT, dialect)
    hibernateProperties.setProperty(AvailableSettings.HBM2DDL_AUTO, "none")
    hibernateProperties.setProperty(AvailableSettings.SHOW_SQL, "false")
    hibernateProperties.setProperty(AvailableSettings.ALLOW_EXTENSIONS_IN_CDI, "true")
    // pair GENERATE_STATISTICS and LOG_SESSION_METRICS
    val showStatistics = "false" // enable if you want to check number of batches
    hibernateProperties.setProperty(AvailableSettings.GENERATE_STATISTICS, showStatistics)
    hibernateProperties.setProperty(AvailableSettings.LOG_SESSION_METRICS, showStatistics)
    hibernateProperties
  }

  @Bean
  def xlrRepositoryTransactionManager(): PlatformTransactionManager = {
    val txManager = new JpaTransactionManager()
    txManager.setEntityManagerFactory(securityEntityManager.getObject)
    txManager.setDataSource(xlrRepositoryDataSource())
    txManager
  }

  @Bean
  def xlrRepositoryTransactionTemplate(): TransactionTemplate = {
    new TransactionTemplate(xlrRepositoryTransactionManager())
  }

  @Bean
  def xlrRepositoryJdbcTemplate(): JdbcTemplate = {
    new JdbcTemplate(xlrRepositoryDataSource())
  }

  @Bean
  def xlrDatabaseInformation: DatabaseInfo = {
    DatabaseInfo(rawRepositoryDataSource())
  }

  @Bean
  def xlrDbInfo(): DatabaseInfo = {
    val dbInfo = DatabaseInfo(rawRepositoryDataSource())
    logger.info(s"Repository database is ${dbInfo.dbName}:${dbInfo.dbVersion}")
    dbInfo
  }

  @Bean
  def xlrRepositoryDialectDetector(): SqlDialectDetector = {
    new SqlDialectDetector()
  }

  @Bean
  def xlrRepositorySqlDialect(): Dialect = {
    xlrRepositoryDialectDetector().detectSqlDialect(xlrDbInfo())
  }

  @Bean
  def xlrPrivilegedThreadFactory(): ThreadFactory =
    Executors.privilegedThreadFactory()

  // Report database
  @Bean
  def rawReportingDataSource(): HikariDataSource = {
    new HikariDataSource(reportingDataSourceConfig(xlrConfig))
  }

  @Bean(destroyMethod = "close", name = Array(XLR_REPORTING_DATA_SOURCE))
  def reportingDataSource(): CloseableDataSource = {
    new XlrDelegatingDataSource(rawReportingDataSource(), databaseProxyFeature, xlrConfig)
  }

  @Bean
  def reportingDataSourceProxy(): DataSource = {
    new TransactionAwareDataSourceProxy(reportingDataSource())
  }

  @Bean
  def reportingDbInitializer(): XLReleaseDbInitializer = {
    new XLReleaseDbInitializer("com/xebialabs/xlrelease/db/archivingTablesInitialize.yaml", reportingDataSourceProxy())
  }

  @Bean
  def reportingTransactionManager(): PlatformTransactionManager = {
    val txManager = new DataSourceTransactionManager()
    txManager.setDataSource(reportingDataSource())
    txManager
  }

  @Bean
  def reportTransactionTemplate(): TransactionTemplate = {
    new TransactionTemplate(reportingTransactionManager())
  }

  @Bean
  def reportingJdbcTemplate(): JdbcTemplate = {
    new JdbcTemplate(reportingDataSource())
  }

  @Bean
  def reportingDialectDetector(): SqlDialectDetector = {
    new SqlDialectDetector()
  }

  @Bean
  def reportingSqlDialect(): Dialect = {
    val jdbcTemplate = new JdbcTemplate(rawReportingDataSource())
    reportingDialectDetector().detectSqlDialect(jdbcTemplate)
  }

  private def xlrRepositoryDataSourceConfig(xlrConfig: XlrConfig): HikariConfig = {
    val hc = new HikariConfig()
    hc.setJdbcUrl(xlrConfig.database.dbUrl)
    hc.setDriverClassName(xlrConfig.database.dbDriverClassname)
    hc.setUsername(xlrConfig.database.dbUsername)
    hc.setPassword(xlrConfig.database.dbPassword)
    hc.setPoolName(xlrConfig.database.poolName)
    hc.setMaximumPoolSize(xlrConfig.database.maxPoolSize)
    hc.setConnectionTimeout(xlrConfig.database.connectionTimeout)
    hc.setMaxLifetime(xlrConfig.database.maxLifeTime)
    hc.setIdleTimeout(xlrConfig.database.idleTimeout)
    hc.setMinimumIdle(xlrConfig.database.minimumIdle)
    hc.setLeakDetectionThreshold(xlrConfig.database.leakConnectionThreshold)
    hc.setMetricsTrackerFactory(metricsTrackerFactory)
    hc.setTransactionIsolation("TRANSACTION_READ_COMMITTED")
    hc.setThreadFactory(xlrPrivilegedThreadFactory())
    hc.setAllowPoolSuspension(true)
    hc
  }

  private def reportingDataSourceConfig(xlrConfig: XlrConfig): HikariConfig = {
    val hc = new HikariConfig()
    hc.setJdbcUrl(xlrConfig.reporting.dbUrl)
    hc.setDriverClassName(xlrConfig.reporting.dbDriverClassname)
    hc.setUsername(xlrConfig.reporting.dbUsername)
    hc.setPassword(xlrConfig.reporting.dbPassword)
    hc.setMaximumPoolSize(xlrConfig.reporting.maxPoolSize)
    hc.setPoolName(xlrConfig.reporting.poolName)
    hc.setConnectionTimeout(xlrConfig.reporting.connectionTimeout)
    hc.setMaxLifetime(xlrConfig.reporting.maxLifeTime)
    hc.setIdleTimeout(xlrConfig.reporting.idleTimeout)
    hc.setMinimumIdle(xlrConfig.reporting.minimumIdle)
    hc.setLeakDetectionThreshold(xlrConfig.reporting.leakConnectionThreshold)
    hc.setMetricsTrackerFactory(metricsTrackerFactory)
    hc.setThreadFactory(xlrPrivilegedThreadFactory())
    hc.setAllowPoolSuspension(true)
    hc
  }
}

object SqlConfiguration {
  final val XLR_REPOSITORY_DATA_SOURCE = "xlrRepositoryDataSource"
  final val XLR_REPORTING_DATA_SOURCE = "reportingDataSource"

  def getDatabaseType(dialect: Dialect): DatabaseType with Serializable = dialect match {
    case DerbyDialect(_) => DatabaseType.Derby
    case OracleDialect(_) => DatabaseType.Oracle
    case Db2Dialect(_) => DatabaseType.DB2
    case MSSQLDialect(_) => DatabaseType.SQLServer
    case CommonDialect(dbName) if dbName.contains("h2") => DatabaseType.H2
    case CommonDialect(dbName) if dbName.contains("mysql") => DatabaseType.MySQL
    case CommonDialect(dbName) if dbName.contains("postgres") => DatabaseType.Postgres
    case _ => DatabaseType.H2
  }

  final val SECURITY_ENTITY_MANAGER = "securityEntityManager"
  final val XLR_TRANSACTION_MANAGER = "xlrRepositoryTransactionManager"
}
