package com.xebialabs.xlrelease.repository.sql.query

import com.xebialabs.xlrelease.db.sql.SqlBuilder.Dialect
import com.xebialabs.xlrelease.domain.UserProfile
import com.xebialabs.xlrelease.repository.query._
import com.xebialabs.xlrelease.repository.sql.persistence.Schema.USER_PROFILE
import com.xebialabs.xlrelease.repository.sql.persistence.{CompressionSupport, UserProfileMapper}
import com.xebialabs.xlrelease.repository.sql.query.SqlUserProfileQueryBuilder.profileAlias
import com.xebialabs.xlrelease.views.users.UserFilters
import org.springframework.jdbc.core.namedparam.NamedParameterJdbcTemplate

class SqlUserProfileQueryBuilder(val dialect: Dialect, val namedTemplate: NamedParameterJdbcTemplate)
  extends UserProfileQueryBuilder
    with FilterQueryBuilderSupport[UserFilters, UserProfile]
    with CompressionSupport
    with UserProfileMapper {

  implicit private val databaseDialect: Dialect = dialect

  import WhereClause._
  import com.xebialabs.xlrelease.repository.sql.persistence.Utils._

  override def withSortParameters(sorts: (String, String)*): SelfType = {
    this.sortOrderMapping ++= sorts
    this
  }

  override def build(): PageableQuery[UserProfile] = {
    withSortParameters(
      "fullName" -> USER_PROFILE.FULL_NAME,
      "username" -> USER_PROFILE.USERNAME,
      "lastActive" -> USER_PROFILE.LAST_ACTIVE,
      "loginAllowed" -> USER_PROFILE.ENABLED
    )
    buildSortOrderClause(s"$profileAlias.${USER_PROFILE.USERNAME}")
    val resultsQueryString = pageableQuery(queryTemplate)
    val resultsQuery = new SqlListQuery[UserProfile](namedTemplate, resultsQueryString, queryParams.toMap, userProfileRowMapper)
    val totalCountQuery = new SqlQuery[Long](namedTemplate, totalQueryTemplate, queryParams.toMap, (rs, _) => rs.getLong(1))
    new SqlPageableQuery[UserProfile](namedTemplate, this.pageable, resultsQuery, totalCountQuery)
  }

  override def from(filter: UserFilters): SelfType = {
    for {
      wc <- WhereClause()
      wc <- trim(filter.username).fold(wc)(username => wc.orLike(s"$profileAlias.${USER_PROFILE.USERNAME}", USER_PROFILE.USERNAME, username))
      wc <- trim(filter.email).fold(wc)(email => wc.orLike(s"$profileAlias.${USER_PROFILE.EMAIL}", USER_PROFILE.EMAIL, email))
      wc <- trim(filter.fullName).fold(wc)(fullName => wc.orLike(s"$profileAlias.${USER_PROFILE.FULL_NAME}", USER_PROFILE.FULL_NAME, fullName))
      wc <- if (filter.onlyDisabled) {
        wc.and(s"$profileAlias.${USER_PROFILE.ENABLED} = :${USER_PROFILE.ENABLED}", USER_PROFILE.ENABLED -> false.asInteger)
      } else {
        wc
      }
    } {
      if (wc.nonEmpty) {
        whereClauses.addOne(wc.clause)
        queryParams.addAll(wc.params)
      }
    }
    this
  }

  private def queryTemplate =
    s"""
       |SELECT
       |  $profileAlias.${USER_PROFILE.USERNAME},
       |  $profileAlias.${USER_PROFILE.EMAIL},
       |  $profileAlias.${USER_PROFILE.FULL_NAME},
       |  $profileAlias.${USER_PROFILE.ENABLED},
       |  $profileAlias.${USER_PROFILE.CONTENT},
       |  $profileAlias.${USER_PROFILE.LAST_ACTIVE}
       | FROM ${USER_PROFILE.TABLE} $profileAlias
       | $whereClause
       | $orderClause
       |""".stripMargin

  private def totalQueryTemplate =
    s"""
       |SELECT COUNT(1)
       | FROM ${USER_PROFILE.TABLE} $profileAlias
       | $whereClause
       |""".stripMargin
}

object SqlUserProfileQueryBuilder {
  val profileAlias = "up"
}