package com.xebialabs.xlrelease.scheduler.logs

import com.xebialabs.xlrelease.db.sql.SqlBuilder
import com.xebialabs.xlrelease.db.sql.SqlBuilder.Dialect
import com.xebialabs.xlrelease.domain.status.TaskStatus
import com.xebialabs.xlrelease.domain.status.TaskStatus._
import com.xebialabs.xlrelease.repository.query._
import com.xebialabs.xlrelease.repository.sql.persistence.Schema.{TASKS, TASK_EXECUTIONS}
import com.xebialabs.xlrelease.repository.sql.persistence.TaskPersistence.hash
import com.xebialabs.xlrelease.repository.sql.persistence.Utils.RichInstantAsTimestamp
import com.xebialabs.xlrelease.scheduler.logs.TaskExecutionRepository._
import org.springframework.jdbc.core.namedparam.NamedParameterJdbcTemplate

object SqlTaskExecutionQueryBuilder {
  def apply(dialect: SqlBuilder.Dialect, template: NamedParameterJdbcTemplate): SqlTaskExecutionQueryBuilder = {
    new SqlTaskExecutionQueryBuilder(dialect, template)
  }
}

class SqlTaskExecutionQueryBuilder(val dialect: Dialect, val namedTemplate: NamedParameterJdbcTemplate)
  extends FiltersQueryBuilder[FindOperation, TaskExecutionEntry]
    with FilterQueryBuilderSupport[FindOperation, TaskExecutionEntry]
    with TaskExecutionRowMapper {

  override def from(findOperation: FindOperation): FiltersQueryBuilder[FindOperation, TaskExecutionEntry] = {
    findOperation match {
      case ByTaskId(taskId) =>
        this.whereClauses += s"te.${TASK_EXECUTIONS.TASK_ID_HASH} = :taskIdHash"
        this.queryParams += "taskIdHash" -> hash(taskId)
      case FinishedTaskAndExecutions(comparator, lastModifiedDate) =>
        val operator = comparator match {
          case GreaterThan => ">"
          case LessThan => "<"
          case _ => throw new IllegalArgumentException(s"$comparator is not supported")
        }
        val finishedTaskStatuses: Array[String] = Array[TaskStatus](COMPLETED, SKIPPED).map(_.value())
        // Find all the task execution for the tasks which are completed, skipped or archived
        this.joinClauses +=
          s"""
             | LEFT JOIN ${TASKS.TABLE} t ON t.${TASKS.TASK_ID_HASH} = te.${TASK_EXECUTIONS.TASK_ID_HASH}
             |""".stripMargin
        this.whereClauses += s"te.${TASK_EXECUTIONS.LAST_MODIFIED_DATE} $operator :lastModifiedDate"
        this.whereClauses += s"(t.${TASKS.STATUS} IS NULL OR t.${TASKS.STATUS} IN (${finishedTaskStatuses.mkString("'", "','", "'")}))"
        this.queryParams += "lastModifiedDate" -> lastModifiedDate.asTimestamp
    }
    this
  }

  override def build(): PageableQuery[TaskExecutionEntry] = {
    remapSortOrderParams()
    buildSortOrderClause("lastModifiedDate")
    val resultsQueryString = pageableQuery(queryTemplate)
    val resultsQuery = new SqlListQuery[TaskExecutionEntry](namedTemplate, resultsQueryString, queryParams.toMap, taskExecutionRowMapper)
    val totalCountQuery = new SqlQuery[Long](namedTemplate, totalQueryTemplate, queryParams.toMap, (rs, _) => rs.getLong(1))
    new SqlPageableQuery[TaskExecutionEntry](namedTemplate, this.pageable, resultsQuery, totalCountQuery)
  }

  private lazy val queryTemplate =
    s"""
       |SELECT
       | te.${TASK_EXECUTIONS.TASK_ID_HASH},
       | te.${TASK_EXECUTIONS.EXECUTION_ID},
       | te.${TASK_EXECUTIONS.LAST_JOB},
       | te.${TASK_EXECUTIONS.LAST_CHUNK},
       | te.${TASK_EXECUTIONS.LAST_MODIFIED_DATE},
       | te.${TASK_EXECUTIONS.END_DATE}
       | FROM ${TASK_EXECUTIONS.TABLE} te
       | $joinClause
       | $whereClause
       | $orderClause
       |""".stripMargin.linesIterator.filter(_.trim.nonEmpty).mkString(s"$NL")

  private lazy val totalQueryTemplate =
    s"""
       | SELECT COUNT(1)
       | FROM ${TASK_EXECUTIONS.TABLE} te
       | $joinClause
       | $whereClause
       |""".stripMargin.linesIterator.filter(_.trim.nonEmpty).mkString(s"$NL")

  private def remapSortOrderParams(): Unit = {
    this.withSortParameters(
      "taskIdHash" -> TASK_EXECUTIONS.TASK_ID_HASH,
      "executionId" -> TASK_EXECUTIONS.EXECUTION_ID,
      "lastJob" -> TASK_EXECUTIONS.LAST_JOB,
      "lastChunk" -> TASK_EXECUTIONS.LAST_CHUNK,
      "lastModifiedDate" -> TASK_EXECUTIONS.LAST_MODIFIED_DATE,
      "endDate" -> TASK_EXECUTIONS.END_DATE
    )
  }
}
