package com.xebialabs.xlrelease.service

import com.codahale.metrics.annotation.Timed
import com.xebialabs.deployit.checks.Checks.checkArgument
import com.xebialabs.deployit.plugin.api.reflect.Type
import com.xebialabs.xlrelease.customscripts.ScriptTypes
import com.xebialabs.xlrelease.db.sql.transaction.IsTransactional
import com.xebialabs.xlrelease.domain._
import com.xebialabs.xlrelease.domain.events.TaskUpdatedEvent
import com.xebialabs.xlrelease.domain.status.TaskStatus
import com.xebialabs.xlrelease.domain.utils.TaskTypes
import com.xebialabs.xlrelease.events.XLReleaseEventBus
import com.xebialabs.xlrelease.repository.TaskRepository
import com.xebialabs.xlrelease.repository.sql.persistence.DependencyPersistence
import com.xebialabs.xlrelease.service.SqlTaskTypeConversion._
import com.xebialabs.xlrelease.service.TaskTypeConverterSyntax._
import com.xebialabs.xlrelease.service.TaskTypeConverters._
import org.springframework.stereotype.Component

import scala.jdk.CollectionConverters._

case class NewType(ciType: Type, customScriptType: Option[Type] = None, containerType: Option[Type] = None)

object SqlTaskTypeConversion {
  private val TYPE_CONTAINER_TASK: Type = Type.valueOf(classOf[ContainerTask])
  private val TYPE_CUSTOM_SCRIPT_TASK: Type = Type.valueOf(classOf[CustomScriptTask])
  private val TYPE_TASK: Type = Type.valueOf(classOf[Task])
  private val TYPE_TASK_GROUP: Type = Type.valueOf(classOf[TaskGroup])
}

@IsTransactional
@Component
class SqlTaskTypeConversion(scriptTypes: ScriptTypes,
                            taskRepository: TaskRepository,
                            eventBus: XLReleaseEventBus,
                            dependencyPersistence: DependencyPersistence,
                            converterService: TaskTypeConverterService
                           ) extends TaskTypeConversion {

  @Timed
  override def changeActiveTaskType(task: Task, targetType: Type): Task = {
    checkTaskIsPlanned(task)
    val newType = resolveNewType(targetType)
    checkTypeChangeIsSupported(newType, task)

    val ciTypeHasNotChanged = task.getType == newType.ciType
    val isNotCustomScriptTypeOrItHasNotChanged = task.getTaskType == newType.customScriptType.orNull
    if (ciTypeHasNotChanged && isNotCustomScriptTypeOrItHasNotChanged) {
      task
    } else {
      val updatedTask = newType.ciType match {
        case TYPE_CUSTOM_SCRIPT_TASK => task.convertTo[CustomScriptTask](newType, converterService)
        case TYPE_CONTAINER_TASK => task.convertTo[ContainerTask](newType, converterService)
        case _ => task.convertTo[Task](newType, converterService)
      }

      purgeGateTaskDependencies(task)
      val result = taskRepository.updateType(updatedTask)
      eventBus.publish(TaskUpdatedEvent(task, result))

      result
    }
  }

  private def checkTypeChangeIsSupported(newType: NewType, oldTask: Task): Unit = {
    val newTypeIsTaskType = newType.ciType.instanceOf(TYPE_TASK)
    checkArgument(isPythonScript(newType.ciType) || newTypeIsTaskType,
      s"Cannot change type of task '${oldTask.getTitle}' to '$newType' because it is not one of supported types: ${getAllSupportedTypes.mkString(",")}")

    if (oldTask.getType.isSubTypeOf(TYPE_TASK_GROUP)) {
      throw new IllegalArgumentException(s"Conversion of task '${oldTask.getTitle}' from '${oldTask.getType}' is not supported.")
    }
  }

  private def resolveNewType(targetType: Type): NewType = {
    if (isPythonScript(targetType)) {
      NewType(SqlTaskTypeConversion.TYPE_CUSTOM_SCRIPT_TASK, Some(targetType))
    } else if (isContainerTask(targetType)) {
      NewType(SqlTaskTypeConversion.TYPE_CONTAINER_TASK, None, Some(targetType))
    } else {
      NewType(targetType, None)
    }
  }

  private def isPythonScript(targetType: Type): Boolean = {
    scriptTypes.getPythonScriptTypes.contains(targetType)
  }

  private def isContainerTask(targetType: Type): Boolean = {
    scriptTypes.getContainerTaskTypes.contains(targetType)
  }

  private def checkTaskIsPlanned(task: Task): Unit = {
    checkArgument(task.isPlanned, "Cannot change type of the task '%s' because it is in state %s and not %s", task.getTitle, task.getStatus, TaskStatus.PLANNED)
  }

  private def getAllSupportedTypes: Seq[Type] = {
    TaskTypes.getDefaultTaskTypes.asScala ++ scriptTypes.getPythonScriptTypes.asScala ++ scriptTypes.getContainerTaskTypes.asScala
  }.toSeq

  private def purgeGateTaskDependencies(task: Task): Unit = task match {
    case gateTask: GateTask => gateTask.getDependencies.asScala.foreach(dependencyPersistence.deleteDependency)
    case _ =>
  }
}
