package com.xebialabs.xlrelease.repository.sql.persistence

import com.xebialabs.deployit.plugin.api.reflect.Type
import com.xebialabs.xlrelease.db.sql.SqlBuilder.Dialect
import com.xebialabs.xlrelease.db.sql.{Sql, SqlBuilder}
import com.xebialabs.xlrelease.domain.ReleaseKind
import com.xebialabs.xlrelease.domain.status.{ReleaseStatus, TaskStatus}
import com.xebialabs.xlrelease.repository.Ids
import com.xebialabs.xlrelease.repository.sql.persistence.Schema._
import com.xebialabs.xlrelease.security.XLReleasePermissions.VIEW_RELEASE
import com.xebialabs.xlrelease.security.sql.db.SecuritySchema.{ROLES, ROLE_PERMISSIONS, ROLE_PRINCIPALS, ROLE_ROLES}
import com.xebialabs.xlrelease.utils.DateVariableUtils

import java.util.Date

object TasksSqlBuilder {

  import com.xebialabs.xlrelease.repository.sql.persistence.Utils._

  def normalizeTags(tags: Set[String]): Set[String] = {
    tags.map(normalizeTag)
  }

  def normalizeTag(tag: String): String = {
    tag.toLowerCase().truncate(COLUMN_LENGTH_SHORT_VARCHAR)
  }
}

class TasksSqlBuilder(implicit dialect: Dialect) extends SqlBuilder[TasksSqlBuilder] {

  import com.xebialabs.xlrelease.repository.sql.persistence.TasksSqlBuilder._

  def selectTaskId(): TasksSqlBuilder = {
    super.select(
      s"""SELECT
         |  DISTINCT tasks.${TASKS.TASK_ID}
         |FROM
         |  ${TASKS.TABLE} tasks
         |  JOIN ${RELEASES.TABLE} releases ON tasks.${TASKS.RELEASE_UID} = releases.${RELEASES.CI_UID}
         """.stripMargin
    )
  }

  def withOneOfReleaseStatuses(statuses: Seq[ReleaseStatus]): TasksSqlBuilder = {
    if (statuses.nonEmpty) {
      conditions += Sql(s"releases.${RELEASES.STATUS} IN (${binding(statuses)})",
        statuses.map(_.value()))
    }
    this
  }

  def withReleaseKind(kind: ReleaseKind): TasksSqlBuilder = {
    conditions += Sql(s"releases.${RELEASES.KIND} = ?", Seq(kind.value()))
    this
  }

  def withTaskTypeNot(taskType: Type): TasksSqlBuilder = {
    conditions += Sql(s"tasks.${TASKS.TASK_TYPE} != ?", Seq(taskType.toString))
    this
  }

  def withTaskTypes(taskTypes: Seq[Type]): TasksSqlBuilder = {
    conditions += Sql(s"tasks.${TASKS.TASK_TYPE} IN (${binding(taskTypes)})", taskTypes.map(_.toString))
    this
  }

  def withOneOfTaskStatuses(statuses: Seq[TaskStatus]): TasksSqlBuilder = {
    if (statuses.nonEmpty) {
      conditions += Sql(s"tasks.${TASKS.STATUS} IN (${binding(statuses)})",
        statuses.map(_.value()))
    }
    this
  }

  def withFilter(filter: String): TasksSqlBuilder = {
    if (filter != null) {
      likeOr(Seq(s"tasks.${TASKS.TITLE}", s"tasks.${TASKS.OWNER}", s"tasks.${TASKS.TEAM}", s"releases.${RELEASES.RELEASE_TITLE}"), filter)
    }
    this
  }

  def withDates(from: Date, to: Date): TasksSqlBuilder = {
    if (from != null && to != null) {
      conditions += Sql(s"(? <= tasks.${TASKS.START_DATE} OR ? <= tasks.${TASKS.END_DATE}) AND tasks.${TASKS.START_DATE} < ?", Seq(from, from, to))
    } else {
      if (from != null) {
        conditions += Sql(s"(? <= tasks.${TASKS.START_DATE} OR ? <= tasks.${TASKS.END_DATE})", Seq(from, from))
      }
      if (to != null) {
        conditions += Sql(s"tasks.${TASKS.START_DATE} < ?", Seq(to))
      }
    }

    this
  }

  def withScheduledStartDate(date: Date): TasksSqlBuilder = {
    conditions += Sql(
      s"""
         |tasks.${TASKS.START_DATE} = ?
         |""".stripMargin, Seq(
        new Date(DateVariableUtils.truncateMilliseconds(date.getTime))));
    this
  }

  def withAnyOfTags(tags: Set[String]): TasksSqlBuilder = {
    val normalizedTags = normalizeTags(tags)
    if (normalizedTags.nonEmpty) {
      val sql =
        s"""
           |tasks.${TASKS.CI_UID} IN (
           |  SELECT DISTINCT task_tags.${TASK_TAGS.CI_UID}
           |  FROM ${TASK_TAGS.TABLE} task_tags
           |  WHERE task_tags.${TASK_TAGS.VALUE} IN (${binding(normalizedTags)})
           |)
           |""".stripMargin
      conditions += Sql(sql, normalizedTags)
    }
    this
  }

  def withFolderIds(folderIds: Set[String]): TasksSqlBuilder = {
    val normalizedFolderIds = folderIds.map(Ids.getName)
    if (normalizedFolderIds.nonEmpty) {
      joinFolders()
      conditions += Sql(
        s"folders.${FOLDERS.FOLDER_ID} IN (${binding(normalizedFolderIds)})",
        normalizedFolderIds
      )
    }
    this
  }

  def withAssignedToMe(principals: Iterable[String]): TasksSqlBuilder = {
    if (principals.nonEmpty) {
      conditions += Sql(
        s"LOWER(tasks.${TASKS.OWNER}) IN (${binding(principals)})",
        principals.map(_.toLowerCase)
      )
    } else {
      nothingToBeFound = true
    }
    this
  }

  def withAssignedToMeOrMyTeams(principals: Iterable[String], roleIds: Iterable[String]): TasksSqlBuilder = {
    joinTeamsToTasks()

    var orConditions = Seq[Sql]()

    if (principals.nonEmpty) {
      orConditions ++= Seq(Sql(
        s"LOWER(tasks.${TASKS.OWNER}) IN (${binding(principals)})",
        principals.map(_.toLowerCase)
      ))
      orConditions ++= Seq(Sql(
        s"teams.${ROLES.id} IN (SELECT ${ROLE_PRINCIPALS.roleId} FROM ${ROLE_PRINCIPALS.TABLE} assignedToMeOrTeamsPrincipals WHERE ${ROLE_PRINCIPALS.roleId} LIKE 'Team%' AND LOWER(assignedToMeOrTeamsPrincipals.${ROLE_PRINCIPALS.principalName}) IN (${binding(principals)}))",
        principals.map(_.toLowerCase)
      ))
    }

    if (roleIds.nonEmpty) {
      orConditions ++= Seq(Sql(
        s"teams.${ROLES.id} IN (SELECT ${ROLE_ROLES.roleId} FROM ${ROLE_ROLES.TABLE} assignedToMeOrTeamsRoles WHERE assignedToMeOrTeamsRoles.${ROLE_ROLES.memberRoleId} IN (${binding(roleIds)}))",
        roleIds
      ))
    }

    if (orConditions.nonEmpty) {
      conditions += Sql(
        s"(${orConditions.map(c => s"(${c.sql})").mkString(" OR ")})",
        orConditions.flatMap(_.parameters)
      )
    } else {
      conditions += Sql("(1 = 0)", Seq())
    }
    this
  }

  def withAssignedToAnybody(principals: Iterable[String], roleIds: Iterable[String], isAdmin: Boolean): TasksSqlBuilder = {
    if (!isAdmin) {
      var orConditions = Seq[Sql]()
      if (roleIds.nonEmpty) {
        orConditions ++= Seq(Sql(s"(assignedToTeamRoles.MEMBER_ROLE_ID IN (${binding(roleIds)}))", roleIds))
      }
      if (principals.nonEmpty) {
        orConditions ++= Seq(Sql(s"(LOWER(assignedToTeamPrincipals.PRINCIPAL_NAME) IN (${binding(principals)}))", principals.map(_.toLowerCase())))
      }
      if (orConditions.isEmpty) {
        conditions += Sql("(1 = 0)", Seq())
      } else {
        conditions += Sql(
          s"""releases.${RELEASES.SECURITY_UID} IN (
             |  SELECT assignedToTeams.${ROLES.ciId} FROM ${ROLES.TABLE} assignedToTeams
             |  JOIN ${ROLE_PERMISSIONS.TABLE} assignedToTeamPermissions ON assignedToTeams.${ROLES.id} = assignedToTeamPermissions.${ROLE_PERMISSIONS.roleId}
             |  LEFT JOIN ${ROLE_PRINCIPALS.TABLE} assignedToTeamPrincipals ON assignedToTeams.${ROLES.id} = assignedToTeamPrincipals.${ROLE_PRINCIPALS.roleId}
             |  LEFT JOIN ${ROLE_ROLES.TABLE} assignedToTeamRoles ON assignedToTeams.${ROLES.id} = assignedToTeamRoles.${ROLE_ROLES.roleId}
             |  WHERE assignedToTeamPermissions.${ROLE_PERMISSIONS.permissionName} = '${VIEW_RELEASE.getPermissionName}' AND (${orConditions.map(_.sql).mkString(" OR ")}))""".stripMargin,
          orConditions.flatMap(_.parameters)
        )
      }
    }
    this
  }

  def withAnyFacets(): TasksSqlBuilder = {
    joinFacets()
    this
  }

  def orderByTaskId(): TasksSqlBuilder = {
    this.orderBy(s"tasks.${TASKS.TASK_ID}")
    this
  }

  private def binding(values: Iterable[AnyRef]): String = {
    values.toSeq.map(_ => "?").mkString(",")
  }

  private def joinTeams(): Unit =
    addJoin(s"JOIN ${ROLES.TABLE} teams ON releases.${RELEASES.SECURITY_UID} = teams.${ROLES.ciId}")

  private def joinTeamsToTasks(): Unit =
    addJoin(s"LEFT JOIN ${ROLES.TABLE} teams ON releases.${RELEASES.SECURITY_UID} = teams.${ROLES.ciId} AND tasks.${TASKS.TEAM} = teams.${ROLES.name}")

  private def joinTeamRoles(): Unit =
    addJoin(s"LEFT JOIN ${ROLE_ROLES.TABLE} teamRoles ON teams.${ROLES.id} = teamRoles.${ROLE_ROLES.roleId}")

  private def joinTeamPrincipals(): Unit =
    addJoin(s"LEFT JOIN ${ROLE_PRINCIPALS.TABLE} teamPrincipals ON teams.${ROLES.id} = teamPrincipals.${ROLE_PRINCIPALS.roleId}")

  private def joinTeamPermissions(): Unit =
    addJoin(s"JOIN ${ROLE_PERMISSIONS.TABLE} teamPermissions ON teams.${ROLES.id} = teamPermissions.${ROLE_PERMISSIONS.roleId}")

  private def joinFolders(): Unit = {
    addJoin(s"JOIN ${FOLDERS.TABLE} folders ON releases.${RELEASES.FOLDER_CI_UID} = folders.${FOLDERS.CI_UID}")
  }

  private def joinFacets(): Unit = {
    addJoin(s"JOIN ${FACETS.TABLE} facets ON tasks.${TASKS.CI_UID} = facets.${FACETS.APPLIED_TO_TASK_UID}")
  }

  override def newInstance: TasksSqlBuilder = new TasksSqlBuilder()

}
