package com.xebialabs.xlrelease.repository.sql

import com.codahale.metrics.annotation.Timed
import com.xebialabs.deployit.checks.Checks.checkArgument
import com.xebialabs.deployit.plugin.api.reflect.Type
import com.xebialabs.deployit.security.permission.Permission
import com.xebialabs.deployit.security.{PermissionEditor, Role}
import com.xebialabs.xlrelease.domain.Team
import com.xebialabs.xlrelease.repository.Ids._
import com.xebialabs.xlrelease.repository._
import com.xebialabs.xlrelease.repository.sql.persistence._
import com.xebialabs.xlrelease.security.SecuredCi
import com.xebialabs.xlrelease.security.sql.SqlRoleService
import grizzled.slf4j.Logging
import org.springframework.util.StringUtils.isEmpty

import java.util
import scala.jdk.CollectionConverters._

private[sql] trait PlatformSecurityMappingSupport {

  protected def fromPlatformRole(containerId: String, role: Role, permissions: Seq[Permission]): Team = {
    val team: Team = Type.valueOf(classOf[Team]).getDescriptor.newInstance(fromPlatformId(containerId, role.getId))
    team.setRoles(role.getRoles)
    team.setMembers(role.getPrincipals)
    team.setTeamName(role.getName)
    team.setPermissions(permissions.collect { case a:Permission => a.getPermissionName}.asJavaMutable())
    team
  }

  protected def toPlatformRole(team: Team): Role = {
    checkArgument(!isEmpty(team.getTeamName), s"Team name is missing in team ${team.getId}")
    val role = new Role(toPlatformId(team.getId), team.getTeamName)
    role.setRoles(team.getRoles)
    role.setPrincipals(team.getMembers)
    role
  }

  protected def toPlatformId(id: String): String = Ids.getName(id)

  protected def fromPlatformId(containerId: String, id: String): String = containerId + "/" + id
}

class SqlTeamRepository(val permissionEditor: PermissionEditor,
                        val roleService: SqlRoleService,
                        val securedCis: SecuredCis) extends TeamRepository with PlatformSecurityMappingSupport with Logging {

  @Timed
  override def create(containerId: String, team: Team): Team = {
    removeMissingRoles(team)

    val role = toPlatformRole(team)
    val securityUid = securedCis.getSecuredCi(containerId).getSecurityUid

    roleService.create(securityUid, role)
    updatePermissions(securityUid, role -> team.getPermissions)
    updateInheritanceFlag(containerId, Seq(team))
    team
  }

  @Timed
  override def update(team: Team): Team = {
    val containerId = getParentId(team.getId)
    removeMissingRoles(team)

    val role = toPlatformRole(team)
    val securityUid = securedCis.getSecuredCi(containerId).getSecurityUid

    roleService.update(securityUid, role)
    updatePermissions(securityUid, role -> team.getPermissions)
    team
  }

  @Timed
  override def getTeams(containerId: String): util.List[Team] = getTeams(securedCis.getSecuredCi(containerId))

  @Timed
  override def getTeams(ci: SecuredCi): util.List[Team] = {
    val roles = roleService.readRoleAssignments(ci.getSecurityUid).asScala.toList

    (roles match {
      case Nil => Seq.empty
      case _ =>
        val permissionsByRole = permissionEditor.readPermissions(ci.getSecurityUid)
        roles.map(role => fromPlatformRole(ci.getId, role, permissionsByRole.getOrDefault(role, new util.HashSet[Permission]()).asScala.toSeq))
    }).asJavaMutable()
  }

  @Timed
  override def delete(teamId: String): Unit = {
    val containerId = getParentId(teamId)
    val securityUid = securedCis.getSecuredCi(containerId).getSecurityUid
    roleService.deleteRoles(securityUid, toPlatformId(teamId))
    updateInheritanceFlag(containerId, getTeams(containerId).asScala.toSeq)
  }

  @Timed
  override def deleteTeamsFromPlatform(containerId: String): Unit = deleteTeamsFromPlatform(securedCis.getSecuredCi(containerId))

  @Timed
  override def deleteTeamsFromPlatform(container: SecuredCi): Unit = {
    val roles = roleService.getRoles(container.getSecurityUid).asScala.toSeq
    roleService.deleteRoles(container.getSecurityUid, roles.map(_.getId): _*)
    if (isFolderId(container.getId)) {
      securedCis.setAsEffectiveSecuredCi(container.getId, isEffective = false)
    }
  }

  @Timed
  override def saveTeamsToPlatform(containerId: String, teams: util.List[Team]): util.List[Team] =
    saveTeamsToPlatform(containerId, teams.asScala.toSeq).asJavaMutable()

  private def saveTeamsToPlatform(containerId: String, teams: Seq[Team]): Seq[Team] = {
    removeMissingRoles(teams: _*)

    val securityUid = securedCis.getSecuredCi(containerId).getSecurityUid

    val roles = teams.map(team => toPlatformRole(team) -> team.getPermissions)
    roleService.writeRoleAssignments(securityUid, roles.map(_._1).asJava)
    updatePermissions(securityUid, roles: _*)

    updateInheritanceFlag(containerId, teams)

    teams
  }

  private def updatePermissions(securityUid: String, rolesWithPermissions: (Role, util.List[String])*): Unit = {
    val permissions = permissionEditor.readPermissions(securityUid)
    rolesWithPermissions.foreach {
      case (role, rolePermissions) => permissions.put(role, rolePermissions.asScala.map(Permission.find).toSet.asJava)
    }
    permissionEditor.editPermissions(securityUid, permissions)
  }

  private def updateInheritanceFlag(containerId: String, teams: Seq[Team]): Unit = {
    securedCis.setAsEffectiveSecuredCi(containerId, teams.nonEmpty)
  }

  private def removeMissingRoles(teams: Team*): Unit = {
    lazy val availableRoles = roleService.getRoles().asScala

    teams.foreach { team =>
      val nonExistingRoles = team.getRoles.asScala.filter(roleName => !availableRoles.exists(_.getName == roleName))
      if (nonExistingRoles.nonEmpty) {
        logger.warn(s"The following role(s) do not exist: $nonExistingRoles. They were removed from the team ${team.getTeamName} [${team.getId}].")
        team.getRoles.removeAll(nonExistingRoles.asJava)
      }
    }
  }

}
