package com.xebialabs.xlrelease.repository.sql

import com.codahale.metrics.annotation.Timed
import com.xebialabs.deployit.exception.NotFoundException
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem
import com.xebialabs.xlrelease.db.sql.transaction.{IsReadOnly, IsTransactional}
import com.xebialabs.xlrelease.domain.Release
import com.xebialabs.xlrelease.domain.facet.Facet
import com.xebialabs.xlrelease.domain.variables.Variable
import com.xebialabs.xlrelease.repository.Ids.releaseIdFrom
import com.xebialabs.xlrelease.repository.sql.persistence.CiId._
import com.xebialabs.xlrelease.repository.{FacetRepositoryDispatcher, ReleaseRepository, ReleaseVariableRepository}
import com.xebialabs.xlrelease.serialization.json.repository.ResolveOptions
import com.xebialabs.xlrelease.variable.VariableHelper.withoutVariableSyntax
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Repository

@Repository
@IsTransactional
class SqlReleaseVariableRepository @Autowired()(val releaseRepository: ReleaseRepository, val facetRepositoryDispatcher: FacetRepositoryDispatcher)
  extends ReleaseVariableRepository {

  @Timed
  override def create(variable: Variable, updatedRelease: Release): Variable = {
    releaseRepository.update(updatedRelease)
    variable
  }

  @Timed
  override def create(variables: Seq[Variable], updatedRelease: Release): Unit = {
    releaseRepository.update(updatedRelease)
  }

  @Timed
  override def update(original: Variable, updated: Variable, updatedRelease: Release): Unit = {
    releaseRepository.update(updatedRelease)
  }

  @Timed
  override def update(originalList: Seq[Variable], updatedList: Seq[Variable], updatedRelease: Release): Unit = {
    releaseRepository.update(updatedRelease)
  }

  @Timed
  override def delete(variableId: String, updatedRelease: Release): Unit = {
    releaseRepository.update(updatedRelease)
  }

  @Timed
  override def replace(updatedRelease: Release, updatedReleaseCis: Seq[ConfigurationItem]): Unit = {
    releaseRepository.update(updatedRelease)
    val facetCis = updatedReleaseCis.filter(_.getType.getDescriptor.isAssignableTo(classOf[Facet])).map(_.asInstanceOf[Facet])
    facetCis.foreach { facetCi =>
      facetRepositoryDispatcher.liveRepository.update(facetCi)
    }
  }

  @Timed
  @IsReadOnly
  override def exists(variableId: String): Boolean = {
    if (releaseRepository.exists(releaseIdFrom(variableId))) {
      val release = releaseRepository.findById(releaseIdFrom(variableId.normalized))
      release.getVariableById(variableId.normalized).isPresent
    } else {
      false
    }
  }

  @Timed
  override def findById(variableId: String): Variable = {
    val release = releaseRepository.findById(releaseIdFrom(variableId), ResolveOptions.WITHOUT_DECORATORS)
    val maybeVariable = release.getVariableById(variableId.normalized)
    maybeVariable.orElseThrow(() => new NotFoundException("Repository entity [%s] not found", variableId))
  }

  @Timed
  override def findByKey(key: String, releaseId: String): Variable = {
    val release = releaseRepository.findById(releaseId)
    release.getVariablesByKeys.get(withoutVariableSyntax(key))
  }

}
