package com.xebialabs.xlrelease.repository.sql

import com.xebialabs.deployit.checks.Checks.IncorrectArgumentException
import com.xebialabs.xlrelease.db.sql.transaction.IsTransactional
import com.xebialabs.xlrelease.domain.Release
import com.xebialabs.xlrelease.plugins.dashboard.domain.Dashboard
import com.xebialabs.xlrelease.repository.{FolderRepository, Ids, PersistenceInterceptor, ReleaseRepository}
import org.springframework.stereotype.Component
import org.springframework.util.StringUtils

import scala.jdk.CollectionConverters._

@Component
@IsTransactional
class ReleasePersistenceInterceptor(releaseRepository: ReleaseRepository,
                                    folderRepository: FolderRepository)
  extends PersistenceInterceptor[Release] {

  releaseRepository.registerPersistenceInterceptor(this)

  override def onCreate(ci: Release): Unit = {
    onCreateOrUpdate(ci)
  }

  override def onUpdate(ci: Release): Unit = {
    onCreateOrUpdate(ci)
  }

  private def onCreateOrUpdate(release: Release): Unit = {
    validateDefaultTargetFolder(release)
    validateWorkflowProperties(release)
  }

  private def validateDefaultTargetFolder(release: Release): Unit = {
    val doesNotAllowTargetFolderOverride = !release.getAllowTargetFolderOverride
    val defaultTargetFolderId = release.getDefaultTargetFolderId
    val defaultTargetFolderIdIsEmpty = !StringUtils.hasText(defaultTargetFolderId)
    if (release.isTemplate && doesNotAllowTargetFolderOverride && defaultTargetFolderIdIsEmpty) {
      val msg = s"Override of the target folder is not allowed, but default target folder is not provided for release with ID [${release.getId}]"
      throw new IncorrectArgumentException(msg)
    }
    if (!defaultTargetFolderIdIsEmpty) {
      val optionalFolder = folderRepository.findById(defaultTargetFolderId, depth = 1)
      if (optionalFolder.isEmpty) {
        throw new IncorrectArgumentException(s"Cannot find folder with id [$defaultTargetFolderId]")
      }
    }
  }

  private def validateWorkflowProperties(release: Release): Unit = {
    validateNotARootWorkflow(release)
    validateNoDashboardForWorkflow(release)
    validateNoLockTaskPresentForWorkflow(release)
  }

  private def validateNotARootWorkflow(release: Release): Unit = {
    val parentFolder = Ids.getParentId(release.getId)
    if (release.isWorkflow && parentFolder == Ids.ROOT_FOLDER_ID) {
      throw new IncorrectArgumentException("Workflows can be created only inside a folder")
    }
  }

  private def validateNoDashboardForWorkflow(release: Release): Unit = {
    if (release.isWorkflow) {
      val hasDashboards = release.getExtensions.asScala.exists(_.isInstanceOf[Dashboard])
      if (hasDashboards) {
        throw new IncorrectArgumentException("Dashboards are not supported for workflows")
      }
    }
  }

  private def validateNoLockTaskPresentForWorkflow(release: Release): Unit = {
    if (release.isWorkflow) {
      val hasLockTasks = release.getAllTasks.asScala.exists(_.isLocked)
      if (hasLockTasks) {
        throw new IncorrectArgumentException("Lock tasks are not supported for workflows")
      }
    }
  }
}
