package com.xebialabs.xlrelease.repository.sql.persistence

import com.xebialabs.deployit.repository.ItemAlreadyExistsException
import com.xebialabs.xlrelease.db.sql.transaction.{IsReadOnly, IsTransactional}
import com.xebialabs.xlrelease.domain.TemplateLogo
import com.xebialabs.xlrelease.repository.Ids.getFolderlessId
import com.xebialabs.xlrelease.repository.ReleaseJsonParser.IdExtension
import com.xebialabs.xlrelease.repository.{CiCloneHelper, TemplateMetadata}
import com.xebialabs.xlrelease.repository.sql.persistence.CiId.CiId
import com.xebialabs.xlrelease.repository.sql.persistence.Schema.TEMPLATE_METADATA
import com.xebialabs.xlrelease.repository.sql.persistence.Utils._
import com.xebialabs.xlrelease.serialization.json.utils.CiSerializerHelper
import com.xebialabs.xlrelease.utils.CiHelper.rewriteWithNewId
import org.springframework.dao.{DuplicateKeyException, EmptyResultDataAccessException}
import org.springframework.jdbc.core.support.SqlLobValue
import org.springframework.jdbc.core.{RowMapper, SqlParameterValue}
import org.springframework.util.StreamUtils

import java.nio.charset.StandardCharsets
import java.sql.Types
import scala.util.Using

@IsTransactional
trait TemplateMetadataPersistence { self: ReleasePersistence =>

  private val STMT_INSERT_TEMPLATE_META =
    s"""
       | INSERT INTO ${TEMPLATE_METADATA.TABLE}
       | ( ${TEMPLATE_METADATA.RELEASE_UID}
       | , ${TEMPLATE_METADATA.AUTHOR}
       | , ${TEMPLATE_METADATA.DESCRIPTION}
       | , ${TEMPLATE_METADATA.LOGO_DATA}
       | )
       | VALUES
       | ( :${TEMPLATE_METADATA.RELEASE_UID}
       | , :${TEMPLATE_METADATA.AUTHOR}
       | , :${TEMPLATE_METADATA.DESCRIPTION}
       | , :${TEMPLATE_METADATA.LOGO_DATA}
       | )
       |""".stripMargin

  def createTemplateMetadata(templateMeta: TemplateMetadata): Unit = {
    val templateLogo = templateMeta.templateLogo
    val logoBytes: Array[Byte] = serializeTemplateLogo(templateLogo)
    val descriptionBytes: Array[Byte] = getDescriptionBytes(templateMeta.description)

    val p = params(
      TEMPLATE_METADATA.RELEASE_UID -> templateMeta.releaseUid,
      TEMPLATE_METADATA.AUTHOR -> templateMeta.author,
      TEMPLATE_METADATA.DESCRIPTION -> new SqlParameterValue(Types.BLOB, new SqlLobValue(descriptionBytes)),
      TEMPLATE_METADATA.LOGO_DATA -> new SqlParameterValue(Types.BLOB, new SqlLobValue(logoBytes))
    )
    try {
      namedTemplate.update(STMT_INSERT_TEMPLATE_META, p)
    } catch {
      case _: DuplicateKeyException =>
        throw new ItemAlreadyExistsException(s"Template metadata already exists for ${templateMeta.releaseUid}")
    }

    if (templateLogo != null) {
      val artifactId = templateLogo.getId
      val artifactFile = templateLogo.getFile
      val artifactName = artifactFile.getName
      val content = artifactFile.getInputStream
      self.insertArtifact(templateMeta.releaseUid, artifactId, artifactName, content)
    }
  }

  private val STMT_UPDATE_TEMPLATE_META =
    s"""
       | UPDATE ${TEMPLATE_METADATA.TABLE}
       | SET
       |  ${TEMPLATE_METADATA.AUTHOR} = :${TEMPLATE_METADATA.AUTHOR},
       |  ${TEMPLATE_METADATA.DESCRIPTION} = :${TEMPLATE_METADATA.DESCRIPTION}
       | WHERE ${TEMPLATE_METADATA.RELEASE_UID} = :${TEMPLATE_METADATA.RELEASE_UID}
       |""".stripMargin

  def updateTemplateMetadata(releaseUid: CiUid, author: String, description: String): Boolean = {
    val descriptionBytes: Array[Byte] = getDescriptionBytes(description)
    val p = Map[String, Any](
      TEMPLATE_METADATA.RELEASE_UID -> releaseUid,
      TEMPLATE_METADATA.AUTHOR -> author,
      TEMPLATE_METADATA.DESCRIPTION -> new SqlParameterValue(Types.BLOB, new SqlLobValue(descriptionBytes))
    )
    val res = namedTemplate.update(STMT_UPDATE_TEMPLATE_META, p)
    res == 1
  }

  private val STMT_UPDATE_LOGO =
    s""" UPDATE ${TEMPLATE_METADATA.TABLE}
       | SET
       | ${TEMPLATE_METADATA.LOGO_DATA} = :${TEMPLATE_METADATA.LOGO_DATA}
       | WHERE ${TEMPLATE_METADATA.RELEASE_UID} = :${TEMPLATE_METADATA.RELEASE_UID}
       |""".stripMargin

  def updateLogo(releaseUid: CiUid, originalLogo: Option[TemplateLogo], updatedLogo: TemplateLogo): Boolean = {
    // delete the existing logo
    originalLogo.foreach { templateLogo =>
      self.deleteArtifactById(templateLogo.getId, releaseUid)
    }

    //insert new logo
    self.insertArtifact(releaseUid, updatedLogo.getId, updatedLogo.getFile.getName, updatedLogo.getFile.getInputStream)

    // update template metadata
    val logoBytes: Array[Byte] = serializeTemplateLogo(updatedLogo)
    val p = Map[String, Any](
      TEMPLATE_METADATA.RELEASE_UID -> releaseUid,
      TEMPLATE_METADATA.LOGO_DATA -> new SqlParameterValue(Types.BLOB, new SqlLobValue(logoBytes))
    )
    val res = namedTemplate.update(STMT_UPDATE_LOGO, p)
    res == 1
  }

  private val STMT_DELETE_TEMPLATE_META =
    s"DELETE FROM ${TEMPLATE_METADATA.TABLE} WHERE ${TEMPLATE_METADATA.RELEASE_UID} = :${TEMPLATE_METADATA.RELEASE_UID}"

  def deleteTemplateMetadata(releaseUid: CiUid): Boolean = {
    val res = namedTemplate.update(STMT_DELETE_TEMPLATE_META, params(TEMPLATE_METADATA.RELEASE_UID -> releaseUid))
    res == 1
  }

  private val STMT_FIND_BY_RELEASE_UID =
    s"""
       | SELECT
       | ${TEMPLATE_METADATA.RELEASE_UID}
       | , ${TEMPLATE_METADATA.AUTHOR}
       | , ${TEMPLATE_METADATA.DESCRIPTION}
       | , ${TEMPLATE_METADATA.LOGO_DATA}
       | FROM ${TEMPLATE_METADATA.TABLE}
       | WHERE ${TEMPLATE_METADATA.RELEASE_UID} = :${TEMPLATE_METADATA.RELEASE_UID}
       |""".stripMargin

  @IsReadOnly
  def findTemplateMetadata(releaseUid: CiUid): Option[TemplateMetadata] = {
    try {
      Option(namedTemplate.queryForObject(STMT_FIND_BY_RELEASE_UID,
        params(TEMPLATE_METADATA.RELEASE_UID -> releaseUid),
        toTemplateMeta
      ))
    } catch {
      case _: EmptyResultDataAccessException =>
        None
    }
  }

  @IsReadOnly
  def findTemplateLogoById(logoId: CiId): Option[TemplateLogo] = {
    loadSourceArtifact(logoId.normalized)(classOf[TemplateLogo])
  }

  private def serializeTemplateLogo(templateLogo: TemplateLogo): Array[Byte] = {
    if (templateLogo != null) {
      val copy = CiCloneHelper.cloneCi(templateLogo)
      // don't store folder info in the JSON so that it does not have to be updated on move operations
      rewriteWithNewId(copy, getFolderlessId(copy.getId))
      val logoJson = CiSerializerHelper.serialize(copy)
      logoJson.getBytes(StandardCharsets.UTF_8)
    } else {
      null
    }
  }

  private def getDescriptionBytes(description: String) = {
    val descriptionBytes: Array[Byte] = if (description != null) {
      description.getBytes(StandardCharsets.UTF_8)
    } else {
      null
    }
    descriptionBytes
  }

  private val toTemplateMeta: RowMapper[TemplateMetadata] = {
    case (rs, _) =>
      val releaseUid = rs.getInt(TEMPLATE_METADATA.RELEASE_UID)
      val author = rs.getString(TEMPLATE_METADATA.AUTHOR)
      val descriptionStream = rs.getBinaryStream(TEMPLATE_METADATA.DESCRIPTION)
      val description = if (descriptionStream == null) {
        null
      } else {
        Using.resource(descriptionStream) { is =>
          val logoBytes = StreamUtils.copyToByteArray(is)
          new String(logoBytes, StandardCharsets.UTF_8)
        }
      }

      val logoStream = rs.getBinaryStream(TEMPLATE_METADATA.LOGO_DATA)
      val templateLogo = if (logoStream == null) {
        null
      } else {
        Using.resource(logoStream) { is =>
          val logoBytes = StreamUtils.copyToByteArray(is)
          val logoJson = new String(logoBytes, StandardCharsets.UTF_8)
          val deserializedLogo = CiSerializerHelper.deserialize(logoJson, null).asInstanceOf[TemplateLogo]
          val templateLogoWithLoadedArtifact = self.loadSourceArtifact(deserializedLogo.getId)(classOf[TemplateLogo])
          templateLogoWithLoadedArtifact match {
            case Some(v) =>
              v.setContentType(deserializedLogo.getContentType)
              v.setExportFilename(deserializedLogo.getExportFilename)
              v
            case None => null
          }
        }
      }

      TemplateMetadata(releaseUid, author, description, templateLogo)
  }

}
