package com.xebialabs.xlrelease.repository.sql.persistence.reference

import com.xebialabs.xlrelease.repository.sql.persistence.{CiUid, PersistenceSupport}
import com.xebialabs.xlrelease.repository.sql.persistence.Schema.ReferenceMapTable
import grizzled.slf4j.Logging

import scala.jdk.CollectionConverters._

abstract class ReferenceMapTablePersistenceCommon[T <: ReferenceMapTable](table: T)
  extends ReferenceMapTablePersistence
    with PersistenceSupport
    with Logging {
  private val SOURCE_PARAM = "sourceId"
  private val TARGET_PARAM = "targetId"
  private val MULTI_TARGET_PARAM = "targetIds"
  private val MULTI_TARGET_DELETE_BATCH_SIZE = 512

  private val STMT_SELECT =
    s"""
      | SELECT ${table.sourceColumn}, ${table.targetColumn}
      | FROM ${table.TABLE}
      |""".stripMargin

  private val STMT_GET_All_TARGETS =
    s"""
       | $STMT_SELECT
       | WHERE ${table.sourceColumn} = :$SOURCE_PARAM
       |""".stripMargin

  private val STMT_INSERT: String =
    s"""
       |INSERT INTO ${table.TABLE} (
       |  ${table.sourceColumn},
       |  ${table.targetColumn}
       |) VALUES (
       |  :$SOURCE_PARAM,
       |  :$TARGET_PARAM
       |)
     """.stripMargin

  private lazy val STMT_DELETE: String =
    s"""
       |DELETE FROM ${table.TABLE}
       | WHERE
       |  ${table.sourceColumn} = :$SOURCE_PARAM
       |  AND ${table.targetColumn} = :$TARGET_PARAM
       |""".stripMargin

  private lazy val STMT_COUNT_REFS: String =
    s"SELECT COUNT(*) FROM ${table.TABLE} WHERE ${table.targetColumn} = :$TARGET_PARAM"

  private lazy val STMT_DELETE_SOURCE: String =
    s"""|DELETE FROM ${table.TABLE}
        | WHERE ${table.sourceColumn} = :$SOURCE_PARAM
     """.stripMargin

  private lazy val STMT_DELETE_TARGET: String =
    s"""|DELETE FROM ${table.TABLE}
        | WHERE ${table.targetColumn} = :$TARGET_PARAM
     """.stripMargin

  private lazy val STMT_DELETE_TARGETS: String =
    s"""|DELETE FROM ${table.TABLE}
        | WHERE ${table.targetColumn} IN (:$MULTI_TARGET_PARAM)
     """.stripMargin

  override def insert(sourceId: CiUid, targetIds: Set[CiUid]): Unit = {
    if(targetIds.nonEmpty) {
      sqlBatch(STMT_INSERT, targetIds.map(targetId => Map(SOURCE_PARAM -> sourceId, TARGET_PARAM -> targetId)))
    }
  }

  override def delete(sourceId: CiUid, targetIds: Set[CiUid]): Unit = {
    if(targetIds.nonEmpty) {
      sqlBatch(STMT_DELETE, targetIds.map(targetId => Map(SOURCE_PARAM -> sourceId, TARGET_PARAM -> targetId)))
    }
  }

  override def isReferenced(targetId: CiUid): Boolean = {
    sqlQuery(STMT_COUNT_REFS, Map(TARGET_PARAM -> targetId), _.getInt(1) > 0).head
  }

  override def getTargetIds(sourceId: CiUid): List[CiUid] = {
    sqlQuery(STMT_GET_All_TARGETS, Map(SOURCE_PARAM -> sourceId), _.getInt(table.targetColumn).asInstanceOf[CiUid]).toList
  }

  override def deleteRefsBySource(sourceId: CiUid): Unit = {
    sqlUpdate(STMT_DELETE_SOURCE, Map(SOURCE_PARAM -> sourceId), _ => ())
  }

  override def deleteRefsByTarget(targetId: CiUid): Unit = {
    sqlUpdate(STMT_DELETE_TARGET, Map(TARGET_PARAM -> targetId), _ => ())
  }

  override def deleteRefsByTargets(targetIds: Set[CiUid]): Unit = {
    targetIds.grouped(MULTI_TARGET_DELETE_BATCH_SIZE).foreach(ids => {
      sqlUpdate(STMT_DELETE_TARGETS, Map(MULTI_TARGET_PARAM -> ids.asJava), _ => ())
    })
  }
}
