package com.xebialabs.xlrelease.repository.sql.persistence

import com.xebialabs.deployit.exception.NotFoundException
import com.xebialabs.xlrelease.db.sql.SqlBuilder.Dialect
import com.xebialabs.xlrelease.db.sql.transaction.{IsReadOnly, IsTransactional}
import com.xebialabs.xlrelease.domain._
import com.xebialabs.xlrelease.repository.Ids
import com.xebialabs.xlrelease.repository.Ids._
import com.xebialabs.xlrelease.repository.sql.SqlRepository
import com.xebialabs.xlrelease.repository.sql.persistence.CiId._
import com.xebialabs.xlrelease.repository.sql.persistence.DependencyPersistence.TaskDependency
import com.xebialabs.xlrelease.repository.sql.persistence.ReleasesSqlBuilder.selectReleaseFullPathSql
import com.xebialabs.xlrelease.repository.sql.persistence.Schema.{DEPENDENCIES, FOLDERS, RELEASES, TASKS}
import com.xebialabs.xlrelease.repository.sql.persistence.Utils.params
import com.xebialabs.xlrelease.repository.sql.persistence.data.{DependencyRow, TargetRow}
import com.xebialabs.xlrelease.utils.FolderId
import grizzled.slf4j.Logging
import org.springframework.dao.EmptyResultDataAccessException
import org.springframework.jdbc.core.namedparam.{MapSqlParameterSource, SqlParameterSource}
import org.springframework.jdbc.core.{JdbcTemplate, RowMapper}

import java.sql.ResultSet
import scala.jdk.CollectionConverters._
import scala.util.{Failure, Success, Try}

@IsTransactional
class DependencyPersistence(val jdbcTemplate: JdbcTemplate,
                            val dialect: Dialect)
  extends SqlRepository with PersistenceSupport with Logging {

  private val BATCH_SIZE: Integer = 512;

  // builds a variable list of question marks string and the corresponding arguments Array
  private def varargs[T](args: Iterable[T]): (String, Array[AnyRef]) = args.toSeq match {
    case seq =>
      (seq.map(_ => "?").mkString(", "), seq.map(_.asInstanceOf[AnyRef]).toArray)
  }

  private def filterStatus(status: Option[Set[String]]): (String, Array[AnyRef]) = status.fold("" -> Array.empty[AnyRef])(statuses =>
    varargs(statuses) match {
      case (placeholdersStatus, argsStatus) =>
        s"AND d.${DEPENDENCIES.GATE_STATUS} IN ($placeholdersStatus)" -> argsStatus
    }
  )

  private val defaultMapper: RowMapper[DependencyRow] = (rs: ResultSet, _: Int) =>
    DependencyRow(
      folderId = FolderId(rs.getString(FOLDERS.FOLDER_PATH)) / rs.getString(FOLDERS.FOLDER_ID),
      taskId = rs.getString(TASKS.TASK_ID),
      dependencyId = rs.getString(DEPENDENCIES.DEPENDENCY_ID),
      gateStatus = rs.getString(DEPENDENCIES.GATE_STATUS),
      targetReleaseId = Option(rs.getString(DEPENDENCIES.TARGET_RELEASE_ID)).map(FolderId(_).absolute),
      targetId = rs.getString(DEPENDENCIES.TARGET_ID)
    )

  private val defaultQuery =
    s"""SELECT
       | fr.${FOLDERS.FOLDER_PATH},
       | fr.${FOLDERS.FOLDER_ID},
       | k.${TASKS.TASK_ID},
       | d.${DEPENDENCIES.DEPENDENCY_ID},
       | d.${DEPENDENCIES.GATE_STATUS},
       | d.${DEPENDENCIES.TARGET_ID},
       | ${selectReleaseFullPathSql(tableAlias = "t", folderTableAlias = "ft", columnAlias = DEPENDENCIES.TARGET_RELEASE_ID, dialect = dialect)}
       |  FROM ${DEPENDENCIES.TABLE} d
       |  JOIN ${TASKS.TABLE} k ON
       |    k.${TASKS.CI_UID} = d.${DEPENDENCIES.GATE_TASK_UID}
       |  JOIN ${RELEASES.TABLE} r ON
       |    r.${RELEASES.CI_UID} = k.${TASKS.RELEASE_UID}
       |  JOIN ${FOLDERS.TABLE} fr ON
       |    fr.${FOLDERS.CI_UID} = r.${RELEASES.FOLDER_CI_UID}
       |  JOIN ${RELEASES.TABLE} t ON
       |    t.${RELEASES.CI_UID} = d.${DEPENDENCIES.TARGET_RELEASE_UID}
       |  JOIN ${FOLDERS.TABLE} ft ON
       |    ft.${FOLDERS.CI_UID} = t.${RELEASES.FOLDER_CI_UID}""".stripMargin

  private val STMT_FIND_BY_GATE_RELEASE_UID =
    s"""SELECT
       | fr.${FOLDERS.FOLDER_PATH},
       | fr.${FOLDERS.FOLDER_ID},
       | k.${TASKS.TASK_ID},
       | d.${DEPENDENCIES.DEPENDENCY_ID},
       | d.${DEPENDENCIES.GATE_STATUS},
       | d.${DEPENDENCIES.TARGET_ID},
       | ${selectReleaseFullPathSql(tableAlias = "t", folderTableAlias = "ft", columnAlias = DEPENDENCIES.TARGET_RELEASE_ID, dialect = dialect)}
       |  FROM ${RELEASES.TABLE} r
       |  JOIN ${FOLDERS.TABLE} fr ON fr.${FOLDERS.CI_UID} = r.${RELEASES.FOLDER_CI_UID}
       |  JOIN ${TASKS.TABLE} k ON r.${RELEASES.CI_UID} = k.${TASKS.RELEASE_UID}
       |  JOIN ${DEPENDENCIES.TABLE} d ON k.${TASKS.CI_UID} = d.${DEPENDENCIES.GATE_TASK_UID}
       |  JOIN ${RELEASES.TABLE} t ON t.${RELEASES.CI_UID} = d.${DEPENDENCIES.TARGET_RELEASE_UID}
       |  JOIN ${FOLDERS.TABLE} ft ON ft.${FOLDERS.CI_UID} = t.${RELEASES.FOLDER_CI_UID}
       | WHERE r.${RELEASES.CI_UID} = :${RELEASES.CI_UID}
       | """.stripMargin
  @IsReadOnly
  def findByGateReleaseUid(releaseUid: Int): Seq[DependencyRow] = {
    logger.trace(s"Find dependencies by gate releaseUid: '$releaseUid'")
    sqlQuery(STMT_FIND_BY_GATE_RELEASE_UID, params(RELEASES.CI_UID -> releaseUid), defaultMapper).toSeq
  }

  def findByPartialTargetIds(parentIds: Set[CiId], status: Option[Set[String]]): Seq[DependencyRow] = {
    logger.trace(s"Find dependencies by targetIds starting with $parentIds with statuses $status")
    if (parentIds.isEmpty) {
      Nil
    } else {
      val clause = parentIds.map(id =>
        (getReleaselessChildId(id.normalized), getName(releaseIdFrom(id.normalized))) match {
          case (null, releaseId) => s"t.${RELEASES.RELEASE_ID} LIKE '$releaseId%'"
          case (targetId, releaseId) => s"d.${DEPENDENCIES.TARGET_ID} LIKE '$targetId%' AND t.${RELEASES.RELEASE_ID} = '$releaseId'"
        }
      ).mkString("(", " OR ", ")")
      val (statusClause, args) = filterStatus(status)
      jdbcTemplate.query(
        s"""$defaultQuery
           |  WHERE $clause
           |  $statusClause""".stripMargin,
        args,
        defaultMapper
      ).asScala.toSeq
    }
  }

  private val STMT_FIND_BY_ID =
    s"""$defaultQuery
       |  WHERE
       |    d.${DEPENDENCIES.DEPENDENCY_ID} = :dependencyId
       |    AND d.${DEPENDENCIES.GATE_TASK_UID} = :taskCiUid""".stripMargin

  def findById(dependencyId: CiId, taskCiUid: CiUid): DependencyRow = {
    logger.trace(s"Find dependency by id: '$dependencyId'")
    Try {
      sqlQuery(STMT_FIND_BY_ID, params(
        "dependencyId" -> getName(dependencyId),
        "taskCiUid" -> taskCiUid
      ), defaultMapper).headOption.getOrElse(
        throw new NotFoundException(s"Dependency with Id '$dependencyId' not found.")
      )
    }.recoverWith[DependencyRow] {
      case e: EmptyResultDataAccessException =>
        Failure(new NotFoundException(e, s"Dependency with Id '$dependencyId' not found."))
    }.fold(throw _, identity)
  }

  private val STMT_GET_TARGET_ID: String =
    s"""SELECT
       |  f.${FOLDERS.FOLDER_PATH},
       |  f.${FOLDERS.FOLDER_ID},
       |  r.${RELEASES.RELEASE_ID},
       |  d.${DEPENDENCIES.TARGET_ID}
       |  FROM ${DEPENDENCIES.TABLE} d
       |  LEFT OUTER JOIN ${RELEASES.TABLE} r ON r.${RELEASES.CI_UID} = d.${DEPENDENCIES.TARGET_RELEASE_UID}
       |  LEFT OUTER JOIN ${FOLDERS.TABLE} f ON f.${FOLDERS.CI_UID} = r.${RELEASES.FOLDER_CI_UID}
       |  WHERE ${DEPENDENCIES.DEPENDENCY_ID} = :dependencyId
       |  AND d.${DEPENDENCIES.GATE_TASK_UID} = :taskCiUid""".stripMargin

  def getTargetId(dependencyId: CiId, taskCiUid: CiUid): Option[String] = {
    logger.trace(s"Loading targetId of a dependency $dependencyId")
    sqlQuery(STMT_GET_TARGET_ID, params(
      "dependencyId" -> getName(dependencyId),
      "taskCiUid" -> taskCiUid
    ), rs =>
      TargetRow(
        rs.getString(FOLDERS.FOLDER_PATH),
        rs.getString(FOLDERS.FOLDER_ID),
        rs.getString(RELEASES.RELEASE_ID),
        rs.getString(DEPENDENCIES.TARGET_ID)
      ).getTargetId
    ).headOption.flatten
  }

  def findByTargetIds(targetIds: Set[CiId], status: Option[Set[String]]): Seq[DependencyRow] = {
    logger.trace(s"Find dependencies by targetIds: $targetIds with statuses $status")
    if (targetIds.isEmpty) {
      Nil
    } else {
      val clause = targetIds.map(id =>
        (getReleaselessChildId(id.normalized), getName(releaseIdFrom(id))) match {
          case (null, releaseId) => s"d.${DEPENDENCIES.TARGET_ID} is NULL AND t.${RELEASES.RELEASE_ID} = '$releaseId'"
          case (targetId, releaseId) => s"d.${DEPENDENCIES.TARGET_ID} = '$targetId' AND t.${RELEASES.RELEASE_ID} = '$releaseId'"
        }
      ).mkString("(", " OR ", ")")
      val (statusQuery, statusArgs) = filterStatus(status)
      jdbcTemplate.query[DependencyRow](
        s"""$defaultQuery
           |  WHERE $clause
           |  $statusQuery
        """.stripMargin,
        statusArgs,
        defaultMapper
      ).asScala.toSeq
    }
  }

  private val STMT_INSERT_DEPENDENCY =
    s"""INSERT INTO ${DEPENDENCIES.TABLE} (
       | ${DEPENDENCIES.DEPENDENCY_ID},
       | ${DEPENDENCIES.GATE_TASK_UID},
       | ${DEPENDENCIES.GATE_STATUS},
       | ${DEPENDENCIES.TARGET_ID},
       | ${DEPENDENCIES.TARGET_RELEASE_UID}
       |) VALUES (
       |  :dependencyId,
       |  :taskCiUid,
       |  :gateStatus,
       |  :targetId,
       |  (SELECT r.${RELEASES.CI_UID} FROM ${RELEASES.TABLE} r WHERE r.${RELEASES.RELEASE_ID} = :targetReleaseId)
       |)""".stripMargin

  def insertDependency(dependency: Dependency): Unit = {
    logger.trace(s"Inserting dependency ${dependency.getId} of Gate ${dependency.getGateTask.getId} to Target ${dependency.getTargetId}")
    val taskCiUid = dependency.getGateTask.getCiUid;
    val (targetId, targetReleaseId) = getTargets(dependency)
    sqlExec(STMT_INSERT_DEPENDENCY, params(
      "dependencyId" -> getName(dependency.getId),
      "taskCiUid" -> taskCiUid,
      "gateStatus" -> dependency.getGateTask.getStatus.name,
      "targetId" -> targetId,
      "targetReleaseId" -> Option(targetReleaseId).getOrElse("")
    ), ps => Try(ps.execute()).fold(
      t => throw new ReleaseStoreException(s"Failed to insert dependency ${dependency.getId}", t),
      _ => ()
    ))
  }

  private val STMT_INSERT_DEPENDENCY_WITH_GATE_TASK_UID =
    s"""INSERT INTO ${DEPENDENCIES.TABLE} (
       | ${DEPENDENCIES.DEPENDENCY_ID},
       | ${DEPENDENCIES.GATE_TASK_UID},
       | ${DEPENDENCIES.GATE_STATUS},
       | ${DEPENDENCIES.TARGET_ID},
       | ${DEPENDENCIES.TARGET_RELEASE_UID}
       |) VALUES (
       |  :dependencyId,
       |  :gateTaskUid,
       |  :gateStatus,
       |  :targetId,
       |  (SELECT r.${RELEASES.CI_UID} FROM ${RELEASES.TABLE} r WHERE r.${RELEASES.RELEASE_ID} = :targetReleaseId)
       |)""".stripMargin

  def insertDependencyWithGateTaskUid(gateTaskUid: CiUid)(dependency: Dependency): Unit = {
    logger.trace(s"Inserting dependency ${dependency.getId} of Gate ${dependency.getGateTask.getId} to Target ${dependency.getTargetId}")
    val (targetId, targetReleaseId) = getTargets(dependency)
    sqlExec(STMT_INSERT_DEPENDENCY_WITH_GATE_TASK_UID, params(
      "dependencyId" -> getName(dependency.getId),
      "gateTaskUid" -> gateTaskUid,
      "gateStatus" -> dependency.getGateTask.getStatus.name,
      "targetId" -> targetId,
      "targetReleaseId" -> Option(targetReleaseId).getOrElse("")
    ), ps => Try(ps.execute()).fold(
      t => throw new ReleaseStoreException(s"Failed to insert dependency ${dependency.getId}", t),
      _ => ()
    ))
  }

  def batchInsert(taskDependencies: Set[TaskDependency]): Unit = {
    val batchArgs: Array[SqlParameterSource] = taskDependencies.collect {
      case TaskDependency(gateTaskUid, dependency: Dependency) =>
        val sqlParameterSource = new MapSqlParameterSource()
        val (targetId, targetReleaseId) = getTargets(dependency)
        sqlParameterSource.addValue("dependencyId", getName(dependency.getId))
        sqlParameterSource.addValue("gateTaskUid", gateTaskUid)
        sqlParameterSource.addValue("gateStatus", dependency.getGateTask.getStatus.name)
        sqlParameterSource.addValue("targetId", targetId)
        sqlParameterSource.addValue("targetReleaseId", Option(targetReleaseId).getOrElse(""))
        sqlParameterSource
    }.toArray
    namedTemplate.batchUpdate(STMT_INSERT_DEPENDENCY_WITH_GATE_TASK_UID, batchArgs)
  }

  private val STMT_DELETE =
    s"""DELETE FROM ${DEPENDENCIES.TABLE}
       | WHERE
       |  ${DEPENDENCIES.DEPENDENCY_ID} = :dependencyId
       |  AND ${DEPENDENCIES.GATE_TASK_UID} = :taskCiUid""".stripMargin

  def deleteDependency(id: String, taskCiUid: CiUid): Unit = {
    logger.trace(s"Deleting dependency ${id}")
    sqlUpdate(
      STMT_DELETE, params(
        "dependencyId" -> getName(id),
        "taskCiUid" -> taskCiUid
      ), _ => ()
    )
    logger.trace(s"Deleting dependency ${id} done")
  }

  def deleteDependency(dependency: Dependency): Unit = {
    logger.trace(s"Deleting dependency ${dependency.getId} of Gate ${dependency.getGateTask.getId} to Target ${dependency.getTargetId}")
    deleteDependency(dependency.getId, dependency.getGateTask.getCiUid)
  }

  private val STMT_BATCH_DELETE =
    s"""DELETE FROM ${DEPENDENCIES.TABLE}
       | WHERE
       |  ${DEPENDENCIES.DEPENDENCY_ID} = :${DEPENDENCIES.DEPENDENCY_ID}
       |  AND ${DEPENDENCIES.GATE_TASK_UID} = :${DEPENDENCIES.GATE_TASK_UID}""".stripMargin

  def batchDeleteDependencies(dependencies: Set[Dependency]): Unit = {
    sqlBatch(STMT_BATCH_DELETE, dependencies.collect {
      case d: Dependency => params(DEPENDENCIES.DEPENDENCY_ID -> getName(d.getId), DEPENDENCIES.GATE_TASK_UID -> d.getGateTask.getCiUid)
    })
  }

  private val STMT_DELETE_BY_RELEASE_UID =
    s"""DELETE FROM ${DEPENDENCIES.TABLE}
       | WHERE
       |  ${DEPENDENCIES.TARGET_RELEASE_UID} = :releaseUid""".stripMargin

  def deleteByReleaseUid(releaseUid: CiUid): Boolean = {
    logger.trace(s"Deleting dependency by releaseUid ${releaseUid}")
    val result = sqlUpdate(STMT_DELETE_BY_RELEASE_UID, params("releaseUid" -> releaseUid), _ != 0)
    logger.trace(s"Deleting dependency by releaseUid ${releaseUid} done")
    result
  }

  private val STMT_DELETE_BY_GATE_TASK_UIDS =
    s"""DELETE FROM ${DEPENDENCIES.TABLE}
       | WHERE
       |  ${DEPENDENCIES.GATE_TASK_UID} IN (:taskUids)
       |""".stripMargin

  def deleteByTaskUids(taskUids: Seq[CiUid]): Boolean = {
    logger.trace(s"Deleting dependency by taskUids ${taskUids}")
    val result = taskUids.grouped(BATCH_SIZE).foldLeft(false)((result, batch) => {
      val batchResult = sqlUpdate(STMT_DELETE_BY_GATE_TASK_UIDS, params("taskUids" -> batch.map(_.asInstanceOf[Integer]).toList.asJava), _ != 0)
      batchResult || result
    })
    logger.trace(s"Deleting dependency by releaseUid ${taskUids} done")
    result
  }

  private val STMT_UPDATE =
    s"""UPDATE ${DEPENDENCIES.TABLE}
       | SET
       |  ${DEPENDENCIES.GATE_STATUS} = :gateStatus,
       |  ${DEPENDENCIES.TARGET_ID} = :targetId,
       |  ${DEPENDENCIES.TARGET_RELEASE_UID} = :targetReleaseUid
       |  WHERE
       |    ${DEPENDENCIES.DEPENDENCY_ID} = :dependencyId
       |    AND ${DEPENDENCIES.GATE_TASK_UID} = :taskCiUid""".stripMargin

  def updateDependency(dependency: Dependency): Unit = {
    val targetReleaseCiUid: CiUid = Option(dependency.getTarget[PlanItem]).map(_.getReleaseUid).orNull
    logger.trace(s"Updating dependency ${dependency.getId} of Gate ${dependency.getGateTask.getId} to Target ${dependency.getTargetId}")
    val (targetId, _) = getTargets(dependency)
    val taskCiUid = dependency.getGateTask.getCiUid
    sqlExec(
      STMT_UPDATE,
      params(
        "gateStatus" -> dependency.getGateTask.getStatus.name,
        "targetId" -> targetId,
        "targetReleaseUid" -> targetReleaseCiUid,
        "dependencyId" -> getName(dependency.getId),
        "taskCiUid" -> taskCiUid
      ), ps => Try(ps.executeUpdate()) match {
        case Success(n) => if (n == 0) {
          throw new NotFoundException(s"Could not find dependency to update by ID ${dependency.getId}, taskCiUid = ${taskCiUid}")
        } else {
          ()
        }
        case Failure(ex) => throw new ReleaseStoreException(s"Failed to update dependency ${dependency.getId}", ex)
      }
    )

    logger.trace(s"Updating dependency ${dependency.getId} of Gate ${dependency.getGateTask.getId} to Target ${dependency.getTargetId} done")
  }

  private val STMT_BATCH_UPDATE =
    s"""UPDATE ${DEPENDENCIES.TABLE}
       | SET
       |  ${DEPENDENCIES.GATE_STATUS} = :${DEPENDENCIES.GATE_STATUS},
       |  ${DEPENDENCIES.TARGET_ID} = :${DEPENDENCIES.TARGET_ID},
       |  ${DEPENDENCIES.TARGET_RELEASE_UID} = :${DEPENDENCIES.TARGET_RELEASE_UID}
       |  WHERE
       |    ${DEPENDENCIES.DEPENDENCY_ID} = :${DEPENDENCIES.DEPENDENCY_ID}
       |    AND ${DEPENDENCIES.GATE_TASK_UID} = :${DEPENDENCIES.GATE_TASK_UID}""".stripMargin

  def batchUpdateDependencies(dependencies: Set[Dependency]): Unit = {
    sqlBatch(STMT_BATCH_UPDATE, dependencies.collect {
      case d: Dependency =>
        val (targetId, targetReleaseId) = getTargets(d)
        val potentialTarget = Option(d.getTarget[PlanItem])
        val targetReleaseUid = potentialTarget.map(_.getReleaseUid).orNull
        params(
          DEPENDENCIES.GATE_STATUS -> d.getGateTask.getStatus.name(),
          DEPENDENCIES.TARGET_ID -> targetId,
          DEPENDENCIES.TARGET_RELEASE_UID -> targetReleaseUid,
          DEPENDENCIES.DEPENDENCY_ID -> getName(d.getId),
          DEPENDENCIES.GATE_TASK_UID -> d.getGateTask.getCiUid
        )
    })
  }

  private def getTargets(dependency: Dependency): (String, String) = {
    val targetId = dependency.getTargetId.normalized
    if (!dependency.hasVariableTarget && !dependency.isArchived) {
      validateDependencyTarget(dependency)
      if (isReleaseId(targetId)) {
        (null, getName(targetId))
      } else {
        Try((getReleaselessChildId(targetId), getName(releaseIdFrom(targetId)))).getOrElse((targetId, null))
      }
    } else {
      (targetId, null)
    }
  }

  private def validateDependencyTarget(dependency: Dependency) = {
    Option(dependency.getTarget[PlanItem]).map(_.getId).map(Ids.getFolderlessId).map(Ids.releaseIdFrom).foreach(targetOwnId => {
      Option(dependency.getTargetId).map(Ids.getFolderlessId).map(Ids.releaseIdFrom).foreach(targetId => {
        if (targetOwnId != targetId) {
          throw new IllegalArgumentException("Dependency targetId does not match target");
        }
      })
    })
  }
}

object DependencyPersistence {
  case class TaskDependency(taskUid: CiUid, dependency: Dependency)
}