package com.xebialabs.xlrelease.upgrade.db

import com.xebialabs.deployit.security.PermissionEnforcer
import com.xebialabs.deployit.server.api.upgrade.{Upgrade, Version}
import com.xebialabs.xlrelease.config.XlrConfig
import com.xebialabs.xlrelease.db.sql.SqlBuilder.Dialect
import com.xebialabs.xlrelease.domain.Release
import com.xebialabs.xlrelease.repository.Ids.getName
import com.xebialabs.xlrelease.repository.sql.SqlRepositoryAdapter
import com.xebialabs.xlrelease.repository.sql.persistence.CiId.CiId
import com.xebialabs.xlrelease.repository.sql.persistence.Schema.{RELEASES, RELEASES_EXT}
import com.xebialabs.xlrelease.repository.sql.persistence.Utils.params
import com.xebialabs.xlrelease.repository.sql.persistence.data.ReleaseExtensionRow
import com.xebialabs.xlrelease.repository.sql.persistence.{CiUid, PersistenceSupport, ReleasePersistence}
import com.xebialabs.xlrelease.repository.{CommentRepository, ReleaseExtensionsRepository}
import com.xebialabs.xlrelease.security.XLReleasePermissions
import com.xebialabs.xlrelease.upgrade.Components.XL_RELEASE_COMPONENT
import com.xebialabs.xlrelease.upgrade.UpgradeSupport.{BatchSupport, TransactionSupport}
import grizzled.slf4j.Logging
import org.springframework.beans.factory.annotation.{Autowired, Qualifier}
import org.springframework.jdbc.core.JdbcTemplate
import org.springframework.security.authentication.TestingAuthenticationToken
import org.springframework.security.core.context.{SecurityContext, SecurityContextHolder, SecurityContextImpl}
import org.springframework.stereotype.Component
import org.springframework.transaction.support.TransactionTemplate

import scala.collection.parallel.CollectionConverters._
import scala.jdk.CollectionConverters._


/**
  * Upgrade for: ReleaseExtension and Comments
  * They need to be both in same upgrade to not loose data when clearing fields before storing the release json.
  */
@Component
class XLRelease820TablesUpgrade @Autowired()(@Qualifier("xlrRepositoryJdbcTemplate") val jdbcTemplate: JdbcTemplate,
                                             @Qualifier("xlrRepositorySqlDialect") implicit val dialect: Dialect,
                                             @Qualifier("xlrRepositoryTransactionTemplate") val transactionTemplate: TransactionTemplate,
                                             releasePersistence: ReleasePersistence,
                                             repositoryAdapter: SqlRepositoryAdapter,
                                             releaseExtensionsRepository: ReleaseExtensionsRepository,
                                             commentRepository: CommentRepository,
                                             xlrConfig: XlrConfig)
  extends Upgrade
    with PersistenceSupport
    with Logging
    with BatchSupport
    with TransactionSupport {

  private val savedSecurityContext: SecurityContext = SecurityContextHolder.getContext

  private val batchSize: Int = Math.max(1, (xlrConfig.database.maxPoolSize - 2) / 2)

  // LoginRule copy/paste, needed by releaseSearchService
  private def login(): Unit = {
    val username = XLReleasePermissions.ADMIN_USERNAME
    val context = new SecurityContextImpl
    context.setAuthentication(new TestingAuthenticationToken(username, username, Seq(PermissionEnforcer.ROLE_ADMIN): _*))
    SecurityContextHolder.setContext(context)
  }

  private def logout(): Unit = {
    SecurityContextHolder.setContext(savedSecurityContext)
  }

  override def upgradeVersion(): Version = Version.valueOf(XL_RELEASE_COMPONENT, "8.2.0#5")

  override def doUpgrade(): Boolean = {
    logger.info("Starting release extensions and comments upgrade")
    login()
    val releaseIds = getReleasesIds()
    val totalReleaseNum = releaseIds.length
    val sizeDigits = digits(totalReleaseNum)

    def padReleaseNum: Int => String = pad(sizeDigits)

    def releaseCounter(releaseNum: Int): String = s"Release[${padReleaseNum(releaseNum)}/${padReleaseNum(totalReleaseNum)}]"

    logger.info(s"Found $totalReleaseNum releases, upgrading in batch of $batchSize")
    doInBatch(releaseIds.zipWithIndex, batchSize, "releases") { batch =>
      val items = if (xlrConfig.upgrader.parallel) {
        batch.items.par
      } else {
        batch.items
      }
      items.iterator.foreach { case (releaseId, releaseNum) =>
        val pre = s"${batch.logPrefix} ${releaseCounter(releaseNum + 1)}[${getName(releaseId)}]"
        doInTransaction {
          upgradeRelease(pre)(releaseId)
        }
      }
    }
    logger.info("Finished release extensions and comments upgrade")
    logout()
    true
  }

  def upgradeRelease(logPrefix: String)(releaseId: String): Unit = {
    def log(msg: String): Unit = logger.debug(s"$logPrefix $msg")

    log("Checking for already extracted extensions")
    val release = repositoryAdapter.read[Release](releaseId)
    val releaseExtensions = release.getExtensions.asScala
    val missingExtensionsIdsSet = findMissingExtensionsIds(release)
    val releaseExtensionsNames = releaseExtensions.map(ext => getName(ext.getId)).toSet

    log(s"Release has extensions: ${releaseExtensionsNames.mkString(", ")}")

    if (missingExtensionsIdsSet.isEmpty) {
      log("Extensions already extracted, skipping.")
      if (releaseExtensions.nonEmpty) {
        log(s"Moving $releaseExtensionsNames extensions to external table, under release UID ${release.getCiUid}")
        releaseExtensionsRepository.createAll(release, releaseExtensions.map(_.getId).toSet)
      }
    } else {
      log(s"Extracting remaining ${missingExtensionsIdsSet.size} extensions")
      releaseExtensionsRepository.createAll(release, missingExtensionsIdsSet)
    }
    log("Extracting and persisting task comments")
    release.getAllTasks.asScala.foreach { task =>
      commentRepository.createAll(task.getId, task.getComments.asScala.toSeq)
    }

    log("Updating Release JSON with extensions and comments removed")
    release.setExtensions(List.empty.asJava)
    releasePersistence.updateReleaseJson(release.getCiUid, release)
  }

  private def getReleasesIds(): Seq[String] =
    sqlQuery(s"""SELECT ${RELEASES.RELEASE_ID} FROM ${RELEASES.TABLE} ORDER BY ${RELEASES.STATUS} ASC""", params(), _.getString(RELEASES.RELEASE_ID)).toSeq

  private def findMissingExtensionsIds(release: Release): Set[CiId] = {
    val existingNames = findGenericExtensionsIds(release.getCiUid).map(getName).toSet ++ findRiskId(release.getId).map(getName).toSet
    val allNames = release.getExtensions.asScala.map(_.getId).map(getName).toSet
    allNames diff existingNames
  }


  private val STMT_RELEASE_EXTENSIONS_IDS: String =
    s"""|SELECT ${RELEASES_EXT.EXTENSION_ID}
        |FROM ${RELEASES_EXT.TABLE}
        |WHERE ${RELEASES_EXT.RELEASE_UID} = :releaseUid
     """.stripMargin

  private def findGenericExtensionsIds(releaseUid: CiUid): Seq[CiId] = {
    sqlQuery(STMT_RELEASE_EXTENSIONS_IDS,
      params("releaseUid" -> releaseUid),
      _.getString(RELEASES_EXT.EXTENSION_ID)
    ).toSeq
  }

  private def findRiskId(releaseId: CiId): Option[CiId] = {
    val riskId = ReleaseExtensionRow.toExternalId(releaseId, "Risk")
    Some(riskId).filter(releaseExtensionsRepository.exists)
  }
}
