package com.xebialabs.xlrelease.spring.config

import com.xebialabs.xlrelease.db.sql.SqlBuilder.Dialect
import com.xebialabs.xlrelease.repository.FacetRepository.GenericFacetRepository
import com.xebialabs.xlrelease.repository.ReleaseExtensionsRepository.GenericReleaseExtensionRepository
import com.xebialabs.xlrelease.repository._
import com.xebialabs.xlrelease.repository.sql._
import com.xebialabs.xlrelease.repository.sql.persistence.UserProfilePersistence
import com.xebialabs.xlrelease.scheduler.logs.{SqlTaskExecutionRepository, TaskExecutionRepository}
import com.xebialabs.xlrelease.scheduler.repository.JobRepository
import com.xebialabs.xlrelease.scheduler.sql.SqlJobRepository
import com.xebialabs.xlrelease.security.sql.SecurityUserCacheConfigurationCondition
import com.xebialabs.xlrelease.service.{SqlTaskBackup, TaskBackup}
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.context.annotation.{Bean, Conditional, Configuration, Primary}
import org.springframework.jdbc.core.JdbcTemplate

@Configuration
class SqlRepositoriesConfiguration(sqlConfiguration: SqlConfiguration,
                                   sqlPersistenceConfiguration: SqlPersistenceConfiguration,
                                   sqlSecurityConfiguration: SqlSecurityConfiguration) {
  @Autowired
  var dependencyTargetLoader: DependencyTargetLoader = _

  @Bean
  def teamRepository(): SqlTeamRepository = {
    new SqlTeamRepository(sqlSecurityConfiguration.permissionEditor(),
      sqlSecurityConfiguration.roleService(),
      securedCis())
  }

  @Bean
  def facetRepository(): GenericFacetRepository with FacetRepository.ForLive = {
    new SqlFacetRepository(
      sqlConfiguration.ciIdService(),
      sqlPersistenceConfiguration.facetPersistence(),
      sqlPersistenceConfiguration.sqlRepositoryAdapter(),
      sqlConfiguration.xlrRepositorySqlDialect()
    )
  }

  @Bean
  def facetRepositoryDispatcher(): FacetRepositoryDispatcher = new FacetRepositoryDispatcher(sqlConfiguration.ciIdService(), facetRepository())

  @Bean
  def releaseRepository(): ReleaseRepository = {
    new SqlReleaseRepository(
      sqlPersistenceConfiguration.releasePersistence(),
      sqlPersistenceConfiguration.taskPersistence(),
      sqlPersistenceConfiguration.dependencyPersistence(),
      sqlPersistenceConfiguration.commentPersistence(),
      teamRepository(),
      releaseExtensionsRepository(),
      sqlPersistenceConfiguration.releaseConfigurationReferencePersistence(),
      sqlPersistenceConfiguration.sqlRepositoryAdapter(),
      facetRepositoryDispatcher())(sqlConfiguration.xlrRepositorySqlDialect())
  }

  @Bean
  def phaseRepository(): PhaseRepository = {
    new SqlPhaseRepository(
      sqlPersistenceConfiguration.releasePersistence(),
      sqlPersistenceConfiguration.taskPersistence(),
      sqlPersistenceConfiguration.releaseConfigurationReferencePersistence(),
      sqlPersistenceConfiguration.commentPersistence(),
      dependencyTargetLoader,
      sqlPersistenceConfiguration.dependencyPersistence(),
      facetRepositoryDispatcher(),
      sqlPersistenceConfiguration.sqlRepositoryAdapter()
    )
  }

  @Bean
  def taskRepository(): TaskRepository = {
    new SqlTaskRepository(
      sqlPersistenceConfiguration.releasePersistence(),
      sqlPersistenceConfiguration.dependencyPersistence(),
      sqlPersistenceConfiguration.taskPersistence(),
      commentRepository(),
      sqlPersistenceConfiguration.sqlRepositoryAdapter(),
      sqlPersistenceConfiguration.releaseConfigurationReferencePersistence(),
      facetRepositoryDispatcher(),
      sqlConfiguration.xlrRepositorySqlDialect()
    )
  }

  @Bean
  def commentRepository(): CommentRepository = {
    new SqlCommentRepository(
      sqlPersistenceConfiguration.taskPersistence(),
      sqlPersistenceConfiguration.commentPersistence(),
      sqlPersistenceConfiguration.sqlRepositoryAdapter()
    )
  }

  @Bean
  def planItemRepository(): PlanItemRepository = {
    new SqlPlanItemRepository(
      sqlPersistenceConfiguration.releasePersistence(),
      sqlPersistenceConfiguration.taskPersistence(),
      sqlPersistenceConfiguration.sqlRepositoryAdapter()
    )
  }

  @Bean
  def dependencyRepository(): DependencyRepository = {
    new SqlDependencyRepository(
      sqlPersistenceConfiguration.releasePersistence(),
      sqlPersistenceConfiguration.dependencyPersistence(),
      sqlPersistenceConfiguration.sqlRepositoryAdapter()
    )
  }

  @Bean
  def gateRepository(): GateConditionRepository = {
    new SqlGateConditionRepository(
      sqlPersistenceConfiguration.releasePersistence(),
      sqlPersistenceConfiguration.dependencyPersistence(),
      sqlPersistenceConfiguration.sqlRepositoryAdapter()
    )
  }

  @Bean
  def attachmentRepository(): AttachmentRepository = {
    new SqlAttachmentRepository(sqlPersistenceConfiguration.releasePersistence())
  }

  @Bean
  def genericReleaseExtensionRepository(): GenericReleaseExtensionRepository = {
    new SqlReleaseExtensionsRepository(
      sqlPersistenceConfiguration.releaseExtensionPersistence(),
      sqlPersistenceConfiguration.sqlRepositoryAdapter(),
      sqlPersistenceConfiguration.releasePersistence()
    )
  }

  @Bean
  def releaseExtensionsRepository(): ReleaseExtensionsRepository = {
    new ReleaseExtensionsRepository(genericReleaseExtensionRepository())
  }

  @Bean
  def linkRepository(): LinkRepository = {
    new SqlLinkRepository(
      sqlPersistenceConfiguration.releasePersistence(),
      sqlPersistenceConfiguration.dependencyPersistence(),
      sqlPersistenceConfiguration.sqlRepositoryAdapter())
  }

  @Bean
  def calendarEntryRepository(): CalendarEntryRepository = {
    new SqlCalendarEntryRepository(
      sqlPersistenceConfiguration.calendarEntryPersistence(),
      sqlPersistenceConfiguration.sqlRepositoryAdapter()
    )
  }


  @Bean
  def taskBackup(dependencyTargetResolver: DependencyTargetResolver): TaskBackup = {
    new SqlTaskBackup(
      sqlPersistenceConfiguration.taskBackupPersistence(),
      sqlPersistenceConfiguration.releasePersistence(),
      sqlPersistenceConfiguration.sqlRepositoryAdapter(),
      dependencyTargetResolver)
  }


  @Bean
  def configurationRepository(): ConfigurationRepository = new SqlConfigurationRepository(
    sqlPersistenceConfiguration.configurationPersistence(),
    sqlPersistenceConfiguration.releaseConfigurationReferencePersistence(),
    sqlPersistenceConfiguration.triggerConfigurationReferencePersistence(),
    sqlPersistenceConfiguration.sqlRepositoryAdapter(),
    sqlPersistenceConfiguration.folderPersistence(),
    sqlPersistenceConfiguration.releasePersistence(),
    jobRepository()
  )

  @Bean
  def userProfileRepository(userProfilePersistence: UserProfilePersistence): UserProfileRepository = {
    new SqlUserProfileRepository(userProfilePersistence)
  }

  @Bean
  @Primary
  @Conditional(value = Array(classOf[SecurityUserCacheConfigurationCondition]))
  def cachingUserProfileRepositoryWrapper(userProfileRepository: UserProfileRepository, evicter: UserProfileCacheEvicter): UserProfileRepository = {
    new CachingSqlUserProfileRepositoryWrapper(userProfileRepository, evicter)
  }

  @Bean
  def securedCis(): SecuredCis = {
    new SqlSecuredCis(sqlPersistenceConfiguration.folderPersistence(), sqlPersistenceConfiguration.releasePersistence())
  }

  @Bean
  def springSessionRepository(): SpringSessionRepository = new SqlSpringSessionRepository(
    sqlPersistenceConfiguration.springSessionPersistence()
  )

  @Primary
  @Bean
  def jobRepository(): JobRepository = new SqlJobRepository(
    sqlConfiguration.xlrRepositoryJdbcTemplate(),
    sqlConfiguration.xlrRepositorySqlDialect()
  )

  @Bean
  def userTokenRepository(): UserTokenRepository = {
    implicit val dialect: Dialect = sqlConfiguration.xlrRepositorySqlDialect()
    implicit val jdbcTemplate: JdbcTemplate = sqlConfiguration.xlrRepositoryJdbcTemplate()
    implicit val sqlRepositoryAdapter: SqlRepositoryAdapter = sqlPersistenceConfiguration.sqlRepositoryAdapter()
    new SqlUserTokenRepository()
  }

  @Bean
  def jobRunnerRepository(): JobRunnerRepository = {
    new SqlJobRunnerRepository(configurationRepository())
  }

  @Bean
  def taskLogRepository(): TaskExecutionRepository = {
    implicit val jdbcTemplate = sqlConfiguration.xlrRepositoryJdbcTemplate()
    implicit val dialect: Dialect = sqlConfiguration.xlrRepositorySqlDialect()
    new SqlTaskExecutionRepository()
  }
}
