package com.xebialabs.xlrelease.repository.sql.persistence

import com.github.benmanes.caffeine.cache.Cache
import com.xebialabs.xlrelease.cache.caffeine.CaffeineCacheBuilder
import com.xebialabs.xlrelease.config.XlrConfig
import com.xebialabs.xlrelease.config.XlrConfig.ExtendedConfig
import com.xebialabs.xlrelease.repository.sql.persistence.CiId._
import com.xebialabs.xlrelease.repository.sql.persistence.data.ReleaseRow
import com.xebialabs.xlrelease.service.FeatureService
import grizzled.slf4j.Logging
import org.springframework.jmx.`export`.annotation.{ManagedOperation, ManagedResource}

@ManagedResource(objectName = "com.xebialabs.xlrelease.settings:name=ReleaseCacheService", description = "MBean to configure internal release cache")
sealed trait ReleaseCacheService extends FeatureService {

  override def name(): CiId = "ReleaseCacheService"

  @ManagedOperation(description = "Invalidate provided entry with provided Release ID as a key")
  def invalidate(releaseId: CiId): Unit

  @ManagedOperation(description = "Invalidate all entries")
  def invalidateAll(): Unit

  @ManagedOperation(description = "Get cache statistics as a string")
  def getStats(): String

}

object ReleaseCacheService extends ReleaseCacheService with Logging with CaffeineCacheBuilder {

  private val cacheName = "release-rows"
  private val cacheConf = XlrConfig.getInstance.cache.config
  private val isClustered = XlrConfig.getInstance.isClusterEnabled

  @volatile
  private var enabled: Boolean = {
    val releaseRowCacheEnabled = cacheConf.getConfig(cacheName).getOptionalBoolean("enabled").getOrElse(false)
    if (isClustered && releaseRowCacheEnabled) {
      logger.warn(s"Cannot enable $cacheName for clustered environment")
    }
    !isClustered && releaseRowCacheEnabled
  }

  private val releaseCache: Cache[CiId, ReleaseRow] = {
    val cacheSettings = XlrConfig.getInstance.cache
    buildCache("releaseCacheService", cacheName, "default-cache-values", cacheSettings)
  }


  def get(releaseId: CiId): Option[ReleaseRow] = {
    if (enabled) {
      Option(releaseCache.getIfPresent(releaseId.shortId)).map { r: ReleaseRow =>
        logger.trace(s"Fetching ${releaseId.shortId} from cache")
        r
      }
    } else {
      None
    }
  }

  def put(releaseId: CiId, releaseRow: ReleaseRow): Unit = {
    if (enabled) {
      logger.trace(s"Caching ${releaseId.shortId}")
      releaseCache.put(releaseId.shortId, releaseRow)
    }
  }

  override def invalidate(releaseId: CiId): Unit = {
    releaseCache.invalidate(releaseId.shortId)
  }

  override def invalidateAll(): Unit = {
    releaseCache.invalidateAll()
  }

  override def enable(): Unit = {
    enabled = true
  }

  override def disable(): Unit = {
    enabled = false
    releaseCache.invalidateAll()
  }

  def isEnabled(): Boolean = {
    enabled
  }

  override def getStats(): String = {
    val cacheStats = releaseCache.stats()
    cacheStats.toString
  }
}
