package com.xebialabs.xlrelease.spring.config

import com.github.benmanes.caffeine.cache.stats.CacheStats
import com.github.benmanes.caffeine.cache.{Cache, CacheLoader, Caffeine, Policy}
import com.xebialabs.xlrelease.config.XlrConfig
import com.xebialabs.xlrelease.security.sql.SecurityCacheConfigurationCondition
import grizzled.slf4j.Logging
import org.springframework.cache.CacheManager
import org.springframework.cache.annotation.EnableCaching
import org.springframework.cache.caffeine.CaffeineCacheManager
import org.springframework.context.annotation.{Bean, Conditional, Configuration, Primary}

import java.time.Duration
import java.util.concurrent.ConcurrentMap
import java.util.function
import java.{lang, util}
import scala.jdk.CollectionConverters._

@Configuration
@Conditional(value = Array(classOf[SecurityCacheConfigurationCondition]))
@EnableCaching(proxyTargetClass = true)
class SecurityCacheConfiguration {

  @Bean
  @Primary
  def securityCacheManager(xlrConfig: XlrConfig): CacheManager = {
    val cacheManager: CaffeineCacheManager = new CaffeineCacheManager() {
      override def createNativeCaffeineCache(name: String): Cache[AnyRef, AnyRef] = {
        if (name.startsWith("security")) {
          val maxCacheSize = xlrConfig.cache.maxSize("security")
          val maxTtl = Duration.ofMinutes(xlrConfig.cache.ttl("security").toMinutes)
          new CaffeineCacheDelegate(name, createCacheBuilder(maxTtl, maxCacheSize).build())
        } else {
          super.createNativeCaffeineCache(name)
        }
      }

      override def setCacheLoader(cacheLoader: CacheLoader[AnyRef, AnyRef]): Unit = {
        throw new UnsupportedOperationException("Cache loading not support by this manager implementation")
      }
    }
    cacheManager.setCaffeine(createCacheBuilder(Duration.ofHours(1), 2500))
    cacheManager
  }

  private def createCacheBuilder(expirationTime: Duration, maxCacheEntries: Long) = {
    val caffeineTileCacheBuilder = Caffeine.newBuilder()
      .expireAfterAccess(expirationTime)
      .maximumSize(maxCacheEntries)
      .softValues()
      .recordStats()
    caffeineTileCacheBuilder
  }
}

class CaffeineCacheDelegate[K, V](name: String, cache: Cache[K, V]) extends Cache[K, V] with Logging {
  override def getIfPresent(key: K): V = cache.getIfPresent(key)

  override def get(key: K, mappingFunction: function.Function[_ >: K, _ <: V]): V = cache.get(key, mappingFunction)

  override def getAllPresent(keys: lang.Iterable[_ <: K]): util.Map[K, V] = cache.getAllPresent(keys)

  override def put(key: K, value: V): Unit = cache.put(key, value)

  override def putAll(map: util.Map[_ <: K, _ <: V]): Unit = cache.putAll(map)

  override def invalidate(key: K): Unit = this.invalidateAll(util.Arrays.asList(key))

  override def invalidateAll(keys: lang.Iterable[_ <: K]): Unit = {
    logger.debug(s"Invalidating $keys from cache [$name]")
    val existingKeys = cache.asMap().keySet().asScala.filter(_ != null).map(_.toString)
    val existingKeysToRemove = keys.asScala.filter(_ != null).flatMap(keyToRemove => {
      existingKeys.filter(existingKey => existingKey == keyToRemove || (keyToRemove.toString.startsWith("regex:") && existingKey.matches(keyToRemove.toString.replace("regex:", ""))))
    }).asJava
    cache.invalidateAll(existingKeysToRemove.asInstanceOf[lang.Iterable[_ <: K]])
    logger.debug(s"Invalidated $existingKeysToRemove from cache [$name]")
  }

  override def invalidateAll(): Unit = {
    logger.debug(s"Invalidating all from cache [$name]")
    cache.invalidateAll()
  }

  override def estimatedSize(): Long = cache.estimatedSize()

  override def stats(): CacheStats = cache.stats()

  override def asMap(): ConcurrentMap[K, V] = cache.asMap()

  override def cleanUp(): Unit = cache.cleanUp()

  override def policy(): Policy[K, V] = cache.policy()

  override def getAll(keys: lang.Iterable[_ <: K], mappingFunction: function.Function[_ >: util.Set[_ <: K], _ <: util.Map[_ <: K, _ <: V]]): util.Map[K, V] = cache.getAll(keys, mappingFunction)
}

