package com.xebialabs.xlrelease.upgrade.db

import com.xebialabs.deployit.server.api.upgrade.{Upgrade, Version}
import com.xebialabs.xlrelease.api.v1.forms.VariableOrValue
import com.xebialabs.xlrelease.db.sql.SqlBuilder.Dialect
import com.xebialabs.xlrelease.domain.status.ReleaseStatus._
import com.xebialabs.xlrelease.repository.{CiCloneHelper, FacetRepositoryDispatcher, ReleaseExtensionsRepository, ReleaseRepository}
import com.xebialabs.xlrelease.upgrade.Components.XL_RELEASE_COMPONENT
import com.xebialabs.xlrelease.upgrade.UpgradeSupport.{BatchSupport, TransactionSupport}
import com.xebialabs.xlrelease.upgrade.json.FolderVariableUpgradeUtils._
import com.xebialabs.xlrelease.variable.VariableHelper.withoutVariableSyntax
import grizzled.slf4j.Logging
import org.springframework.beans.factory.annotation.{Autowired, Qualifier}
import org.springframework.jdbc.core.JdbcTemplate
import org.springframework.stereotype.Component
import org.springframework.transaction.support.TransactionTemplate

import java.util.stream.Collectors
import scala.collection.mutable
import scala.jdk.CollectionConverters._


@Component
class XLRelease860FolderVariableUpgrade @Autowired()(@Qualifier("xlrRepositoryJdbcTemplate") val jdbcTemplate: JdbcTemplate,
                                                     @Qualifier("xlrRepositorySqlDialect") implicit val dialect: Dialect,
                                                     @Qualifier("xlrRepositoryTransactionTemplate") val transactionTemplate: TransactionTemplate,
                                                     facetRepositoryDispatcher: FacetRepositoryDispatcher,
                                                     releaseExtensionsRepository: ReleaseExtensionsRepository,
                                                     releaseRepository: ReleaseRepository)
  extends Upgrade with Logging with BatchSupport with TransactionSupport {

  override def upgradeVersion(): Version = Version.valueOf(XL_RELEASE_COMPONENT, "8.6.0#3")

  override def doUpgrade(): Boolean = {
    logger.info("Starting XLRelease860FolderVariableUpgrade")
    val releaseIds = releaseRepository.findIdsByStatus(TEMPLATE, PLANNED, IN_PROGRESS, PAUSED, FAILING, FAILED)

    doInBatch(releaseIds) { batch =>
      doInTransaction {
        batch.items.foreach { releaseId =>
          val release = releaseRepository.findById(releaseId)

          val releaseVariableKeys = release.getVariablesByKeys.keySet().asScala
          val alreadyUsed = mutable.Set.empty[String] ++ releaseVariableKeys
          val oldToNewKeys = releaseVariableKeys.collect { case variableKey if variableKey.startsWith(FOLDER_VAR_PREFIX) =>
            val newKey = convertFolderVarKeyToReleaseVarKey(variableKey, alreadyUsed)
            alreadyUsed += newKey
            variableKey -> newKey
          }.toMap

          val newVariables = oldToNewKeys.map { case (oldKey, newKey) =>
            val newVar = CiCloneHelper.cloneCi(release.getVariablesByKeys.get(oldKey))
            newVar.setKey(newKey)
            newVar
          }.toList.asJava

          if (!newVariables.isEmpty) {
            release.addVariables(newVariables)

            for {
              varRef <- release.collectVariableReferences().asScala
              key = withoutVariableSyntax(varRef.getKey) if key.startsWith(FOLDER_VAR_PREFIX) && releaseVariableKeys.contains(key)
              v = release.getVariablesByKeys.get(key)
              up <- varRef.getUsagePoints.asScala
            } {
              up.replaceVariable(v, new VariableOrValue(oldToNewKeys(key), null))
            }

            // retain only those that don't start with folder.
            val toRetain = release.getVariables.stream()
              .filter(variable => !oldToNewKeys.contains(variable.getKey))
              .collect(Collectors.toList())
            release.setVariables(toRetain)

            release.getReleaseTriggers.forEach { trigger =>
              trigger.getVariables.forEach { variable =>
                oldToNewKeys.get(variable.getKey).foreach(variable.setKey)
              }
            }

            release.getAllTasks.forEach(_.getFacets.forEach(facetRepositoryDispatcher.liveRepository.update))
            release.getExtensions.forEach(releaseExtensionsRepository.update)
            releaseRepository.update(release)
          }
        }
      }
    }
    logger.info("Finished XLRelease860FolderVariableUpgrade")
    true
  }
}


