package com.xebialabs.xlrelease.repository.sql.proxy

import com.xebialabs.deployit.plugin.api.reflect.Type
import com.xebialabs.deployit.plugin.api.services.Repository
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem
import com.xebialabs.xlrelease.config.XlrConfig
import com.xebialabs.xlrelease.repository.IdMatchers.{ConfigurationId, PlanItemId, ValueProviderId, VariableId}
import com.xebialabs.xlrelease.repository.Ids
import com.xebialabs.xlrelease.repository.proxy.{CiReferenceProxyFactory, ResolvableConfigurationItemReference}
import com.xebialabs.xlrelease.repository.sql.persistence.ReleasePersistence
import com.xebialabs.xlrelease.repository.sql.persistence.configuration.ConfigurationReferencePersistence
import com.xebialabs.xlrelease.serialization.json.repository.ResolverRepository
import com.xebialabs.xlrelease.support.akka.spring.ScalaSpringSupport
import com.xebialabs.xltype.serialization.CiReference
import grizzled.slf4j.Logging
import org.springframework.context.{ApplicationContext, ApplicationContextAware}

import scala.beans.BeanProperty

trait ProxyBasedResolverRepository extends ResolverRepository with ApplicationContextAware with ScalaSpringSupport with Logging {
  self: Repository =>

  @BeanProperty
  protected var applicationContext: ApplicationContext = _

  def configurationPersistence: ConfigurationReferencePersistence

  def referenceTargetTypeResolver: CiReferenceTargetTypeResolver

  def releasePersistence: ReleasePersistence

  lazy val ciReferenceProxyFactory: CiReferenceProxyFactory = springBean[CiReferenceProxyFactory]

  /**
   * Resolves CiReference "just in time".
   * It should not load object from the database for id, equals, hashcode, type and other type related meta-methods.
   *
   * @param id actual ID of the object that should be loaded (ciReference has more than 1)
   * @param ciReference
   * @return proxy object that will resolve CiReference "just in time"
   */
  override def resolve(id: String, ciReference: CiReference): ConfigurationItem = {
    id match {
      // previously a special case: if it's an configuration object that does not exist - silently drop it
      case ConfigurationId(_) => this.read(id) // just read configurations
      // archived plan items
      case PlanItemId(_) if !releasePersistence.existsRelease(Ids.releaseIdFrom(id)) =>
        // if plan items _release_ cannot be found in the database - maybe it's archived
        // after the release is archived we should still see the dependency as completed (even after task restore)
        null
      case ValueProviderId(_) if Ids.isInRelease(id) =>
        // value providers are special case:
        //  - create release task (CRT) can use a template that has variables with value providers
        //  - when we select such a template server updates CRT with task related variables that may have value providers
        //    that point to the value provider of a template variable! <- this is IMPORTANT
        //  - that's why we have to resolve value providers directly here
        this.read(id)
      case VariableId(_) if Ids.isInRelease(id) =>
        // when release is created from a template we use ReleaseResource.createRelease(ReleaseForm)
        // ReleaseForm uses VariableView for variables, but unfortunately VariableView uses domain object ValueProviderConfiguration as valueProvider
        // as a consequence when ReleaseForm is deserialized we will get valueprovider JSON and deserializer will not be able to figure out it's a 1:1
        // nested property on a variable - that's why whenever we get a reference to a variable in a release as a CiReference we have to directly resolve it
        // this should not happen for normal execution paths
        this.read(id)
      case _ => doResolve(id, ciReference)
    }
  }

  protected def doResolve(id: String, ciReference: CiReference): ConfigurationItem = {
    val optionalTargetType: Option[Type] = referenceTargetTypeResolver.typeOf(id, ciReference)
    optionalTargetType match {
      case Some(targetType) =>
        val itemToResolve = ResolvableConfigurationItemReference(id, ciReference, targetType)
        val proxyItem = ciReferenceProxyFactory.proxy(self, itemToResolve)
        proxyItem
      case None =>
        if (XlrConfig.getInstance.features.serialization.failOnUnresolvableReferences) {
          throw new IllegalStateException(s"Unable to deduce type of ci reference: $ciReference for id: $id")
        } else {
          logger.warn(s"Unable to deduce type of ci reference: $ciReference for id: $id")
          null
        }
    }
  }
}
