package com.xebialabs.xlrelease.repository.sql.persistence

import com.xebialabs.xlrelease.db.sql.SqlBuilder.Dialect
import com.xebialabs.xlrelease.db.sql.transaction.IsTransactional
import com.xebialabs.xlrelease.repository.sql.persistence.Schema.PERSISTENT_LOGIN_TOKENS
import org.springframework.beans.factory.annotation.{Autowired, Qualifier}
import org.springframework.jdbc.core.JdbcTemplate
import org.springframework.security.web.authentication.rememberme.PersistentRememberMeToken
import org.springframework.stereotype.Repository

import java.util.Date
import scala.jdk.CollectionConverters._
import scala.util.Try

@Repository
@IsTransactional
class TokenPersistence @Autowired()(implicit @Qualifier("xlrRepositoryJdbcTemplate") val jdbcTemplate: JdbcTemplate,
                                    @Qualifier("xlrRepositorySqlDialect") implicit val dialect: Dialect) extends PersistenceSupport {

  private def params(map: (String, _ <: Any)*): Map[String, Any] = map.toMap[String, Any]

  private val STMT_INSERT_PERSISTENT_LOGIN_TOKENS =
    s"""|INSERT INTO ${PERSISTENT_LOGIN_TOKENS.TABLE}
        |   ( ${PERSISTENT_LOGIN_TOKENS.SERIES}
        |   , ${PERSISTENT_LOGIN_TOKENS.USERNAME}
        |   , ${PERSISTENT_LOGIN_TOKENS.TOKEN}
        |   , ${PERSISTENT_LOGIN_TOKENS.LAST_USED}
        |   )
        | VALUES
        |   ( :series
        |   , :username
        |   , :token
        |   , :lastUsed
        |   )
        """.stripMargin

  def insert(persistentRememberMeToken: PersistentRememberMeToken): Unit = {
    sqlUpdate(
      STMT_INSERT_PERSISTENT_LOGIN_TOKENS,
      params(
        "series" -> persistentRememberMeToken.getSeries,
        "username" -> persistentRememberMeToken.getUsername,
        "token" -> persistentRememberMeToken.getTokenValue,
        "lastUsed" -> persistentRememberMeToken.getDate
      ), _ => ())
  }

  private val STMT_DELETE_PERSISTENT_LOGIN_TOKENS =
    s"""|DELETE FROM ${PERSISTENT_LOGIN_TOKENS.TABLE}
        | WHERE ${PERSISTENT_LOGIN_TOKENS.USERNAME} = :username""".stripMargin

  def removeUserTokens(username: String): Unit = sqlExec(STMT_DELETE_PERSISTENT_LOGIN_TOKENS, params("username" -> username), ps => Try(ps.execute()))

  private val STMT_SELECT_PERSISTENT_LOGIN_TOKENS =
    s"""|SELECT ${PERSISTENT_LOGIN_TOKENS.SERIES}
        |   , ${PERSISTENT_LOGIN_TOKENS.TOKEN}
        |   , ${PERSISTENT_LOGIN_TOKENS.USERNAME}
        |   , ${PERSISTENT_LOGIN_TOKENS.LAST_USED} FROM ${PERSISTENT_LOGIN_TOKENS.TABLE}
        | WHERE ${PERSISTENT_LOGIN_TOKENS.SERIES} = :series
        | ORDER BY ${PERSISTENT_LOGIN_TOKENS.LAST_USED} DESC""".stripMargin

  def getTokenForSeries(series: String): PersistentRememberMeToken =
    sqlQuery(STMT_SELECT_PERSISTENT_LOGIN_TOKENS, params("series" -> series),
      rs => new PersistentRememberMeToken(
        rs.getString(PERSISTENT_LOGIN_TOKENS.USERNAME),
        rs.getString(PERSISTENT_LOGIN_TOKENS.SERIES),
        rs.getString(PERSISTENT_LOGIN_TOKENS.TOKEN),
        rs.getTimestamp(PERSISTENT_LOGIN_TOKENS.LAST_USED))
    ).headOption.orNull

  private val STMT_SELECT_PREVIOUS_PERSISTENT_LOGIN_TOKENS =
    s"""|SELECT ${PERSISTENT_LOGIN_TOKENS.SERIES}
        |   , ${PERSISTENT_LOGIN_TOKENS.TOKEN}
        |   , ${PERSISTENT_LOGIN_TOKENS.USERNAME}
        |   , ${PERSISTENT_LOGIN_TOKENS.LAST_USED} FROM ${PERSISTENT_LOGIN_TOKENS.TABLE}
        | WHERE ${PERSISTENT_LOGIN_TOKENS.SERIES} = :series
        | AND ${PERSISTENT_LOGIN_TOKENS.TOKEN} != :token
        | ORDER BY ${PERSISTENT_LOGIN_TOKENS.LAST_USED} DESC""".stripMargin

  def getPreviousTokenForSeries(series: String, token: String): Seq[PersistentRememberMeToken] = {
    sqlQuery(STMT_SELECT_PREVIOUS_PERSISTENT_LOGIN_TOKENS, params("series" -> series, "token" -> token),
      rs => new PersistentRememberMeToken(
        rs.getString(PERSISTENT_LOGIN_TOKENS.USERNAME),
        rs.getString(PERSISTENT_LOGIN_TOKENS.SERIES),
        rs.getString(PERSISTENT_LOGIN_TOKENS.TOKEN),
        rs.getTimestamp(PERSISTENT_LOGIN_TOKENS.LAST_USED))
    ).toSeq
  }

  private val STMT_DELETE_TOKEN_IN_SERIES_BEFORE_GIVEN_DATE =
    s"""|DELETE FROM ${PERSISTENT_LOGIN_TOKENS.TABLE}
        | WHERE ${PERSISTENT_LOGIN_TOKENS.SERIES} = :series
        | AND ${PERSISTENT_LOGIN_TOKENS.LAST_USED} < :date""".stripMargin

  def removeTokenInSeriesBeforeGivenDate(series: String, date: Date): Unit = {
    sqlExec(STMT_DELETE_TOKEN_IN_SERIES_BEFORE_GIVEN_DATE,
      params("series" -> series, "date" -> date),
      ps => Try(ps.execute()))
  }

  private val STMT_GET_USERS_FOR_SERIES =
    s"""|SELECT ${PERSISTENT_LOGIN_TOKENS.USERNAME} FROM
        |${PERSISTENT_LOGIN_TOKENS.TABLE} WHERE
        |${PERSISTENT_LOGIN_TOKENS.SERIES} = :series""".stripMargin

  private def getUsersBasedOnSeries(series: String): Seq[String] =
    sqlQuery(STMT_GET_USERS_FOR_SERIES, params("series" -> series), rs => rs.getString(PERSISTENT_LOGIN_TOKENS.USERNAME)).toSeq

  private val STMT_DELETE_PERSISTENT_LOGIN_TOKENS_WITH_SERIES =
    s"""|DELETE FROM ${PERSISTENT_LOGIN_TOKENS.TABLE}
        | WHERE ${PERSISTENT_LOGIN_TOKENS.USERNAME} IN
        | (:usernameList)""".stripMargin

  def removeUserTokensBasedOnSeries(series: String): Unit =
    sqlExec(STMT_DELETE_PERSISTENT_LOGIN_TOKENS_WITH_SERIES, params( "usernameList" -> getUsersBasedOnSeries(series).asJava ), ps => Try(ps.execute()))

}
