package com.xebialabs.xlrelease.spring.config

import com.xebialabs.deployit.security.PermissionEnforcer
import com.xebialabs.license.service.{DefaultLicenseCiCounterFactory, LicenseService, LicenseServiceFactory}
import com.xebialabs.license.{License, LicenseValidationFilter}
import com.xebialabs.xlplatform.repository.sql.SqlRepositoryMetadataService
import com.xebialabs.xlrelease.api.internal.InternalMetadataDecoratorService
import com.xebialabs.xlrelease.config.{ArchivingSettingsManager, XlrConfig}
import com.xebialabs.xlrelease.events.XLReleaseEventBus
import com.xebialabs.xlrelease.license.XLRLicenseInstallService
import com.xebialabs.xlrelease.repository.LicenseRepository
import com.xebialabs.xlrelease.repository.sql.persistence.{ReleaseCacheEventHandler, ReleaseCacheService}
import com.xebialabs.xlrelease.security.{PermissionChecker, PermissionCheckerHolder}
import com.xebialabs.xlrelease.service._
import com.xebialabs.xlrelease.views.converters.TasksViewConverter
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.context.annotation.{Bean, Configuration, Lazy}

@Configuration
class SqlServiceConfiguration(xlrConfig: XlrConfig,
                              sqlRepositories: SqlRepositoriesConfiguration,
                              sqlConfiguration: SqlConfiguration,
                              sqlSecurityConfiguration: SqlSecurityConfiguration,
                              sqlFoldersConfiguration: SqlFoldersConfiguration,
                              sqlPersistenceConfiguration: SqlPersistenceConfiguration, // should not use persistence
                              eventBus: XLReleaseEventBus) {

  @Autowired var decoratorService: InternalMetadataDecoratorService = _
  @Autowired var permissionEnforcer: PermissionEnforcer = _
  @Autowired var teamService: TeamService = _
  @Autowired var archivingService: ArchivingService = _
  @Autowired var archivingManager: ArchivingSettingsManager = _
  @Autowired
  @Lazy var taskService: TaskService = _
  @Autowired var permissionChecker: PermissionChecker = _
  @Autowired var broadcastService: BroadcastService = _
  @Autowired var licenseRepository: LicenseRepository = _

  @Bean
  def permissionCheckerHolder(): PermissionCheckerHolder.type = {
    PermissionCheckerHolder.init(permissionChecker)
    PermissionCheckerHolder
  }

  @Bean
  def preArchiveService(): PreArchiveService = {
    new PreArchiveService(xlrConfig, archivingService, sqlRepositories.releaseRepository(), archivingManager)
  }

  @Bean
  def releaseSearchService(): ReleaseSearchService = {
    new SqlReleaseSearchService(
      sqlPersistenceConfiguration.releasePersistence(),
      archivingService,
      sqlPersistenceConfiguration.sqlRepositoryAdapter(),
      teamService,
      decoratorService,
      sqlConfiguration.xlrRepositorySqlDialect(),
      permissionEnforcer,
      sqlSecurityConfiguration.roleService(),
      sqlRepositories.securedCis(),
      sqlRepositories.releaseExtensionsRepository())
  }

  @Bean
  def taskSearchService(taskAccessService: TaskAccessService, tasksViewConverter: TasksViewConverter): TaskSearchService = {
    new SqlTaskSearchService(
      sqlSecurityConfiguration.roleService(),
      taskAccessService,
      tasksViewConverter,
      decoratorService,
      sqlPersistenceConfiguration.releasePersistence(),
      sqlPersistenceConfiguration.taskPersistence(),
      sqlPersistenceConfiguration.commentPersistence(),
      sqlPersistenceConfiguration.sqlRepositoryAdapter(),
      permissionEnforcer,
      sqlConfiguration.xlrRepositorySqlDialect()
    )
  }

  @Bean
  //noinspection ScalaStyle
  def folderService(): FolderService = {
    new FolderService(sqlFoldersConfiguration.folders(),
      teamService,
      sqlRepositories.releaseRepository(),
      taskService,
      sqlRepositories.configurationRepository(),
      sqlConfiguration.ciIdService(),
      releaseSearchService(),
      permissionChecker,
      eventBus,
      archivingService)
  }

  @Bean
  def repositoryMetadataService() = new SqlRepositoryMetadataService(sqlConfiguration.xlrRepositoryDatabase())

  @Bean
  def licenseService(): LicenseService = {
    new LicenseServiceFactory(repositoryMetadataService(),
      new DefaultLicenseCiCounterFactory(),
      xlrConfig.rootConfig,
      xlrConfig.license_location,
      License.PRODUCT_XL_RELEASE
    ).createLicenseService()
  }

  @Bean
  def licenseCheckFilter(): LicenseValidationFilter = {
    val licenseValidationFilter = new LicenseValidationFilter
    licenseValidationFilter.setLicenseService(licenseService())
    licenseValidationFilter
  }

  @Bean
  def licenseInstallService(): XLRLicenseInstallService = {
    new XLRLicenseInstallService(broadcastService, licenseService(), licenseRepository)
  }

  @Bean
  def userTokenService(): UserTokenService = {
    new UserTokenService(eventBus, sqlRepositories.userTokenRepository())
  }

  @Bean
  def releaseCacheEventHandler: ReleaseCacheEventHandler = {
    new ReleaseCacheEventHandler()
  }

  @Bean
  def releaseCacheService: ReleaseCacheService = {
    ReleaseCacheService
  }
}
