package com.xebialabs.xlrelease.repository.sql

import com.codahale.metrics.annotation.Timed
import com.xebialabs.xlrelease.db.sql.transaction.IsTransactional
import com.xebialabs.xlrelease.domain.{Attachment, Release, Task}
import com.xebialabs.xlrelease.exception.LogFriendlyNotFoundException
import com.xebialabs.xlrelease.repository._
import com.xebialabs.xlrelease.repository.sql.persistence.{CiUid, ReleasePersistence}
import grizzled.slf4j.Logging

import java.io.InputStream
import scala.jdk.CollectionConverters._

@IsTransactional
class SqlAttachmentRepository(releasePersistence: ReleasePersistence)
  extends AttachmentRepository with Logging {

  @Timed
  override def findById(attachmentId: String): Attachment = {
    logger.debug(s"Searching for attachment [$attachmentId]")
    releasePersistence.findAttachmentById(attachmentId)
      .getOrElse(throw new LogFriendlyNotFoundException("Repository entity [%s] not found", attachmentId))
  }

  /**
    * An Attachment is added to the release, and is optionally attached to a Task.
    */
  @Timed
  override def create(release: Release, taskId: String, attachment: Attachment): Attachment = {
    logger.debug(s"Adding attachment to task [${taskId}]")
    val releaseTask = release.getTask(taskId)
    val taskAttachments = releaseTask.getAttachments.asScala.toSet + attachment
    releaseTask.setAttachments(taskAttachments.toList.asJava)
    create(release, attachment)
  }

  @Timed
  override def create(release: Release, attachment: Attachment): Attachment = {
    logger.debug(s"Adding attachment to release [${release.getId}]")
    release.getAttachments.add(attachment)
    releasePersistence.update(None, release)
    attachment
  }


  override def insertAttachment(releaseId: String, attachment: Attachment): Boolean = {
    releasePersistence.insertAttachments(releaseId, List(attachment))
  }

  @Timed
  override def delete(release: Release, attachment: Attachment): Unit = {
    logger.info(s"Deleting attachment reference '${attachment.getId}' from release '${release.getId}'")
    release.deleteAttachment(attachment.getId)
    releasePersistence.update(release)
    /*
    If template, we only delete the reference to attachment, but leave the attachment itself in the database.
    This is to allow us to work with template versioning, to restore attachment when reverting to a previous version of a template.
     */
    if (!release.isTemplate) {
      releasePersistence.deleteAttachmentById(attachment.getId, release.getCiUid)
    }
  }

  @Timed
  override def delete(task: Task, attachment: Attachment): Unit = {
    task.deleteAttachment(attachment.getId)
    val release = task.getRelease

    if (release.getTasksUsingAttachment(attachment.getId).isEmpty) {
      delete(release, attachment)
    } else {
      // Only delete the reference on the task
      logger.info(s"Deleting attachment reference ${attachment.getId} from task ${task.getId}")
      releasePersistence.update(release)
    }
  }

  def batchInsert(release: Release, attachments: Seq[Attachment]): Boolean = {
    try {
      releasePersistence.insertAttachments(release.getId, attachments)
    } catch {
      case e: RuntimeException =>
        logger.error(e.getMessage, e)
        false
    }
  }

  def insert(releaseUid: CiUid, artifactId: String, artifactName: String, content: InputStream): Boolean = {
    try {
      releasePersistence.insertArtifact(releaseUid, artifactId, artifactName, content)
    } catch {
      case e: RuntimeException =>
        logger.error(e.getMessage, e)
        false
    }
  }
}
