package com.xebialabs.xlrelease.repository.sql.persistence

import com.xebialabs.xlrelease.repository.sql.persistence.Schema.{FOLDERS, PATHS, VIEW}
import com.xebialabs.xlrelease.repository.sql.persistence.Utils.rowMapper
import com.xebialabs.xlrelease.repository.sql.persistence.data.FolderRow
import com.xebialabs.xlrelease.repository.sql.persistence.data.FolderRow.{FolderEntry, Root}
import com.xebialabs.xlrelease.security.sql.db.SecuritySchema.{ROLES, ROLE_PERMISSIONS, ROLE_PRINCIPALS, ROLE_ROLES}
import com.xebialabs.xlrelease.utils.FolderId
import org.springframework.jdbc.core.RowMapper


object FindDescendantsQueryBuilder {
  private val selectDescendants: String =
    s"""
       | SELECT DISTINCT
       |   d.${VIEW.FOLDER_ID},
       |   d.${VIEW.FOLDER_PATH},
       |   d.${VIEW.DESCENDANT_UID},
       |   d.${VIEW.SECURITY_UID},
       |   d.${VIEW.TOKEN},
       |   d.${VIEW.NAME},
       |   p.${PATHS.ANCESTOR_UID},
       |   d.${VIEW.DEPTH}
       | FROM ${FOLDERS.TABLE} f
       """.stripMargin

  private val descendantsOf: String =
    s"""
       | JOIN ${VIEW.TABLE} d
       |    ON ( d.${VIEW.ANCESTOR_UID} = f.${FOLDERS.CI_UID} AND d.${PATHS.DEPTH} <= :depth )
       """.stripMargin

  // parentOf (or root for itself)
  private val parentOf: String =
    s"""
       | JOIN ${PATHS.TABLE} p
       |    ON  ( p.${PATHS.DESCENDANT_UID} = d.${VIEW.DESCENDANT_UID}
       |          AND (     p.${PATHS.DEPTH} = 1
       |               OR ( p.${PATHS.DEPTH} = 0 AND p.${PATHS.ANCESTOR_UID} = ${Root.uid} ) ) )
       """.stripMargin

  private def findAncestorWithPermission(folderCondition: String): String =
    s"""
       | LEFT JOIN ${ROLES.TABLE} roles ON d.${FOLDERS.SECURITY_UID} = roles.${ROLES.ciId}
       | LEFT JOIN ${ROLE_PERMISSIONS.TABLE} permissions
       |  ON roles.${ROLES.id} = permissions.${ROLE_PERMISSIONS.roleId}
       | LEFT JOIN ${ROLE_ROLES.TABLE} teamRoles
       |  ON roles.${ROLES.id} = teamRoles.${ROLE_ROLES.roleId}
       | LEFT JOIN ${ROLE_PRINCIPALS.TABLE} rolePrincipals
       |  ON teamRoles.${ROLE_ROLES.memberRoleId} = rolePrincipals.${ROLE_PRINCIPALS.roleId}
       | LEFT JOIN ${ROLE_PRINCIPALS.TABLE} teamPrincipals
       |  ON roles.${ROLES.id} = teamPrincipals.${ROLE_PRINCIPALS.roleId}
       | WHERE
       |  $folderCondition
       |  AND (
       |    f.${FOLDERS.FOLDER_ID} = d.${FOLDERS.FOLDER_ID}
       |    OR (
       |      permissions.${ROLE_PERMISSIONS.permissionName} = :permissionName
       |      AND (
       |        LOWER(rolePrincipals.${ROLE_PRINCIPALS.principalName}) IN (:principalNames)
       |        OR
       |        LOWER(teamPrincipals.${ROLE_PRINCIPALS.principalName}) IN (:principalNames)
       |      )
       |    )
       |  )
     """.stripMargin

  private val folderIdCondition: String = s"f.${FOLDERS.FOLDER_ID} = :folderId"

  private val folderUidCondition: String = s"f.${FOLDERS.CI_UID} = :ciUid"

  private val findAncestorByIdAndPermission: String = findAncestorWithPermission(folderIdCondition)

  private val findAncestorById: String =
    s"""
       | WHERE $folderIdCondition
       """.stripMargin

  private val findAncestorByUidAndPermission: String = findAncestorWithPermission(folderUidCondition)

  private val findAncestorByUid: String =
    s"""
       | WHERE $folderUidCondition
       """.stripMargin

  private val grouped: String =
    s"""
       | GROUP BY
       |  d.${VIEW.DESCENDANT_UID},
       |  p.${PATHS.ANCESTOR_UID},
       |  d.${VIEW.SECURITY_UID},
       |  d.${VIEW.TOKEN},
       |  d.${VIEW.FOLDER_PATH},
       |  d.${VIEW.FOLDER_ID},
       |  d.${VIEW.NAME},
       |  d.${VIEW.DEPTH}
       """.stripMargin

  private val topDownOrder: String =
    s"""
       | ORDER BY
       |   d.${VIEW.DEPTH} ASC,
       |   d.${VIEW.NAME} ASC,
       |   p.${PATHS.ANCESTOR_UID} ASC,
       |   d.${VIEW.DESCENDANT_UID} ASC
      """.stripMargin

  val toFolderData: RowMapper[FolderRow] = rowMapper { rs =>
    val uid = rs.getInt(VIEW.DESCENDANT_UID)
    if (uid == Root.uid) {
      Root
    } else {
      FolderEntry(
        uid = uid,
        token = Token.fromString(rs.getString(VIEW.TOKEN)).get,
        securityUid = rs.getInt(VIEW.SECURITY_UID),
        folderId = FolderId(rs.getString(VIEW.FOLDER_PATH)) / rs.getString(VIEW.FOLDER_ID),
        name = rs.getString(VIEW.NAME),
        parentUid = rs.getInt(PATHS.ANCESTOR_UID),
        depth = rs.getInt(VIEW.DEPTH)
      )
    }
  }

  def findByUid: String = selectDescendants ++
    descendantsOf ++
    parentOf ++
    findAncestorByUid ++
    grouped ++
    topDownOrder

  def findByUidHavingPermission: String = selectDescendants ++
    descendantsOf ++
    parentOf ++
    findAncestorByUidAndPermission ++
    grouped ++
    topDownOrder

  def findById: String = selectDescendants ++
    descendantsOf ++
    parentOf ++
    findAncestorById ++
    grouped ++
    topDownOrder

  def findByIdHavingPermission: String = selectDescendants ++
    descendantsOf ++
    parentOf ++
    findAncestorByIdAndPermission ++
    grouped ++
    topDownOrder
}
