package com.xebialabs.xlrelease.repository.sql

import com.codahale.metrics.annotation.Timed
import com.xebialabs.deployit.checks.Checks.checkArgument
import com.xebialabs.deployit.exception.NotFoundException
import com.xebialabs.deployit.plugin.api.reflect.{PropertyKind, Type}
import com.xebialabs.deployit.plumbing.serialization.ResolutionContext
import com.xebialabs.deployit.repository.{ItemAlreadyExistsException, ItemInUseException}
import com.xebialabs.xlrelease.db.sql.transaction.{IsReadOnly, IsTransactional}
import com.xebialabs.xlrelease.domain.PropertyConfiguration.PROPERTY_CARDINALITY
import com.xebialabs.xlrelease.domain._
import com.xebialabs.xlrelease.domain.variables.reference.UsagePoint
import com.xebialabs.xlrelease.repository.query.ReleaseBasicData
import com.xebialabs.xlrelease.repository.sql.persistence.configuration.ConfigurationPersistence.ConfigurationRow
import com.xebialabs.xlrelease.repository.sql.persistence.configuration._
import com.xebialabs.xlrelease.repository.sql.persistence.{CiUid, FolderPersistence}
import com.xebialabs.xlrelease.repository.{ConfigurationRepository, Ids}
import com.xebialabs.xlrelease.service.ConfigurationVariableService
import com.xebialabs.xlrelease.utils.TypeHelper
import com.xebialabs.xlrelease.validation.{ExtendedValidationContextImpl, XlrValidationsFailedException}
import grizzled.slf4j.Logging

import java.util.{Optional, List => JList}
import scala.collection.mutable.ListBuffer
import scala.compat.java8.OptionConverters._
import scala.jdk.CollectionConverters._

@IsTransactional
class SqlConfigurationRepository(configurationPersistence: ConfigurationPersistence,
                                 releaseConfigurationReferencePersistence: ReleaseConfigurationReferencePersistence,
                                 triggerConfigurationReferencePersistence: TriggerConfigurationReferencePersistence,
                                 sqlRepositoryAdapter: SqlRepositoryAdapter,
                                 folderPersistence: FolderPersistence) extends ConfigurationRepository
  with Logging {

  @Timed
  override def create[T <: BaseConfiguration](configuration: T): T = {
    create(configuration, getFolderCiUid(configuration.getFolderId))
  }

  @Timed
  override def create[T <: BaseConfiguration](configuration: T, folderCiUid: CiUid): T = {
    checkCardinality(configuration)
    validate(configuration)
    interceptCreate(configuration)
    configurationPersistence.insert(configuration, folderCiUid)
    afterCreate(configuration)
    configuration
  }

  private def getFolderCiUid(folderId: String): CiUid = {
    folderId match {
      case null => null
      case folderId => folderPersistence.getUid(folderId)
    }
  }

  private def checkCardinality[T <: BaseConfiguration](configuration: T): Unit = {
    val configurationType = Type.valueOf(classOf[BaseConfiguration])
    import com.xebialabs.xlrelease.repository.sql.SqlConfigurationRepository._

    def groupType[C <: BaseConfiguration](configuration: C): Option[Type] = {
      val superClasses = configuration.getType.getDescriptor.getSuperClasses
      // the first one where cardinality matches the cardinality of the config itself
      superClasses.asScala.find { superType =>
        superType.isSubTypeOf(configurationType) && superType.hasCardinality &&
          superType.getDescriptor.getPropertyDescriptor(PROPERTY_CARDINALITY).getDefaultValue.asInstanceOf[Int] == configuration.getCardinality
      }
    }

    if (configuration.hasCardinality) {
      val configGroupType = groupType(configuration).getOrElse(configuration.getType)
      val foundConfigs = findAllByTypeAndTitle[BaseConfiguration](configGroupType, title = null, configuration.getFolderId, folderOnly = false).asScala
        .filter(_.getFolderId == configuration.getFolderId)
      val cardinality = configuration.getCardinality
      if (cardinality.isDefined && foundConfigs.size >= cardinality.get) {
        val folderId = if (configuration.getFolderId == null) Ids.ROOT_FOLDER_ID else configuration.getFolderId
        throw new ItemAlreadyExistsException(s"There are already ${foundConfigs.length} items of type '$configGroupType' defined on the folder '$folderId'")
      }
    }
  }


  @Timed
  override def read[T <: BaseConfiguration](configurationId: String): T =
    Option(sqlRepositoryAdapter.read(configurationId))
      .getOrElse(throw new NotFoundException(s"Configuration $configurationId not found"))

  @Timed
  override def update[T <: BaseConfiguration](configuration: T): T = {
    update(configuration, getFolderCiUid(configuration.getFolderId))
  }

  @Timed
  override def update[T <: BaseConfiguration](configuration: T, folderCiUid: CiUid): T = {
    validate(configuration)
    interceptUpdate(configuration)
    configurationPersistence.update(configuration)
    afterUpdate(configuration)
    configuration
  }

  @Timed
  override def findAllByType[T <: BaseConfiguration](ciType: Type): JList[T] =
    configurationPersistence.findByTypes(getAllSubTypesOf(ciType))
      .flatMap(readConfiguration[T])
      .toBuffer
      .asJava

  @Timed
  override def findAllByTypeAndTitle[T <: BaseConfiguration](ciType: Type, title: String): JList[T] = {
    findAllByTypeAndTitle(ciType, title, folderId = null, folderOnly = false)
  }


  @Timed
  override def findAllByTypeAndTitle[T <: BaseConfiguration](ciType: Type, title: String = null, folderId: String, folderOnly: Boolean): JList[T] = {
    configurationPersistence.findByTypesTitleAndFolder(
      getAllSubTypesOf(ciType),
      Option(title).filter(_.trim.nonEmpty),
      Option(folderId).map(if (folderOnly) Right.apply else Left.apply)
    ).flatMap(readConfiguration[T]).toBuffer.asJava
  }

  private def readConfiguration[T <: BaseConfiguration](row: ConfigurationRow): Option[T] = {
    row match {
      case (folderIdOpt, rawConfiguration) =>
        sqlRepositoryAdapter.deserialize[T](rawConfiguration).map { conf =>
          folderIdOpt.fold(conf) { folderId =>
            conf.setFolderId(folderId.absolute)
            conf
          }
        }
    }
  }

  @Timed
  @IsReadOnly
  def existsByTypeAndTitle[T <: BaseConfiguration](ciType: Type, title: String = null): Boolean = {
    configurationPersistence.existsByTypeAndTitle(getAllSubTypesOf(ciType), title)
  }

  @Timed
  override def findFirstByType[T <: BaseConfiguration](ciType: Type): Optional[T] = {
    findFirstByType(ciType, ResolutionContext.GLOBAL)
  }

  @Timed
  override def findFirstByType[T <: BaseConfiguration](ciType: Type, context: ResolutionContext): Optional[T] = {
    configurationPersistence.findFirstByTypes(getAllSubTypesOf(ciType), context.folderId)
      .flatMap(readConfiguration[T])
      .asJava
  }

  @Timed
  @IsReadOnly
  override def exists(configurationId: String): Boolean = configurationPersistence.exists(configurationId)

  private def checkDelete(configurationId: String, configurationReferencePersistence: ConfigurationReferencePersistence): Unit = {
    if (configurationReferencePersistence.isReferenced(configurationId)) {
      val referencing = configurationReferencePersistence.getReferencingEntities(configurationId)
      val entities = referencing.map(data =>
        s"[${data.title} (${data.id})]"
      )
      throw new ItemInUseException("%s is still referenced by %s", configurationId, entities.mkString(", "))
    }
  }

  @Timed
  override def delete(configurationId: String): Unit = {
    checkDelete(configurationId, releaseConfigurationReferencePersistence)
    checkDelete(configurationId, triggerConfigurationReferencePersistence)
    interceptDelete(configurationId)
    configurationPersistence.delete(configurationId)
    afterDelete(configurationId)
  }

  @Timed
  override def getReferenceReleases(configurationId: String): JList[ReleaseBasicData] = {
    releaseConfigurationReferencePersistence.getReferencingEntities(configurationId).map(_.asReleaseData).asJava
  }

  private def getAllSubTypesOf[T <: BaseConfiguration](ciType: Type) = {
    TypeHelper.getAllSubtypesOf(ciType).map(_.toString)
  }

  @Timed
  override def getAllTypes: Seq[String] = configurationPersistence.findAllConfigurationTypes

  @Timed
  override def deleteByTypes(ciTypes: Seq[String]): Unit = {
    val configurationCiUids = configurationPersistence.findUidsByTypes(ciTypes)
    releaseConfigurationReferencePersistence.deleteRefsByConfigurationUids(configurationCiUids)
    triggerConfigurationReferencePersistence.deleteRefsByConfigurationUids(configurationCiUids)
    configurationPersistence.deleteByTypes(ciTypes)
  }

  private def validate(configuration: BaseConfiguration): Unit = {
    val passVarUsagePoints = ListBuffer.empty[UsagePoint]
    configuration match {
      case conf: Configuration =>
        for {
          usagePoints <- ConfigurationVariableService.getUsagePointsByVars(Seq(conf)).values
          up <- usagePoints
        } {
          val targetProperty = up.getTargetProperty
          val desc = targetProperty.getDescriptor
          checkArgument(desc.getKind == PropertyKind.STRING && desc.isPassword,
            "Non-password-type variables in configuration variableMapping are not supported")
          // Hack warning: temporarily set variableMapped password variables to some value
          // to avoid validation errors on required password properties
          // ideally we should make platform validators aware of variableMapping
          targetProperty.setValue("<temporary_value>")
          passVarUsagePoints += up
        }
      case _ =>
    }

    val desc = configuration.getType.getDescriptor
    val extendedValidationContext = new ExtendedValidationContextImpl(configuration)
    desc.validate(extendedValidationContext, configuration)
    val messages = extendedValidationContext.getMessages
    if (!messages.isEmpty) {
      configuration.get$validationMessages().addAll(messages)
    }
    if (!configuration.get$validationMessages().isEmpty) {
      throw new XlrValidationsFailedException(configuration)
    }

    passVarUsagePoints.toList.foreach { up =>
      up.getTargetProperty.setValue(null)
    }
  }
}

object SqlConfigurationRepository {

  private implicit class TypeOps(xlType: Type) {
    def hasCardinality: Boolean = hasProperty(PROPERTY_CARDINALITY)

    def hasProperty(propertyName: String): Boolean = xlType.getDescriptor.getPropertyDescriptor(propertyName) != null
  }

  private implicit class BaseConfigurationOps(ci: BaseConfiguration) {
    private val ciType = ci.getType

    def hasCardinality: Boolean = ci.hasProperty(PROPERTY_CARDINALITY)

    def getCardinality: Option[Int] = {
      if (hasCardinality) {
        Some(ciType.getDescriptor.getPropertyDescriptor(PROPERTY_CARDINALITY).getDefaultValue.asInstanceOf[Int])
      } else {
        None
      }
    }
  }

}
