package com.xebialabs.xlrelease.repository.sql

import com.codahale.metrics.annotation.Timed
import com.xebialabs.deployit.repository.{ItemAlreadyExistsException, ItemInUseException}
import com.xebialabs.xlrelease.db.sql.SqlBuilder.Dialect
import com.xebialabs.xlrelease.db.sql.transaction.{IsReadOnly, IsTransactional}
import com.xebialabs.xlrelease.domain.events.ReleaseCreationSource
import com.xebialabs.xlrelease.domain.status.ReleaseStatus
import com.xebialabs.xlrelease.domain.status.ReleaseStatus.TEMPLATE
import com.xebialabs.xlrelease.domain.utils.syntax._
import com.xebialabs.xlrelease.domain.{Release, ReleaseExtension, Task}
import com.xebialabs.xlrelease.exception.LogFriendlyNotFoundException
import com.xebialabs.xlrelease.repository._
import com.xebialabs.xlrelease.repository.sql.persistence.CiId._
import com.xebialabs.xlrelease.repository.sql.persistence.CommentPersistence.CommentRow.fromComment
import com.xebialabs.xlrelease.repository.sql.persistence.CommentPersistence.TaskCommentRow
import com.xebialabs.xlrelease.repository.sql.persistence.DependencyPersistence.TaskDependency
import com.xebialabs.xlrelease.repository.sql.persistence.TaskTagsPersistence.TaskTag
import com.xebialabs.xlrelease.repository.sql.persistence.TasksSqlBuilder.normalizeTag
import com.xebialabs.xlrelease.repository.sql.persistence._
import com.xebialabs.xlrelease.repository.sql.persistence.configuration.ConfigurationReferencePersistence
import com.xebialabs.xlrelease.security.SecuredCi
import com.xebialabs.xlrelease.serialization.json.repository.{ResolveOptions, ResolveOptionsBuilder}
import com.xebialabs.xlrelease.utils.Diff
import grizzled.slf4j.Logging

import java.util.Collections._
import java.util.{Date, List => JList, Set => JSet}
import scala.annotation.varargs
import scala.jdk.CollectionConverters._

@IsTransactional
class SqlReleaseRepository(releasePersistence: ReleasePersistence,
                           protected val taskPersistence: TaskPersistence,
                           protected val dependencyPersistence: DependencyPersistence,
                           protected val commentPersistence: CommentPersistence,
                           protected val teamRepository: TeamRepository,
                           protected val releaseExtensionsRepository: ReleaseExtensionsRepository,
                           val configurationPersistence: ConfigurationReferencePersistence,
                           val repositoryAdapter: SqlRepositoryAdapter,
                           facetRepositoryDispatcher: FacetRepositoryDispatcher)(
                            implicit val sqlDialect: Dialect)
  extends ReleaseRepository
    with InterceptedRepository[Release]
    with Logging
    with DeserializationSupport
    with ConfigurationReferencesSupport
    with SqlReleaseRepositoryHelper {

  @Timed
  @IsReadOnly
  override def exists(id: String): Boolean = {
    releasePersistence.existsRelease(id.normalized)
  }

  @Timed
  override def create(release: Release, releaseCreationSource: ReleaseCreationSource): Release = {
    if (releasePersistence.existsRelease(release.getId)) {
      throw new ItemAlreadyExistsException("A release with ID [%s] already exists", release.getId)
    }
    interceptCreate(release)
    releasePersistence.insert(release)

    val tasks = release.getAllTasks.asScala.toSet
    val releaseUid = release.getCiUid
    batchInsertTasks(releaseUid, tasks)

    release.getExtensions.forEach(releaseExtensionsRepository.create[ReleaseExtension])
    updateConfigurationRefs(release)
    facetRepositoryDispatcher.liveRepository.createFromTasks(tasks.toSeq)
    release
  }

  private def batchInsertTasks(releaseUid: Integer, tasks: Set[Task]): Unit = {
    taskPersistence.batchInsert(tasks, releaseUid)
    // it looks like some databases do not return generated keys after executeBatch is called
    val taskUidsByTaskId: Map[String, CiUid] = taskPersistence.releaseTaskUids(releaseUid)
    tasks.foreach(task => task.setCiUid(taskUidsByTaskId(Ids.getFolderlessId(task.getId))))
    val taskTags = for {
      t <- tasks
      tag <- Set.from(t.getTags.asScala.map(normalizeTag))
      taskUid <- taskUidsByTaskId.get(t.getId.shortId)
    } yield TaskTag(taskUid, tag)
    taskPersistence.batchInsertTags(taskTags.toList)

    val taskCommentRows = for {
      t <- tasks
      comment <- t.getComments.asScala
      taskUid <- taskUidsByTaskId.get(t.getId.shortId)
    } yield TaskCommentRow(taskUid, commentRow = fromComment(comment))
    commentPersistence.batchCreate(taskCommentRows)

    val taskDependencies = for {
      t <- tasks
      dependency <- t.dependencies
      taskUid <- taskUidsByTaskId.get(t.getId.shortId)
    } yield TaskDependency(taskUid, dependency)
    dependencyPersistence.batchInsert(taskDependencies)

  }

  @Timed
  override def findById(id: String): Release = {
    // TODO this shouldn't load everything, to fix on 9.1
    getRelease(id, new ResolveOptionsBuilder().withEverything.build)
  }

  @Timed
  override def findById(id: String, resolveOptions: ResolveOptions): Release = {
    getRelease(id, resolveOptions)
  }

  @varargs
  @Timed
  @IsReadOnly
  override def findIdsByStatus(statuses: ReleaseStatus*): Seq[String] = {
    val sqlWithParams = new ReleasesSqlBuilder()
      .selectReleaseId()
      .withOneOfStatuses(statuses)
      .build()
    releasePersistence.findReleaseIdsByQuery(sqlWithParams).toSeq
  }

  @Timed
  override def findByCalendarToken(calendarToken: String): Release = {
    val sqlWithParams = new ReleasesSqlBuilder()
      .selectReleaseData()
      .withCalendarToken(calendarToken)
      .build()
    val releases = releasePersistence.findReleaseDatasByQuery(sqlWithParams)
    releases
      .headOption
      .map(deserializeRelease)
      .orNull
  }

  @Timed
  @IsReadOnly
  def findArchivableReleaseIds(date: Date, pageSize: Int): Seq[String] = {
    val sqlWithParams = new ReleasesSqlBuilder()
      .selectReleaseId()
      .withOneOfStatuses(Seq(ReleaseStatus.COMPLETED, ReleaseStatus.ABORTED))
      .withEndDateBefore(date)
      .withPage(Page(0, pageSize, 0))
      .withPreArchived()
      .build()

    releasePersistence.findReleaseIdsByQuery(sqlWithParams).toSeq
  }

  @Timed
  override def setPreArchived(releaseId: String, preArchived: Boolean): Unit = {
    releasePersistence.setPreArchived(releaseId, preArchived)
  }

  @Timed
  @IsReadOnly
  def findPreArchivableReleases(page: Int, pageSize: Int): Seq[Release] = {
    val inactive = Seq(ReleaseStatus.COMPLETED, ReleaseStatus.ABORTED)
    val sqlWithParams = new ReleasesSqlBuilder()
      .selectReleaseData()
      .withOneOfStatuses(inactive)
      .withPreArchived(false)
      .limitAndOffset(pageSize, page * pageSize)
      .build()

    releasePersistence.findReleaseDatasByQuery(sqlWithParams)
      .flatMap(tryDeserializeRelease)
      .filterNot(_.isTutorial)
      .map(releaseExtensionsRepository.decorate)
      .map(commentPersistence.decorate)
  }

  @Timed
  override def search(searchParams: ReleaseSearchByParams): JList[Release] = searchParams match {
    case ReleaseSearchByParams(page, folderId, statuses, title, rootReleaseId, autoStart) =>
      val sqlBuilder = {
        val base = new ReleasesSqlBuilder()
          .selectReleaseData()
          .withTitle(title)
          .withRootReleaseId(rootReleaseId)
          .withPage(page)

        if (folderId == null) {
          base
        } else {
          folderId.fold(base.withFolder, base.withAncestor)
        }
      }
      if (statuses.nonEmpty) {
        sqlBuilder.withOneOfStatuses(statuses.toIndexedSeq)
      }
      if (autoStart) {
        sqlBuilder.withAutoStart()
      }
      releasePersistence
        .findReleaseDatasByQuery(sqlBuilder.build())
        .flatMap(tryDeserializeRelease)
        .map(releaseExtensionsRepository.decorate)
        .map(commentPersistence.decorate).asJava
  }

  @Timed
  override def delete(id: String, cleanActivityLog: Boolean): Unit = {
    checkIsNotReferencedByDependencies(id)
    releasePersistence.findUidByReleaseId(id).foreach { uid =>
      deleteReleaseReferences(id, uid.intValue(), cleanActivityLog)
    }
    releasePersistence.deleteById(id)
  }

  private def deleteReleaseReferences(id: String, releaseUid: Int, cleanActivityLog: Boolean): Unit = {
    dependencyPersistence.deleteByReleaseUid(releaseUid)
    val taskCiUids = taskPersistence.findTaskCiUidsByReleaseCiUid(releaseUid)
    dependencyPersistence.deleteByTaskUids(taskCiUids)
    commentPersistence.deleteByRelease(releaseUid, taskCiUids)
    taskPersistence.deleteTasksByReleaseUid(releaseUid)
    teamRepository.deleteTeamsFromPlatform(new SecuredCi(id, releaseUid))
    releaseExtensionsRepository.deleteAll(id)
    deleteConfigurationRefs(releaseUid)
    interceptDelete(id)
    if (cleanActivityLog) {
      cleanActivityLogs(id)
    }
  }

  @Timed
  override def deleteWithUid(id: String, releaseUid: Int): Unit = {
    deleteReleaseReferences(id, releaseUid, true)
    releasePersistence.deleteById(id)
  }

  @Timed
  override def move(originalId: String, newId: String): Unit = {
    releasePersistence.move(originalId, newId)
  }

  override def getUid(id: String): Option[CiUid] = releasePersistence.findUidByReleaseId(id)

  @Timed
  override def getStatus(id: String): ReleaseStatus = {
    val status = releasePersistence.findReleaseStatus(id.normalized).orNull
    if (status == null) {
      null
    } else {
      ReleaseStatus.valueOf(status.toUpperCase)
    }
  }

  @Timed
  override def getStatuses(ids: Seq[String]): Seq[ReleaseStatus] = {
    if (ids.isEmpty) Seq() else releasePersistence.findReleaseStatuses(ids).map(status => ReleaseStatus.valueOf(status.toUpperCase))
  }


  @Timed
  @IsReadOnly
  override def getRiskScores(ids: Seq[String]): Seq[Int] = {
    releasePersistence.findReleaseRiskScores(ids)
  }

  @Timed
  override def setRiskScores(releaseId: String, score: Int, totalScore: Int): Unit =
    releasePersistence.updateReleaseRiskScores(releaseId, score, totalScore)

  @Timed
  @IsReadOnly
  override def isTemplate(releaseId: String): Boolean =
    TEMPLATE == getStatus(releaseId)

  @Timed
  @IsReadOnly
  override def getTitle(id: String): String =
    releasePersistence.findReleaseTitle(id.normalized)
      .getOrElse(throw new LogFriendlyNotFoundException("Release [%s] not found", id))

  @Timed
  override def update(release: Release): Release = {
    update(null, release)
  }

  @Timed
  override def update(original: Release, updated: Release): Release = {
    val originalReleaseOpt = Option(original)

    interceptUpdate(updated)
    releasePersistence.update(Option(original), updated)

    originalReleaseOpt.foreach { rel =>
      // phase restart can add new tasks and is done through release update
      val tasks = Diff(
        rel.getAllTasks.asScala,
        updated.getAllTasks.asScala
      ).newValues.toSet
      batchInsertTasks(releaseUid = updated.getCiUid, tasks)
    }
    updateConfigurationRefs(updated)
    updated
  }

  //noinspection ScalaStyle
  @Timed
  override def replace(original: Release, updated: Release): Release = {

    if (Ids.isFolderId(original.getId)) {
      updated.setTeams(emptyList())
    }

    interceptUpdate(updated)
    releasePersistence.update(Option(original), updated)
    if (original != null) {
      // phase restart can add new tasks and is done through release update
      replaceTasks(original, updated)
      replaceReleaseExtensions(original, updated)
      replaceReleaseTeams(original, updated)
    }
    updateConfigurationRefs(updated)
    updated
  }


  @Timed
  @IsReadOnly
  override def getAllTags(limitNumber: Int): JSet[String] = releasePersistence.findAllTags(limitNumber).asJava

  private def getRelease(releaseId: CiId, resolveOptions: ResolveOptions): Release = {
    val release = repositoryAdapter.read[Release](releaseId)
    if (null == release) {
      throw new LogFriendlyNotFoundException(s"Repository entity [$releaseId] not found")
    }
    if (resolveOptions.hasDecorators) {
      releaseExtensionsRepository.decorate(release)
      commentPersistence.decorate(release) // <-- can be HUGE! ENG-8031: lazy loading from UI side?
    }
    release
  }

  private def checkIsNotReferencedByDependencies(planItemIdOrItsChildren: CiId): Unit = {
    val externalIncomingDependencies = dependencyPersistence.findByPartialTargetIds(Set(planItemIdOrItsChildren), None)
      .map(_.fullDependencyId)
      .filterNot(_.startsWith(planItemIdOrItsChildren.normalized))
    if (externalIncomingDependencies.nonEmpty) {
      throw new ItemInUseException(s"Cannot delete [$planItemIdOrItsChildren] because it or one of its children is referenced by " +
        s"one or more dependencies: [${externalIncomingDependencies.mkString(", ")}]")
    }
  }

  @Timed
  @IsReadOnly
  override def findSCMDataById(id: String): Option[Integer] = {
    logger.debug(s"Finding scm data for template or release with id $id")
    releasePersistence.findSCMDataById(id).map(Integer.valueOf)
  }

  @Timed
  @IsReadOnly
  override def findOverdueReleaseIds(): Seq[String] = {
    releasePersistence.findOverdueReleaseIds()
  }

  @Timed
  @IsReadOnly
  override def getReleaseJson(releaseId: String): String = {
    releasePersistence.getReleaseJson(releaseId)
      .getOrElse(throw new LogFriendlyNotFoundException(s"Repository entity [$releaseId] not found"))
  }

  @Timed
  @IsReadOnly
  override def getFullId(releaseId: String): String = {
    val row = releasePersistence.getBasicReleaseRow(releaseId)
      .getOrElse(throw new LogFriendlyNotFoundException(s"Repository entity [$releaseId] not found"))
    Ids.formatWithFolderId(row.folderId, row.releaseId)
  }

  @Timed
  @IsReadOnly
  override def getReleasesWithoutComments(releaseIds: List[String]): List[Release] = {
    val sqlWithParams = new ReleasesSqlBuilder().selectReleaseData()
      .withReleaseIds(releaseIds).build()
    val releases = releasePersistence.findReleaseDatasByQuery(sqlWithParams)
      .flatMap(tryDeserializeRelease)
      .map(releaseExtensionsRepository.decorate)

    releases.sortBy(r => releaseIds.indexOf(Ids.getName(r.getId))).toList
  }

  // Hack to work around circular dependencies in code and make it so activity log cleanup
  // can be handled separately from the other PersistenceInterceptors
  var activityLogCleaners: Seq[PersistenceInterceptor[Release]] = Seq.empty

  def registerActivityLogCleaner(cleaner: PersistenceInterceptor[Release]): Unit = {
    activityLogCleaners = cleaner +: activityLogCleaners
  }

  private def cleanActivityLogs(ciId: String): Unit = {
    activityLogCleaners.foreach(_.onDelete(ciId))
  }
}
