package com.xebialabs.xlrelease.script;

import java.io.File;
import java.io.FilePermission;
import java.lang.reflect.ReflectPermission;
import java.net.NetPermission;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.security.*;
import java.util.Arrays;
import java.util.Enumeration;
import java.util.List;
import java.util.PropertyPermission;
import java.util.stream.Stream;
import javax.management.MBeanPermission;
import javax.management.ObjectName;
import javax.script.ScriptContext;
import javax.script.ScriptEngine;
import javax.script.ScriptException;
import javax.security.auth.AuthPermission;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.util.StringUtils;

import com.xebialabs.xlrelease.script.jython.SandboxAwarePackageManager;
import com.xebialabs.xlrelease.script.security.ScriptCodeSource;

import static com.xebialabs.xlrelease.script.security.ScriptPermissions.grantedClassAccess;
import static java.lang.System.getSecurityManager;
import static java.util.stream.Collectors.joining;

public abstract class Jsr223ScriptExecutor implements ScriptExecutor {
    protected final Logger logger = LoggerFactory.getLogger("com.xebialabs.xlrelease.script.Jsr223ScriptExecutor");
    private final boolean sandboxEnabled;
    private final String workDir;
    private List<String> jsonSmartJars;
    private List<String> jythonStandaloneJars;
    private List<String> groovyJars;

    private Jsr223EngineFactory engineFactory;
    protected ScriptEngine unrestrictedEngine;
    protected ScriptEngine restrictedEngine;
    protected ScriptPermissionsProvider scriptPermissionsProvider;

    protected Jsr223ScriptExecutor(Jsr223EngineFactory engineFactory,
                                   ScriptPermissionsProvider scriptPermissionsProvider,
                                   boolean sandboxEnabled,
                                   String workDir) {
        this.engineFactory = engineFactory;
        this.scriptPermissionsProvider = scriptPermissionsProvider;
        this.sandboxEnabled = sandboxEnabled;
        this.workDir = workDir;
        this.initJarLocations();
        this.reloadScriptEngines();
    }

    private void initJarLocations() {
        this.jsonSmartJars = getJarsLocations("json-smart");
        this.jythonStandaloneJars = getJarsLocations("jython-standalone");
        this.groovyJars = getJarsLocations("groovy");
    }

    private static final String CLASS_PATH = System.getProperty("java.class.path");
    private static final String CLASS_PATH_SEPARATOR = System.getProperty("path.separator");

    private List<String> getJarsLocations(String jarName) {
        List<String> jars = Arrays.stream(CLASS_PATH.split(CLASS_PATH_SEPARATOR)).filter(s -> s.contains(jarName)).toList();
        for (String jar : jars) {
            logger.debug("Minimal permission added on: {}", jar);
        }
        return jars;
    }

    public Object evalScript(XlrScriptContext xlrScriptContext) throws Exception {
        // all of the scripts need to share the same engine - the only difference is the access control context
        Object result = null;
        try {
            boolean checkPolicyPermissions = xlrScriptContext.shouldCheckPolicyPermissions();
            AccessControlContext acc = checkPolicyPermissions ? getAccessControlContext() : null;
            ScriptEngine engine = configureSandboxAndGetEngine(checkPolicyPermissions);
            xlrScriptContext.beforeExecute();
            for (XlrScript script : xlrScriptContext.getScripts()) {
                try {
                    result = evaluate(engine, xlrScriptContext, script, script.checkPermissions() ? acc : null);
                } catch (ScriptException ex) {
                    if (xlrScriptContext.isTimedOut()) {
                        String settingsMsg = String.format("Check value of 'xl.timeouts.%s' configuration property.", xlrScriptContext.getTimeoutSettingKey());
                        throw new ScriptTimeoutException("Script timed out. " + settingsMsg, ex);
                    } else {
                        throw ex;
                    }
                } catch (Exception ex) {
                    logger.warn("Failed script {} '{}'", xlrScriptContext.getExecutionId(), script.name(), ex);
                    throw new ScriptException(ex);
                } catch (Throwable t) {
                    logger.error("Failed script {} '{}'", xlrScriptContext.getExecutionId(), script.name(), t);
                    String msg = String.format("Failed script %s. Cause: %s. Message: %s", xlrScriptContext.getExecutionId(), t.getClass(), t.getMessage());
                    ScriptException scriptException = new ScriptException(msg);
                    scriptException.initCause(t);
                    throw scriptException;
                }
            }
        } finally {
            xlrScriptContext.afterExecute();
            SandboxAwarePackageManager.setSandboxed(false);
        }
        return result;
    }

    private Object evaluate(ScriptEngine engine, XlrScriptContext xlrScriptContext, XlrScript script, AccessControlContext acc) throws Exception {
        // make sure we wrap/unwrap only the scripts that declare it
        Object originalBindingsToWrap = xlrScriptContext.getAttribute(XlrScriptContext.CONTEXT_BINDINGS_TO_WRAP, ScriptContext.ENGINE_SCOPE);
        Object originalBindingsToUnWrap = xlrScriptContext.getAttribute(XlrScriptContext.CONTEXT_BINDINGS_TO_UNWRAP, ScriptContext.ENGINE_SCOPE);
        try {
            if (!script.wrap()) {
                xlrScriptContext.removeAttribute(XlrScriptContext.CONTEXT_BINDINGS_TO_WRAP, ScriptContext.ENGINE_SCOPE);
                xlrScriptContext.removeAttribute(XlrScriptContext.CONTEXT_BINDINGS_TO_UNWRAP, ScriptContext.ENGINE_SCOPE);
            }
            logger.trace("Evaluating script {}: {}", xlrScriptContext.getExecutionId(), script.name());
            String scriptContent = script.scriptSource().scriptContent();
            Object result = evalScriptPrivileged(engine, scriptContent, xlrScriptContext, acc);
            logger.trace("Evaluated script {}", xlrScriptContext.getExecutionId());
            return result;
        } finally {
            xlrScriptContext.setAttribute(XlrScriptContext.CONTEXT_BINDINGS_TO_WRAP, originalBindingsToWrap, ScriptContext.ENGINE_SCOPE);
            xlrScriptContext.setAttribute(XlrScriptContext.CONTEXT_BINDINGS_TO_UNWRAP, originalBindingsToUnWrap, ScriptContext.ENGINE_SCOPE);
        }
    }

    protected abstract Object evalScriptPrivileged(ScriptEngine engine, String script, ScriptContext scriptContext, AccessControlContext accessControlContext) throws Exception;

    protected Object doPrivileged(ScriptEngine engine, String script, ScriptContext scriptContext, AccessControlContext accessControlContext) throws Exception {
        try {
            logger.trace("Content of the script is: {}{}", System.getProperty("line.separator"), script);
            PrivilegedExceptionAction<Object> action = () -> engine.eval(intern(script), scriptContext);
            return AccessController.doPrivileged(action, accessControlContext);
        } catch (PrivilegedActionException exceptionWrapper) {
            logger.debug("PrivilegedActionException: ", exceptionWrapper);
            throw exceptionWrapper.getException();
        }
    }

    protected Boolean isScriptSandboxEnabled() {
        return this.sandboxEnabled;
    }

    protected java.lang.String getWorkDir() {
        return workDir;
    }

    protected AccessControlContext getAccessControlContext() {
        if (sandboxEnabled) {
            ProtectionDomain domain = new ProtectionDomain(new ScriptCodeSource(), extendsMinimalPermissionsWith(scriptPermissionsProvider.getScriptPermissions()));
            return new AccessControlContext(new ProtectionDomain[]{domain});
        } else {
            return null;
        }
    }

    private Permissions extendsMinimalPermissionsWith(PermissionCollection permissions) {
        Permissions minimalPermissions = createMinimalPermissions();
        addSandboxPermissions(minimalPermissions);
        addInternalAccessPermissions(minimalPermissions);
        addJythonLibrariesReadPermissions(minimalPermissions);
        addGroovyLibrariesReadPermissions(minimalPermissions);
        addJsonModule(minimalPermissions);
        addJsonSmartLibraryPermissions(minimalPermissions);
        addOvertherePermissions(minimalPermissions);
        addWorkDirPermissions(minimalPermissions);
        addJavaHomeSecurityLibPermissions(minimalPermissions);
        addCiUtilsPermissions(minimalPermissions);
        addReadPermissionOnPluginsFolder(minimalPermissions);

        if (permissions != null) {
            Enumeration<Permission> permissionEnumeration = permissions.elements();
            while (permissionEnumeration.hasMoreElements()) {
                Permission permission = resolvePermissionIfNecessary(permissionEnumeration.nextElement());
                minimalPermissions.add(permission);
            }
        }

        return minimalPermissions;
    }

    protected Permissions createMinimalPermissions() {
        Permissions minimalPermissions = new Permissions();
        minimalPermissions.add(new PropertyPermission("user.dir", "read"));
        minimalPermissions.add(new PropertyPermission("line.separator", "read"));
        minimalPermissions.add(new PropertyPermission("file.encoding", "read"));
        minimalPermissions.add(new RuntimePermission("createClassLoader"));
        // D-27883: allow cache stats registration when API is called from scripts
        minimalPermissions.add(new MBeanPermission("com.xebialabs.xlrelease.support.cache.caffeine.CaffeineStatsCounter", "-", ObjectName.WILDCARD, "registerMBean"));
        minimalPermissions.add(new RuntimePermission("getProtectionDomain"));
        minimalPermissions.add(new RuntimePermission("getClassLoader"));
        minimalPermissions.add(new SecurityPermission("insertProvider.BC"));
        minimalPermissions.add(new RuntimePermission("accessClassInPackage.sun.util.calendar"));
        return minimalPermissions;
    }

    private void addAPIPermissions(Permissions minimalPermissions) {
        // Public API
        minimalPermissions.add(grantedClassAccess("com.xebialabs.xlrelease"));
        minimalPermissions.add(grantedClassAccess("com.xebialabs.xlrelease.api"));
        minimalPermissions.add(grantedClassAccess("com.xebialabs.xlrelease.api.*"));
        minimalPermissions.add(grantedClassAccess("com.xebialabs.xlrelease.builder"));
        minimalPermissions.add(grantedClassAccess("com.xebialabs.xlrelease.builder.*"));
        minimalPermissions.add(grantedClassAccess("com.xebialabs.deployit.security.Permissions"));
        minimalPermissions.add(grantedClassAccess("com.xebialabs.xlrelease.domain"));
        minimalPermissions.add(grantedClassAccess("com.xebialabs.xlrelease.domain.*"));
        minimalPermissions.add(grantedClassAccess("com.xebialabs.xlrelease.dsl"));
        minimalPermissions.add(grantedClassAccess("com.xebialabs.xlrelease.dsl.*"));
        minimalPermissions.add(grantedClassAccess("com.xebialabs.deployit.plugin.api"));
        minimalPermissions.add(grantedClassAccess("com.xebialabs.deployit.plugin.api.*"));
        minimalPermissions.add(grantedClassAccess("com.xebialabs.xlrelease.risk.api"));
        minimalPermissions.add(grantedClassAccess("com.xebialabs.xlrelease.risk.api.*"));
        minimalPermissions.add(grantedClassAccess("com.xebialabs.xlrelease.risk.domain"));
        minimalPermissions.add(grantedClassAccess("com.xebialabs.xlrelease.risk.domain.*"));
        minimalPermissions.add(grantedClassAccess("com.xebialabs.xlrelease.risk.configuration"));
        minimalPermissions.add(grantedClassAccess("com.xebialabs.xlrelease.risk.configuration.*"));
        minimalPermissions.add(grantedClassAccess("com.xebialabs.xlrelease.repository.PhaseVersion"));
        minimalPermissions.add(grantedClassAccess("com.xebialabs.xlrelease.views.ImportResult"));
        minimalPermissions.add(grantedClassAccess("com.xebialabs.xlrelease.plugins.dashboard.domain"));
        minimalPermissions.add(grantedClassAccess("com.xebialabs.xlrelease.plugins.dashboard.domain.*"));
        minimalPermissions.add(grantedClassAccess("jakarta.ws.*"));
    }

    protected void addJavaPrimitivesPermissions(Permissions minimalPermissions) {
        // Java primitives
        minimalPermissions.add(grantedClassAccess("java"));
        minimalPermissions.add(grantedClassAccess("java.lang"));
        minimalPermissions.add(grantedClassAccess("java.lang.ArithmeticException"));
        minimalPermissions.add(grantedClassAccess("java.lang.ArrayIndexOutOfBoundsException"));
        minimalPermissions.add(grantedClassAccess("java.lang.Boolean"));
        minimalPermissions.add(grantedClassAccess("java.lang.Byte"));
        minimalPermissions.add(grantedClassAccess("java.lang.Character"));
        minimalPermissions.add(grantedClassAccess("java.lang.CharSequence"));
        minimalPermissions.add(grantedClassAccess("java.lang.Cloneable"));
        minimalPermissions.add(grantedClassAccess("java.lang.Comparable"));
        minimalPermissions.add(grantedClassAccess("java.lang.Double"));
        minimalPermissions.add(grantedClassAccess("java.lang.Enum"));
        minimalPermissions.add(grantedClassAccess("java.lang.Error"));
        minimalPermissions.add(grantedClassAccess("java.lang.Exception"));
        minimalPermissions.add(grantedClassAccess("java.lang.Float"));
        minimalPermissions.add(grantedClassAccess("java.lang.FunctionalInterface"));
        minimalPermissions.add(grantedClassAccess("java.lang.IllegalAccessException"));
        minimalPermissions.add(grantedClassAccess("java.lang.IllegalArgumentException"));
        minimalPermissions.add(grantedClassAccess("java.lang.IllegalStateException"));
        minimalPermissions.add(grantedClassAccess("java.lang.IndexOutOfBoundsException"));
        minimalPermissions.add(grantedClassAccess("java.lang.Integer"));
        minimalPermissions.add(grantedClassAccess("java.lang.InterruptedException"));
        minimalPermissions.add(grantedClassAccess("java.lang.Iterable"));
        minimalPermissions.add(grantedClassAccess("java.lang.Long"));
        minimalPermissions.add(grantedClassAccess("java.lang.Math"));
        minimalPermissions.add(grantedClassAccess("java.lang.NullPointerException"));
        minimalPermissions.add(grantedClassAccess("java.lang.Number"));
        minimalPermissions.add(grantedClassAccess("java.lang.NumberFormatException"));
        minimalPermissions.add(grantedClassAccess("java.lang.Object"));
        minimalPermissions.add(grantedClassAccess("java.lang.Runnable"));
        minimalPermissions.add(grantedClassAccess("java.lang.RuntimeException"));
        minimalPermissions.add(grantedClassAccess("java.lang.Short"));
        minimalPermissions.add(grantedClassAccess("java.lang.String"));
        minimalPermissions.add(grantedClassAccess("java.lang.StringBuffer"));
        minimalPermissions.add(grantedClassAccess("java.lang.StringBuilder"));
        minimalPermissions.add(grantedClassAccess("java.lang.SecurityException"));
        minimalPermissions.add(grantedClassAccess("java.lang.System"));
        minimalPermissions.add(grantedClassAccess("java.lang.Throwable"));
        minimalPermissions.add(grantedClassAccess("java.lang.Void"));
        minimalPermissions.add(grantedClassAccess("java.text"));
        minimalPermissions.add(grantedClassAccess("java.text.*"));
        minimalPermissions.add(grantedClassAccess("java.time"));
        minimalPermissions.add(grantedClassAccess("java.time.*"));
        minimalPermissions.add(grantedClassAccess("java.net"));
        minimalPermissions.add(grantedClassAccess("java.net.*"));
        minimalPermissions.add(grantedClassAccess("java.math"));
        minimalPermissions.add(grantedClassAccess("java.math.*"));
        minimalPermissions.add(grantedClassAccess("java.util"));
        minimalPermissions.add(grantedClassAccess("java.util.*"));
        minimalPermissions.add(grantedClassAccess("java.sql"));
        minimalPermissions.add(grantedClassAccess("java.sql.*"));
        minimalPermissions.add(grantedClassAccess("java.lang.ref"));
        minimalPermissions.add(grantedClassAccess("java.lang.ref.*"));
        minimalPermissions.add(grantedClassAccess("java.io"));
        minimalPermissions.add(grantedClassAccess("java.io.*"));
    }

    private void addSandboxPermissions(Permissions minimalPermissions) {
        addAPIPermissions(minimalPermissions);
        addJavaPrimitivesPermissions(minimalPermissions);
        // Groovy primitives
        minimalPermissions.add(grantedClassAccess("groovy.lang.*"));
        minimalPermissions.add(grantedClassAccess("org.codehaus.groovy.*"));

        // Scala primitives
        minimalPermissions.add(grantedClassAccess("scala.*"));

        // Script logging
        minimalPermissions.add(grantedClassAccess("org.slf4j.*"));
        minimalPermissions.add(grantedClassAccess("ch.qos.logback.*"));

        // Script defined classes
        minimalPermissions.add(grantedClassAccess("default.*"));

        // Random backwards compatibility
        minimalPermissions.add(grantedClassAccess("com.xebialabs.xlrelease.plugin"));
        minimalPermissions.add(grantedClassAccess("com.xebialabs.xlrelease.plugin.webhook"));
        minimalPermissions.add(grantedClassAccess("com.xebialabs.xlrelease.plugin.webhook.*"));
        minimalPermissions.add(grantedClassAccess("com.xhaus"));
        minimalPermissions.add(grantedClassAccess("com.xhaus.jyson"));
        minimalPermissions.add(grantedClassAccess("com.xhaus.jyson.*"));

        // Required for TriggersApi
        minimalPermissions.add(grantedClassAccess("org.springframework.data"));
        minimalPermissions.add(grantedClassAccess("org.springframework.data.domain.Sort"));
        minimalPermissions.add(new RuntimePermission("defineClass"));
    }

    protected void addInternalAccessPermissions(Permissions minimalPermissions) {
        minimalPermissions.add(new AuthPermission("modifyPrincipals"));
        minimalPermissions.add(new AuthPermission("modifyPublicCredentials"));
        minimalPermissions.add(new RuntimePermission("accessDeclaredMembers"));
        minimalPermissions.add(new RuntimePermission("accessUserInformation"));
        minimalPermissions.add(new RuntimePermission("accessClassInPackage.sun.security.util"));
        minimalPermissions.add(new RuntimePermission("accessClassInPackage.sun.security.x509"));
        // Required for plugin classloader (when script uses a class defined in a plugin)
        minimalPermissions.add(new RuntimePermission("accessClassInPackage.sun.net.www.protocol.jar"));
        minimalPermissions.add(new ReflectPermission("suppressAccessChecks"));
        minimalPermissions.add(new RuntimePermission("accessClassInPackage.sun.reflect"));
        minimalPermissions.add(new RuntimePermission("reflectionFactoryAccess"));
    }

    private void addOvertherePermissions(Permissions minimalPermissions) {
        minimalPermissions.add(new NetPermission("specifyStreamHandler"));
        minimalPermissions.add(new PropertyPermission("jcifs.properties", "read"));
    }

    protected void addJythonLibrariesReadPermissions(Permissions minimalPermissions) {
        addReadPermissionOnJar(jythonStandaloneJars, minimalPermissions);

        // Required for http-lib!
        minimalPermissions.add(new PropertyPermission("os.name", "read"));
        minimalPermissions.add(new PropertyPermission("os.arch", "read"));
        minimalPermissions.add(grantedClassAccess("org.python.modules._hashlib"));
        minimalPermissions.add(grantedClassAccess("org.python.modules.time.Time"));
    }

    protected void addGroovyLibrariesReadPermissions(Permissions minimalPermissions) {
        addReadPermissionOnJar(groovyJars, minimalPermissions);
    }

    protected void addJsonModule(Permissions permissions) {
        permissions.add(grantedClassAccess("json"));
        permissions.add(grantedClassAccess("json.scanner"));
        permissions.add(grantedClassAccess("_json.make_scanner"));
        permissions.add(grantedClassAccess("_json.make_encoder"));
        permissions.add(grantedClassAccess("_json.scanstring"));
        permissions.add(grantedClassAccess("_json.encode_basestring_ascii"));
        permissions.add(grantedClassAccess("encodings"));
        permissions.add(grantedClassAccess("encodings.hex"));
        permissions.add(grantedClassAccess("encodings.hex_codec.*"));
        permissions.add(grantedClassAccess("org.python.modules.binascii"));
        permissions.add(grantedClassAccess("org.python.modules._json._json"));
        permissions.add(grantedClassAccess("org.python.modules.struct"));
        permissions.add(grantedClassAccess("org.python.modules.struct.*"));
        permissions.add(new PropertyPermission("sun.arch.data.model", "read"));
        permissions.add(new PropertyPermission("guava.concurrent.generate_cancellation_cause", "read"));
    }

    protected void addCiUtilsPermissions(Permissions permissions) {
        permissions.add(grantedClassAccess("com.xebialabs.deployit.plugin.api.reflect"));
        permissions.add(grantedClassAccess("com.xebialabs.deployit.plugin.api.reflect.Type"));
    }


    protected void addJsonSmartLibraryPermissions(Permissions minimalPermissions) {
        addReadPermissionOnJar(jsonSmartJars, minimalPermissions);
        minimalPermissions.add(new PropertyPermission("JSON_SMART_SIMPLE", "read"));
    }

    private void addWorkDirPermissions(Permissions minimalPermissions) {
        String userDir = System.getProperty("user.dir");
        Path workDir = Path.of(getWorkDir());
        String workdirPath = workDir.isAbsolute() ? workDir.toString() : Paths.get(userDir, getWorkDir()).toString();
        minimalPermissions.add(new FilePermission(workdirPath + File.separator + "-", "delete"));
    }

    protected void addJavaHomeSecurityLibPermissions(final Permissions minimalPermissions) {
        String javaHome = System.getProperty("java.home");
        String javaHomePath = Paths.get(javaHome).toString();
        String libSecurityPath = Stream.of(javaHomePath, "lib", "security", "-").collect(joining(File.separator));
        minimalPermissions.add(new FilePermission(libSecurityPath, "read"));
    }

    protected void addReadPermissionOnJar(List<String> jarNames, Permissions minimalPermissions) {
        jarNames.forEach(jarName -> minimalPermissions.add(new FilePermission(jarName, "read")));
    }

    protected void addReadPermissionOnPluginsFolder(Permissions minimalPermissions) {
        String userDir = System.getProperty("user.dir");
        String pluginsPath = Paths.get(userDir, "plugins").toString();
        String hotfixPluginsPath = Paths.get(userDir, "hotfix/plugins").toString();
        minimalPermissions.add(new FilePermission(pluginsPath + File.separator + "-", "read"));
        minimalPermissions.add(new FilePermission(hotfixPluginsPath + File.separator + "-", "read"));
    }

    protected Permission resolvePermissionIfNecessary(final Permission permission) {
        if (permission instanceof UnresolvedPermission) {
            UnresolvedPermission unresolvedPermission = (UnresolvedPermission) permission;
            if (com.xebialabs.xlrelease.script.security.RuntimePermission.class.getName().equals(unresolvedPermission.getUnresolvedType())) {
                return new com.xebialabs.xlrelease.script.security.RuntimePermission(unresolvedPermission.getUnresolvedName(), unresolvedPermission.getUnresolvedActions());
            }
        }
        return permission;
    }

    /**
     * This method sets threadlocal "sandbox" property that is later used by
     * SandboxAwarePackageManager to restrict packages (class namespaces) that are accessible
     */
    protected ScriptEngine configureSandboxAndGetEngine(boolean checkPolicyPermissions) {
        SandboxAwarePackageManager.setSandboxed(isRestricted(checkPolicyPermissions));
        return getScriptEngine(checkPolicyPermissions);
    }

    public void reloadScriptEngines() {
        logger.info("Creating {} engine instances", this.getClass().getSimpleName());
        this.unrestrictedEngine = engineFactory.getScriptEngine(false);
        this.restrictedEngine = engineFactory.getScriptEngine(true);
    }

    private ScriptEngine getScriptEngine(final boolean checkPolicyPermissions) {
        if (isRestricted(checkPolicyPermissions)) {
            return restrictedEngine;
        } else {
            return unrestrictedEngine;
        }
    }

    protected boolean isRestricted(final boolean checkPolicyPermissions) {
        return checkPolicyPermissions && isScriptSandboxEnabled() && getSecurityManager() != null;
    }

    private String intern(String script) {
        if (StringUtils.hasLength(script)) {
            return script.intern();
        } else {
            return "";
        }
    }

}
