package com.xebialabs.xlrelease.script.jython

import com.typesafe.config.Config
import com.xebialabs.xlrelease.script.Jsr223EngineFactory
import grizzled.slf4j.Logging
import org.apache.commons.lang.reflect.FieldUtils.readField
import org.python.core.PySystemState
import org.python.jsr223.{PyScriptEngine, PyScriptEngineFactory}
import org.python.util.PythonInterpreter

import javax.script.ScriptEngine

object JythonEngineInstance extends Jsr223EngineFactory with Logging {
  private var cacheSettings: Config = _

  private val engineFactory: PyScriptEngineFactory = new PyScriptEngineFactory() {
    PySystemState.initialize()
    PySystemState.packageManager = SandboxAwarePackageManager.getInstance
  }

  def apply(cacheSettings: Config): Jsr223EngineFactory = {
    this.cacheSettings = cacheSettings
    restrictedEngine
    unrestrictedEngine
    this
  }

  private lazy val restrictedEngine = createEngine(true)
  private lazy val unrestrictedEngine = createEngine(false)

  def getScriptEngine(restricted: Boolean): ScriptEngine = {
    logger.debug(s"Jython engine [restricted: $restricted] instance started.")
    if (restricted) {
      restrictedEngine
    } else {
      unrestrictedEngine
    }
  }

  private def createEngine(restricted: Boolean): ScriptEngine = {
    try {
      SandboxAwarePackageManager.setSandboxed(restricted)
      val engine = if (restricted) {
        val engine = engineFactory.getScriptEngine.asInstanceOf[PyScriptEngine]
        val interpreter = readField(engine, "interp", true).asInstanceOf[PythonInterpreter]
        val parentLoader = Thread.currentThread().getContextClassLoader
        interpreter.getSystemState.setClassLoader(new JythonScriptClassLoader(parentLoader))
        engine
      } else {
        engineFactory.getScriptEngine.asInstanceOf[PyScriptEngine]
      }
      val engineName = if (restricted) "jython-restricted" else "jython-unrestricted"
      val cacheEnabled = cacheSettings.getBoolean(s"$engineName.enabled")
      logger.info(s"Creating $engineName engine with cache ${if (cacheEnabled) "enabled" else "disabled"}")
      if (cacheEnabled) {
        new XlrJythonEngine(engine, engineName, cacheSettings)
      } else {
        engine
      }
    } finally {
      SandboxAwarePackageManager.setSandboxed(false)
    }
  }
}
