package com.xebialabs.xlrelease.runner.crypto.encrypt

import com.xebialabs.xlrelease.runner.crypto.B64EncodedSecretKey
import com.xebialabs.xlrelease.runner.crypto.encrypt.JobDataEncryptor._
import grizzled.slf4j.Logging
import org.springframework.security.crypto.encrypt.AesBytesEncryptor.CipherAlgorithm
import org.springframework.security.crypto.encrypt.{AesBytesEncryptor, BytesEncryptor, TextEncryptor}
import org.springframework.security.crypto.keygen.{KeyGenerators, StringKeyGenerator}

import java.nio.charset.StandardCharsets
import java.util.Base64
import javax.crypto.spec.{PBEKeySpec, SecretKeySpec}
import javax.crypto.{SecretKey, SecretKeyFactory}

object JobDataEncryptor {
  final val SECRET_KEY_ALGORITHM: String = "PBKDF2WithHmacSHA256"
  final val SECRET_KEY_ITERATION_COUNT: Int = 1024
  final val SECRET_KEY_LENGTH: Int = 256
  final val IV_GENERATOR_BLOCK_SIZE: Int = 16
  final val AES_CIPHER_ALGORITHM: CipherAlgorithm = CipherAlgorithm.GCM
  final val AES_ENCRYPTION_ALGORITHM = "AES"

  def apply(encryptionEnabled: Boolean): JobDataEncryptor = {
    if (encryptionEnabled) {
      new AESJobDataEncryptor()
    } else {
      new NoOpJobDataEncryptor()
    }
  }
}

trait JobDataEncryptor extends TextEncryptor {
  def setSecretKey(rawKey: Array[Byte]): Unit

  def getB64EncodedSecretKey: B64EncodedSecretKey

  def getEncodedSecretKey: Array[Byte]
}

private[encrypt] class NoOpJobDataEncryptor extends JobDataEncryptor with Logging {
  override def setSecretKey(rawKey: Array[Byte]): Unit = {
    logger.trace(s"Operation not support for this encryptor. Provided secret key will be ignored.")
  }

  override def getB64EncodedSecretKey: B64EncodedSecretKey = new B64EncodedSecretKey(null)

  override def encrypt(text: String): String = text

  override def decrypt(encryptedText: String): String = encryptedText

  override def getEncodedSecretKey: Array[Byte] = Array[Byte]()
}

private[encrypt] class AESJobDataEncryptor extends JobDataEncryptor {

  private lazy val encoder = Base64.getEncoder

  private lazy val decoder = Base64.getDecoder

  private lazy val encryptor: BytesEncryptor = {
    val ivGenerator = KeyGenerators.secureRandom(IV_GENERATOR_BLOCK_SIZE)
    val secretKey = getSecretKey
    new AesBytesEncryptor(secretKey, ivGenerator, AES_CIPHER_ALGORITHM)
  }

  private var _secretKey: SecretKey = _

  override def setSecretKey(rawKey: Array[Byte]): Unit = {
    if (this._secretKey == null) {
      if (rawKey != null && rawKey.nonEmpty) {
        this._secretKey = new SecretKeySpec(rawKey, AES_ENCRYPTION_ALGORITHM)
      } else {
        throw new IllegalArgumentException("The secret key is null or empty")
      }
    } else {
      throw new IllegalStateException("Can't change secret key")
    }
  }

  override def getB64EncodedSecretKey: B64EncodedSecretKey = {
    new B64EncodedSecretKey(this._secretKey)
  }

  override def getEncodedSecretKey: Array[Byte] = {
    if (this._secretKey == null) {
      Array[Byte]()
    } else {
      this._secretKey.getEncoded
    }
  }

  override def encrypt(text: String): String = {
    if (null != text) {
      this.encoder.encodeToString(this.encryptor.encrypt(text.getBytes(StandardCharsets.UTF_8)))
    } else {
      throw new IllegalArgumentException("The text is null")
    }
  }

  override def decrypt(encryptedText: String): String = {
    if (null != encryptedText) {
      new String(this.encryptor.decrypt(this.decoder.decode(encryptedText)), StandardCharsets.UTF_8)
    } else {
      throw new IllegalArgumentException("The text is null")
    }
  }

  private def getSecretKey: SecretKey = {
    if (this._secretKey == null) {
      this._secretKey = generateKey()
    }
    this._secretKey
  }

  private def generateKey(): SecretKey = {
    val randomPasswordGenerator: StringKeyGenerator = KeyGenerators.string()
    val salt: String = randomPasswordGenerator.generateKey()
    val password: String = randomPasswordGenerator.generateKey()
    val factory: SecretKeyFactory = SecretKeyFactory.getInstance(SECRET_KEY_ALGORITHM)
    val spec: PBEKeySpec = new PBEKeySpec(password.toCharArray, salt.getBytes, SECRET_KEY_ITERATION_COUNT, SECRET_KEY_LENGTH)
    val secret: SecretKeySpec = new SecretKeySpec(factory.generateSecret(spec).getEncoded, AES_ENCRYPTION_ALGORITHM)
    secret
  }

}
